package test;

import com.badlogic.gdx.ApplicationAdapter;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.backends.lwjgl.LwjglApplication;
import com.badlogic.gdx.backends.lwjgl.LwjglApplicationConfiguration;
import com.badlogic.gdx.graphics.*;
import com.badlogic.gdx.graphics.g3d.*;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.environment.DirectionalLight;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.math.Intersector;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.math.collision.Ray;

public class CubeFaceSelectionDemo extends ApplicationAdapter {
    private PerspectiveCamera cam;
    private ModelBatch modelBatch;
    private Environment environment;
    private Model cubeModel;
    private ModelInstance cubeInstance;

    // Highlight materials
    private Material defaultMaterial;
    private Material highlightMaterial;
    private int selectedFace = -1;

    // Cube geometry vertices
    private float[] verts;

    @Override
    public void create() {
        modelBatch = new ModelBatch();

        cam = new PerspectiveCamera(67, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        cam.position.set(3f, 3f, 6f);
        cam.lookAt(0,0,0);
        cam.near = 0.1f;
        cam.far = 100f;
        cam.update();

        environment = new Environment();
        environment.set(new ColorAttribute(ColorAttribute.AmbientLight, 0.8f, 0.8f, 0.8f, 1f));
        environment.add(new DirectionalLight().set(1, 1, 1, -1, -0.8f, -0.2f));

        ModelBuilder modelBuilder = new ModelBuilder();
        cubeModel = modelBuilder.createBox(
                2f, 2f, 2f,
                new Material(ColorAttribute.createDiffuse(Color.GRAY)),
                VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal);

        cubeInstance = new ModelInstance(cubeModel);

        defaultMaterial = new Material(ColorAttribute.createDiffuse(Color.GRAY));
        highlightMaterial = new Material(ColorAttribute.createDiffuse(Color.RED));

        // Extract vertex positions of the cube (for picking)
        Mesh mesh = cubeModel.meshes.first();
        verts = new float[mesh.getNumVertices() * (mesh.getVertexSize()/4)];
        mesh.getVertices(verts);
    }

    @Override
    public void render() {
        if (Gdx.input.justTouched()) {
            Ray ray = cam.getPickRay(Gdx.input.getX(), Gdx.input.getY());
            selectedFace = pickFace(ray);
        }

        Gdx.gl.glViewport(0, 0, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT | GL20.GL_DEPTH_BUFFER_BIT);

        modelBatch.begin(cam);

        if (selectedFace >= 0) {
            // Replace cubeInstance materials depending on face selection
            cubeInstance.materials.get(0).set(
                    selectedFace == 0 ? highlightMaterial : defaultMaterial
            );
        }

        modelBatch.render(cubeInstance, environment);
        modelBatch.end();
    }

    private int pickFace(Ray ray) {
        // The cube has 12 triangles (6 faces, 2 per face)
        // We'll test intersection triangle by triangle.
        Mesh mesh = cubeModel.meshes.first();
        short[] indices = new short[mesh.getNumIndices()];
        mesh.getIndices(indices);

        float minDst = Float.MAX_VALUE;
        int hitFace = -1;
        Vector3 intersection = new Vector3();

        for (int i = 0; i < indices.length; i += 3) {
            Vector3 v0 = getVertex(indices[i]);
            Vector3 v1 = getVertex(indices[i+1]);
            Vector3 v2 = getVertex(indices[i+2]);

            if (Intersector.intersectRayTriangle(ray, v0, v1, v2, intersection)) {
                float dst = ray.origin.dst2(intersection);
                if (dst < minDst) {
                    minDst = dst;
                    hitFace = i / 6; // each 6 indices = 2 triangles = 1 face
                }
            }
        }
        return hitFace;
    }

    private Vector3 getVertex(int idx) {
        int stride = 6; // position(3) + normal(3)
        return new Vector3(
                verts[idx*stride],
                verts[idx*stride+1],
                verts[idx*stride+2]
        );
    }

    @Override
    public void dispose() {
        modelBatch.dispose();
        cubeModel.dispose();
    }
	 public static void main(String[] arg) {
	        LwjglApplicationConfiguration config = new LwjglApplicationConfiguration();
	        config.title = "CAD Assistant";
	        config.width = 1024;
	        config.height = 768;
	        config.vSyncEnabled = true;
	        new LwjglApplication(new CubeFaceSelectionDemo(), config);
	    }
}
