package test;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import com.badlogic.gdx.*;
import com.badlogic.gdx.backends.lwjgl.LwjglAWTCanvas;
import com.badlogic.gdx.backends.lwjgl.LwjglApplication;
import com.badlogic.gdx.backends.lwjgl.LwjglApplicationConfiguration;
import com.badlogic.gdx.graphics.*;
import com.badlogic.gdx.graphics.g2d.*;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.scenes.scene2d.*;
import com.badlogic.gdx.scenes.scene2d.ui.*;
import com.badlogic.gdx.utils.*;

public class ActorPrecisionTest extends ApplicationAdapter {
    private static final float MM_TO_PIXELS = 1f; // 1mm = 1 pixel (for visibility)
    private static final float ROOM_SIZE_MM = 10_000f; // 10 meters = 10,000 mm

    private Stage stage;
    private Label infoLabel;
    private Image precisionActor;
    private float targetX = 0f; // Target position in mm
    private boolean driftDetected = false;

    @Override
    public void create() {
        stage = new Stage();
        Gdx.input.setInputProcessor(stage);

        // Create UI
        Table table = new Table();
        table.setFillParent(true);
        table.top();
        infoLabel = new Label("Click to move actor 1mm right", new Label.LabelStyle(
            new BitmapFont(), Color.WHITE));
        table.add(infoLabel).pad(10);
        stage.addActor(table);

        // Create actor (1mm x 1mm square)
        Pixmap pixmap = new Pixmap(1, 1, Pixmap.Format.RGBA8888);
        pixmap.setColor(Color.RED);
        pixmap.fill();
        Texture texture = new Texture(pixmap);
        pixmap.dispose();

        precisionActor = new Image(texture);
        precisionActor.setSize(MM_TO_PIXELS, MM_TO_PIXELS);
        precisionActor.setPosition(0, 100); // Start at (0, 100)
        stage.addActor(precisionActor);

        // Add click listener to move 1mm right
        Gdx.input.setInputProcessor(new InputAdapter() {
            @Override
            public boolean touchDown(int screenX, int screenY, int pointer, int button) {
                moveActor1mm();
                return true;
            }
        });
    }

    private void moveActor1mm() {
        targetX += 1f; // Move 1mm right (theoretically)

        // ACTOR POSITION UPDATE (uses float)
        float currentX = precisionActor.getX();
        float newX = currentX + 1f; // <-- FLOAT DRIFT HAPPENS HERE
        precisionActor.setX(newX);

        // Calculate actual vs expected position
        float error = Math.abs(newX - targetX);
        driftDetected = error > 0.001f; // > 0.001mm error

        // Update info label
        String status = String.format(
            "Target: %.3f mm | Actual: %.3f mm | Error: %.6f mm %s",
            targetX, newX, error, driftDetected ? "⚠️ DRIFT!" : ""
        );
        infoLabel.setText(status);
    }

    @Override
    public void render() {
        Gdx.gl.glClearColor(0.1f, 0.1f, 0.1f, 1);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);

        // Draw reference grid (every 1000mm = 1m)
        ShapeRenderer sr = new ShapeRenderer();
        sr.begin(ShapeRenderer.ShapeType.Line);
        sr.setColor(Color.LIGHT_GRAY);
        for (int i = 0; i <= ROOM_SIZE_MM; i += 1000) {
            sr.line(i, 50, i, 150);
        }
        sr.end();

        stage.act();
        stage.draw();
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("LibGDX + JFrame + Grid + Hint");
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setSize(800, 600);

            LwjglAWTCanvas canvas = new LwjglAWTCanvas(new ActorPrecisionTest());
            frame.add(canvas.getCanvas());

//            frame.addComponentListener(new ComponentAdapter() {
//                @Override
//                public void componentResized(ComponentEvent e) {
//                    int newWidth = frame.getContentPane().getWidth();
//                    int newHeight = frame.getContentPane().getHeight();
    //
//                    canvas.getCanvas().setSize(newWidth, newHeight); // adjust canvas size
//                    canvas.getApplicationListener().resize(newWidth, newHeight); // notify LibGDX
//                }
//            });
            frame.setVisible(true);
        });
    }
}