package supercad.graphics;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;

import org.eclipse.swt.widgets.Display;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.lwjgl.util.vector.Matrix;

import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.g3d.model.Node;
import com.badlogic.gdx.graphics.g3d.model.NodePart;
import com.badlogic.gdx.math.Matrix4;

import dressing.io.IOUtilities;
import dressing.ui.ProgressBarMonitor;
import gdxapp.assets.ModelExporter;
import gdxapp.fabs3d.WoodCovering;
import gdxapp.object3d.ModelInfo;
import gdxapp.object3d.ModeledObject;
import gdxapp.object3d.Object3D;
import gdxapp.object3d.WorldObject.ObjectType;
import gdxapp.scenes.Scene;
import gdxapp.shaders.PbrMaterial;

public class InstancedSceneExporter {

	private List<Object3D> sceneObjects;
	private ArrayList<Mesh> meshes = new ArrayList<Mesh>();
	private ArrayList<PbrMaterial> materials = new ArrayList<PbrMaterial>();
	private HashMap<Model, List<Matrix4>> instances = new HashMap<Model, List<Matrix4>>(); 
	private HashMap<Model, String> modelsPath = new HashMap<Model, String>();
	private HashMap<Model, String> dotObjModels = new HashMap<Model, String>();


	public String exportScene(List<Object3D> objects, String folder) {
		this.sceneObjects = objects;
		groupByModel();
		Path rootFolder = Paths.get(folder);
		File file;
		if((file = rootFolder.toFile()).exists()) {
			IOUtilities.deleteFolder(file);
		}
		String mapsFolder = "maps";
		try {
			Files.createDirectories(rootFolder);
			Files.createDirectories(rootFolder.resolve(mapsFolder));
		} catch (IOException e) {
			e.printStackTrace();
		}
		int size = 0;
		for(var entry: instances.entrySet())
			size += entry.getValue().size();
		
		
		Display.getDefault().syncExec(new Runnable() {
			@Override
			public void run() {
				ProgressBarMonitor.setTasks(instances.keySet().size());
			}
		});
		//exports models		
		int i = 0;
		ModelExporter modelExporter = new ModelExporter();
		for(var model: instances.keySet()) {
			String modelName = "object" + i++;
			var path = dotObjModels.get(model);
			if(path == null) {
				modelExporter.exportModelXX(model, rootFolder.toString(), modelName, mapsFolder.toString());
				path = rootFolder.resolve(modelName + ".obj").toString();
			}
			modelsPath.put(model, path);
		}
		//export scene
		JSONArray jsonArray = new JSONArray();
		for(Model model: instances.keySet()) {
			String path = modelsPath.get(model);
			var transforms = instances.get(model).stream().map(mat4 -> serializeTransform(mat4)).toList();
			JSONObject modelJSON = new JSONObject();
			modelJSON.put("model", path);
			modelJSON.put("instances", transforms);
			jsonArray.add(modelJSON);
			
			Display.getDefault().syncExec(new Runnable() {
				@Override
				public void run() {
					ProgressBarMonitor.advance(1);
				}
			});
		}
		var output = rootFolder.resolve("scene.json");
		try {
			Files.createFile(output);
			Files.writeString(output, jsonArray.toString());
			return output.toString();
		} catch (IOException e) {
			e.printStackTrace();
			return null;
		}finally {
			modelExporter.clearRessources();
			modelExporter = null;
		}
	}
	
	private void groupByModel() {
		if(sceneObjects == null)
			return;
		instances.clear();
		sceneObjects.forEach( obj -> {
			var bucket =  instances.get(obj.model);
			if(bucket == null) {
				bucket = new ArrayList<Matrix4>();
				instances.put(obj.model, bucket);
			}
			Node node = obj.nodes.get(0);
			Matrix4 transform = obj.transform.cpy().mul(node.globalTransform);
			bucket.add(transform);
			var worldObject = obj.getWorldObject();
			if(worldObject != null && worldObject.getType() == ObjectType.MODELED) {
				if(worldObject instanceof WoodCovering)
					return;
				ModelInfo info = obj.getWorldObject().getModel().getInfo();
				String path = (String) info.getProperties().get("path_to_HP");
				dotObjModels.put(obj.model, path);
			}
		});
	}
	
	void processModel(Model model) {
		model.nodes.forEach( node -> processNode(node));
	}
	
	void processNode(Node node) {
		node.parts.forEach(part -> processNodePart(part));
	}
	
	void processNodePart(NodePart part) {
		var mesh = part.meshPart.mesh;
		if(!meshes.contains(mesh))
			meshes.add(mesh);
		int meshIndex = meshes.indexOf(mesh);
		var material =  (PbrMaterial)part.material;
		if(materials.contains(material)) {
			materials.add(material);
			
		}
		
		meshes.add(mesh);
	}
	
	public static String serializeTransform(Matrix4 matrix) {
		String serialized = "";
		for(int i = 0; i < 16; i++) {
			serialized += String.valueOf(matrix.getValues()[i]) + ", ";
		}
		
		return serialized.substring(0, serialized.length()-2);
	}
	
	
}
