package api.ui;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.joml.Vector2i;
import org.joml.Vector3f;
import api.backend.ApplicationContext;
import api.events.CustomEvent;
import api.events.EventHandler;
import api.events.EventType;
import api.graphics.ModelInstance;

public class SurfaceAnchors implements EventHandler{

	private Text vText;
	private Text  hText;
	private Composite container;
	
	private int textWidth = 40;
	private int textHeight = 20;

	SurfaceAnchors(Composite container){
		this.container = container;
		createControls();
		subscribe(EventType.DISPLAY_ANCHORS);
	}
	
	public void createControls() {
		var verifyListener = new VerifyListener() {
			@Override
			public void verifyText(VerifyEvent e) {
				if (!e.text.matches("\\d*")) {
			        e.doit = false;
			    }	
			}
		};
		
		vText = new Text(container, SWT.NONE);
		hText = new Text(container, SWT.None);
		vText.addVerifyListener(verifyListener);
		hText.addVerifyListener(verifyListener);
		vText.setVisible(false);
		hText.setVisible(false);
		vText.setSize(textWidth, textHeight);
		hText.setSize(textWidth, textHeight);
	}

	@Override
	public void handle(CustomEvent event) {
		if(event.getType().equals(EventType.DISPLAY_ANCHORS)) {
			if(event.getDetails() == null || event.getDetails().isEmpty()) {
				hide();
				return;
			}
			Vector3f hAnchor = (Vector3f) event.getDetail("hanchor");
			final Vector3f hRef = (Vector3f) event.getDetail("href");
			Vector3f vAnchor = (Vector3f) event.getDetail("vanchor");
			Vector3f vRef = (Vector3f) event.getDetail("vref");
			Vector2i hPos =  (Vector2i) event.getDetail("hpos");
			Vector2i vPos = (Vector2i) event.getDetail("vpos");
			String hValue = (String) event.getDetail("hValue");
			String vValue = (String) event.getDetail("vValue");
			ModelInstance target = (ModelInstance) event.getDetail("target"); 
			var canvas = ApplicationContext.getCanvas();
			var origin = container.toDisplay(0, 0);
			var displayH =  canvas.toDisplay(hPos.x, canvas.getSize().y -hPos.y);
			Vector2i hLocation = new Vector2i(displayH.x - origin.x, displayH.y - origin.y);
			
			var displayV =  canvas.toDisplay(vPos.x, canvas.getSize().y -vPos.y);
			Vector2i vLocation = new Vector2i(displayV.x - origin.x, displayV.y - origin.y);
			hText.setLocation(hLocation.x - textWidth/2, hLocation.y - textHeight / 2);
			hText.setText(hValue);
			vText.setLocation(vLocation.x - textWidth/2, vLocation.y - textHeight/2);
			vText.setText(vValue);
			vText.setVisible(true);
			hText.setVisible(true);
			hText.moveAbove(ApplicationContext.getCanvas());
			vText.moveAbove(ApplicationContext.getCanvas());
			//vText.setFocus();
			clearListeners(SWT.KeyUp);
			
			vText.addListener(SWT.KeyUp, e -> {
				if(e.keyCode == SWT.CR) {
					int dst = Integer.parseInt(vText.getText());
					Vector3f original = new Vector3f(vRef).sub(vAnchor);
					float scale = dst/1000.0f - original.length();
					Vector3f translation = new Vector3f(original).normalize(scale);
					target.translate(translation);
					hide();
				}
			});
			
			hText.addListener(SWT.KeyUp, e -> {
				if(e.keyCode == SWT.CR) {
					int dst = Integer.parseInt(hText.getText());
					Vector3f original = new Vector3f(hRef).sub(hAnchor);
					float scale = dst/1000.0f - original.length();
					Vector3f translation = new Vector3f(original).normalize(scale);
					target.translate(translation);
					hide();
				}
			});
	
		}
	}
	
	public void clearListeners(int type) {
		for(var listener: vText.getListeners(type))
			vText.removeListener(type, listener);
		for(var listener: hText.getListeners(type))
			hText.removeListener(type, listener);
	}
	
	public void hide() {
		clearListeners(SWT.KeyUp);
		hText.setVisible(false);
		vText.setVisible(false);
	}
	
}
