package gdxapp.object3d;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.List;
import javax.swing.SwingUtilities;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g2d.PolygonSpriteBatch;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.Polygon;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.EventListener;
import com.badlogic.gdx.scenes.scene2d.Stage;
import com.badlogic.gdx.utils.Array;
import com.badlogic.gdx.utils.Disposable;

import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.controller.tools.ToolController;
import dressing.mathutils.Surface;
import dressing.mathutils.Triangle;
import dressing.mathutils.Vector4;
import dressing.model.DesignObject3D;
import dressing.model.ProjectManager;
import gdxapp.Commun.AbstractScreen;
import gdxapp.Commun.ActionProcessor;
import gdxapp.Commun.ScreenController;
import gdxapp.assets.AssetsTextures;
import gdxapp.assets.DrawingHelper;
import gdxapp.object3d.WorldObject.ObjectType;
import gdxapp.quotation.Quotation;
import gdxapp.scenes.Scene;
import gdxapp.screens.room.RoomController;
import gdxapp.screens.room.RoomScreen;
import gdxapp.screens.wall.SurfaceScreen;
import gdxapp.screens.wall.Wall2D;
import geometry.Shape;
import space.earlygrey.shapedrawer.ShapeDrawer;
import utils.EmfUtils;

public class Object2D extends Actor implements Disposable {

	protected static ShapeDrawer shapeDrawer;
	private static PolygonSpriteBatch polyBatch;
	protected float z;
	protected WorldObject worldObject;
	protected String textureRegion;
	protected final ArrayList<Edge> edges = new ArrayList<Edge>();
	protected final ArrayList<Vector2> rotatedVertices = new ArrayList<>();
	protected ArrayList<Vector2> attachementPoints = new ArrayList<>();
	protected ArrayList<Edge> surroundingEdges;
	protected Vector2 normal;
	protected Vector2 direction;
	protected float[] vertices;
	protected Polygon polygon;
	protected Vector2 absoluteOrigin; // the origin coords in pixel
	protected boolean topView = true;
	protected boolean isClosest = false;
	protected ArrayList<Quotation> quotations;
	protected Vector2 hQuotPoint, vQuotPoint;
	protected boolean drawInnerQuotation = false;
	protected EventListener eventListener;
	//
	protected List<Surface> beforesurfaces = null;
	protected float oldRotation = 0.0f;
	protected Vector3 oldPosition = null;
	protected Vector4 oldDimention = null;

	public List<Surface> getBeforesurfaces() {
		return beforesurfaces;
	}

	public void setBeforesurfaces(List<Surface> beforesurfaces) {
		this.beforesurfaces = beforesurfaces;
	}

	public float getOldRotation() {
		return oldRotation;
	}

	public void setOldRotation(float oldRotation) {
		this.oldRotation = oldRotation;
	}

	public Vector3 getOldPosition() {
		return oldPosition;
	}

	public void setOldPosition(Vector3 oldPosition) {
		this.oldPosition = oldPosition;
	}

	public Vector4 getOldDimention() {
		return oldDimention;
	}

	public void setOldDimention(Vector4 oldDimention) {
		this.oldDimention = oldDimention;
	}

	public Object2D() {
	}

	public Object2D(WorldObject worldObject) {
		super();
		setOrigin(getWidth() / 2, getHeight() / 2);
		this.worldObject = worldObject;
		eventListener = new Object2DEventProcessor(this);
		addListener(eventListener);
		setVisible(!worldObject.isHidden());
	}

	public void adjustTextureAndColors() {
		if (topView) {
			this.textureRegion = (String) getWorldObject().getProperties().get("top-texture");
		} else {
			this.textureRegion = (String) getWorldObject().getProperties().get("front-texture");
		}
	}

	public void updateOrigin() {
		if (this.absoluteOrigin == null)
			absoluteOrigin = new Vector2();
		absoluteOrigin.x = getX() + 0.5f * getWidth();
		absoluteOrigin.y = getY() + 0.5f * getHeight();
	}

	public ArrayList<Vector2> calculateAttachmentPosition() {

		if (attachementPoints == null) {
			attachementPoints = new ArrayList<Vector2>();
		} else {
			attachementPoints.clear();
		}

		if (getWorldObject() instanceof KitchenElement) {
			DesignObject3D designObject = ((KitchenElement) getWorldObject()).getDesignObject();
			if (Scene.game.getScreen() instanceof RoomScreen
					&& (designObject != null && (EmfUtils.getDesignCaissonType(designObject).equals("HAUT_COINS_L")
							|| EmfUtils.getDesignCaissonType(designObject).equals("BAS_COINS_L")))) {
				if (vertices == null) {
					setVertices();
				}

				for (int i = 0; i < vertices.length; i += 2) {
					Vector2 vertex = new Vector2(vertices[i], vertices[i + 1]);
					attachementPoints.add(MathUtilities.rotate(absoluteOrigin, vertex, getWorldObject().getRotation()));
				}
				return attachementPoints;
			}
		}

		attachementPoints.addAll(getRotatedVertices());
		Vector2 topCenter = new Vector2(getX() + .5f * getWidth(), getY() + getHeight());
		Vector2 bottomCenter = new Vector2(getX() + .5f * getWidth(), getY());
		Vector2 leftCenter = new Vector2(getX(), getY() + .5f * getHeight());
		Vector2 rightCenter = new Vector2(getX() + getWidth(), getY() + .5f * getHeight());
		float rotation = getWorldObject().getRotation();
		if (Scene.game.getScreen() instanceof SurfaceScreen)
			rotation = 0;
		topCenter = MathUtilities.rotate(absoluteOrigin, topCenter, rotation);
		bottomCenter = MathUtilities.rotate(absoluteOrigin, bottomCenter, rotation);
		leftCenter = MathUtilities.rotate(absoluteOrigin, leftCenter, rotation);
		rightCenter = MathUtilities.rotate(absoluteOrigin, rightCenter, rotation);
		attachementPoints.add(topCenter);
		attachementPoints.add(bottomCenter);
		attachementPoints.add(leftCenter);
		attachementPoints.add(rightCenter);
		return attachementPoints;
	}

	public boolean overlap(Object2D actorX) {
		float z = getWorldObject().getRealWorldPosition().y;
		float actorZ = actorX.getWorldObject().getRealWorldPosition().y;
		float height = getWorldObject().getRealWorldDimension().y;
		float colliderHeight = actorX.getWorldObject().getRealWorldDimension().y;
		if (2 * Math.abs(z - actorZ) >= height + colliderHeight) {
			return false;
		}

		if (actorX instanceof GeometryObject2D) {
			GeometryObject2D geomtery = (GeometryObject2D) actorX;
			for (Triangle triangle : geomtery.getTriangles()) {
				for (Vector2 vertex : getRotatedVertices()) {
					if (triangle.contains(vertex))
						return true;
				}
			}
			return false;
		}
		ArrayList<Vector2> rotatedVertices = new ArrayList<Vector2>();
		rotatedVertices.addAll((ArrayList<Vector2>) actorX.getRotatedVertices());
		List<Vector2> vertices = new ArrayList<Vector2>();
		vertices.addAll(getRotatedVertices());
		for (int i = 0; i < vertices.size(); i++) {
			Vector2 vertex = vertices.get(i);
			if (MathUtilities.isPointInsideConvexPolygon(vertex, rotatedVertices)) {
				return true;
			}
		}

		return false;
	}

	public Object2D getClosestObject(Vector2 a, Vector2 b) {
		Object2D closest = null;
		ArrayList<Object2D> targets = new ArrayList<Object2D>();
		if (getStage() != null && getStage().getActors() != null) {
			Array<Actor> actors = new Array<Actor>();
			actors.addAll(getStage().getActors());
			for (Actor actorX : actors) {
				if (actorX instanceof Object2D && actorX != this) {
					Object2D target = (Object2D) actorX;
					if (!target.getWorldObject().isHidden())
						targets.add(target);
				}
			}
			float smallestDistance = -1;
			final List<Vector2> synList = new ArrayList<>();
			synList.addAll(getRotatedVertices());
			for (Vector2 vertex : synList) {
				for (Object2D targetX : targets) {
					ArrayList<Vector2> rotatedVertices = new ArrayList<Vector2>();
					rotatedVertices.addAll(targetX.getRotatedVertices());
					for (Vector2 point : rotatedVertices) {
						float distance = vertex.cpy().sub(point).len();
						if (smallestDistance < 0 || smallestDistance > distance) {
							smallestDistance = distance;
							closest = targetX;
							a.set(vertex.cpy());
							b.set(point.cpy());
						}
					}
				}
			}
		}
		return closest;
	}

	public Shape getClosestSurface() {
		return null;
	}

	public void adjustPosition() {
		Vector2 a = new Vector2(), b = new Vector2();
		Object2D target = getClosestObject(a, b);
		Vector3 translation = new Vector3(b.cpy().sub(a), 0);
		Vector2 position = new Vector2(getX(), getY()).add(b.cpy().sub(a));
		setPosition(position.x, position.y);
	}

	public void adjustRotation() {
		Vector2 a = new Vector2(), b = new Vector2();
		Object2D target = getClosestObject(a, b);
		if (target != null) {
			float rotation = target.getRotation();
			if (target.getWorldObject() instanceof Wall)
				rotation -= 90;
			getWorldObject().setRotation(rotation);
			setRotation(rotation);
		}

	}

	@Override
	protected void rotationChanged() {
		calculateAttachmentPosition();
		super.rotationChanged();
	}

	public ArrayList<Edge> calculateEdges() {
		edges.clear();
		Edge top, left, bottom, right;
		top = new Edge(new Vector2(getRotatedVertices().get(0)), new Vector2(getRotatedVertices().get(3)));
		left = new Edge(new Vector2(getRotatedVertices().get(1)), new Vector2(getRotatedVertices().get(0)));
		bottom = new Edge(new Vector2(getRotatedVertices().get(2)), new Vector2(getRotatedVertices().get(1)));
		right = new Edge(new Vector2(getRotatedVertices().get(3)), new Vector2(getRotatedVertices().get(2)));
		edges.add(top);
		edges.add(left);
		edges.add(bottom);
		edges.add(right);
		return edges;
	}

	public ArrayList<Edge> getSurroundingEdges() {
		ArrayList<Edge> surroundingEdges = new ArrayList<Edge>();
		Stage stage = getStage();
		if (stage != null) {
			Array<Actor> actors = new Array<Actor>();
			actors.addAll(stage.getActors());
			for (Actor actor : actors) {
				if (actor == this)
					continue;
				if (actor instanceof Object2D) {
					surroundingEdges.addAll(((Object2D) actor).calculateEdges());
				}
			}
		}
		return surroundingEdges;
	}

	public ArrayList<Quotation> getQuotations() {
		calculateEdges();
		this.quotations = new ArrayList<Quotation>();
		quotations.add(new Quotation(edges.get(0), (int) (getWorldObject().getRealWorldDimension().z * 1000) + ""));
		quotations.add(new Quotation(edges.get(1), (int) (getWorldObject().getRealWorldDimension().x * 1000) + ""));
		return quotations;
	}

	public void setVertices() {
		vertices = new float[] { getX(), getY(), getX(), getY() + getHeight(), getX() + getWidth(),
				getY() + getHeight(), getX() + getWidth(), getY() };
		if (polygon == null)
			polygon = new Polygon();
		polygon.setVertices(vertices);
		polygon.setOrigin(getX() + 0.5f * getWidth(), getY() + 0.5f * getHeight());
		polygon.setRotation(this.worldObject.getRotation());
	}
	

	public Edge getClosestEdge(ArrayList<Edge> edges, Vector2 out) {
		ArrayList<Vector2> attachementPoints = calculateAttachmentPosition();
		Edge closest = null;
		Vector2 attachmentPoint = null;
		float minDistance = 1000001;
		float distance;
		for (int i = 0; i < attachementPoints.size(); i++) {
			Vector2 vertex = attachementPoints.get(i);
			for (Edge edge : edges) {
				distance = MathUtilities.distance(vertex, edge.getV0(), edge.getV1());
				if (minDistance > distance && distance < 1000) {
					closest = edge;
					minDistance = distance;
					attachmentPoint = vertex;
				}
			}

		}
		if (attachmentPoint != null)
			out.set(attachmentPoint);
		return closest;
	}

	synchronized public void calculateRotatedVertices() {

		rotatedVertices.clear();
		if (getWorldObject().getType() == ObjectType.POLY) {
			GeometryObject geometryObject = (GeometryObject) getWorldObject();
			if (geometryObject.getGeometry().getBorder() != null) {
				rotatedVertices.addAll(((GeometryObject) getWorldObject()).getGeometry().getBorder().getVertices());
			}
		} else if (getWorldObject() instanceof KitchenElement) {
			DesignObject3D designObject = ((KitchenElement) getWorldObject()).getDesignObject();
			if (designObject != null && (EmfUtils.getDesignCaissonType(designObject).equals("HAUT_COINS_L")
					|| EmfUtils.getDesignCaissonType(designObject).equals("BAS_COINS_L"))) {
				try {
					float extrusionLength = (float) EmfUtils.getDesignProfondeur(designObject) * 0.001f
							* RoomController.getInstance().getScaleX();
					extrusionLength = getWidth() - extrusionLength;
					rotatedVertices.add(new Vector2(getX(), getY() + getHeight()));
					rotatedVertices.add(new Vector2(getX() + getWidth(), getY() + getHeight()));
					rotatedVertices.add(new Vector2(getX() + getWidth(), getY()));
					rotatedVertices.add(new Vector2(getX() + extrusionLength, getY()));
					rotatedVertices.add(new Vector2(getX() + extrusionLength, getY() + extrusionLength));
					rotatedVertices.add(new Vector2(getX(), getY() + extrusionLength));
				} catch (Exception e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			} else {
				rotatedVertices.add(new Vector2(getX(), getY() + getHeight())); //
				rotatedVertices.add(new Vector2(getX(), getY()));
				rotatedVertices.add(new Vector2(getX() + getWidth(), getY()));
				rotatedVertices.add(new Vector2(getX() + getWidth(), getY() + getHeight()));
			}
		} else {
			rotatedVertices.add(new Vector2(getX(), getY() + getHeight())); //
			rotatedVertices.add(new Vector2(getX(), getY()));
			rotatedVertices.add(new Vector2(getX() + getWidth(), getY()));
			rotatedVertices.add(new Vector2(getX() + getWidth(), getY() + getHeight()));

		}
		updateOrigin();
		float rotation = getWorldObject().getRotation();
		if (Scene.game.getScreen() instanceof SurfaceScreen)
			rotation = 0;
		for (Vector2 vertex : rotatedVertices) {
			vertex.set(MathUtilities.rotate(absoluteOrigin, vertex, rotation));
		}

		this.normal = rotatedVertices.get(1).cpy().sub(rotatedVertices.get(0)).nor();
		this.direction = this.normal.cpy().rotate(90);
	}

	public void calculateQuotationPoints() {
		this.hQuotPoint = this.absoluteOrigin.cpy().add(this.direction.cpy().scl(this.getHeight() * 0.4f));
		this.vQuotPoint = this.absoluteOrigin.cpy().add(this.normal.cpy().scl(-this.getWidth() * 0.4f));
	}

	@Override
	public void draw(Batch batch, float parentAlpha) {
		if (!getWorldObject().isHidden()) {
			if (Scene.game.getScreen() == RoomScreen.getInstance()) {
				drawTopView(batch);
			} else {
				drawWallView(batch);
			}
		} else {
			remove();
		}
	}

	protected void drawWallView(Batch batch) {
		if (!worldObject.isHidden()) {
			drawBorder(batch);
			if (getWorldObject().getType().equals(ObjectType.MODELED)) {
				batch.draw(textureRegion().getTexture(), getX(), getY(), getWidth() / 2, getHeight() / 2, getWidth(),
						getHeight(), getScaleX(), getScaleY(), getRotation(), textureRegion().getRegionX(),
						textureRegion().getRegionY(), textureRegion().getRegionWidth(),
						textureRegion().getRegionHeight(), false, false);
			}
			if (getWorldObject().isSelected || getWorldObject().isDrawInnerQuotations()
					|| ActionProcessor.getProcessor().isShowInnerQuotations()) {
				DrawingHelper.drawQuotations(this, batch);
			}
		}

		
	}

	protected TextureRegion textureRegion() {
		if (this.textureRegion != null && !this.textureRegion.isBlank())
			return AssetsTextures.getInstance().getTextureRegion(this.textureRegion);
		return AssetsTextures.getInstance().getDefaultTextureRegion();
	}

	protected void drawTopView(Batch batch) {
		setColor(0.5f, 0.5f, 1.0f, 1.0f);
		drawBorder(batch);
		if (getWorldObject().isSelected()) {
			batch.end();
			DrawingHelper.drawDistanceLines(this, batch);
			batch.begin();
		}
		if (!worldObject.isHidden()) {
			if ((getWorldObject() instanceof KitchenElement && (!(((KitchenElement) worldObject).getDesignObject()
					.getName().contains("L")
					|| ((KitchenElement) getWorldObject()).getDesignObject().getName().toUpperCase().contains("ANGLE"))
					|| !this.topView)) || getWorldObject().getType().equals(ObjectType.MODELED)) {
				batch.setColor(getColor());
				batch.draw(textureRegion().getTexture(), getX(), getY(), getWidth() / 2, getHeight() / 2, getWidth(),
						getHeight(), getScaleX(), getScaleY(), getRotation(), textureRegion().getRegionX(),
						textureRegion().getRegionY(), textureRegion().getRegionWidth(),
						textureRegion().getRegionHeight(), false, false);
			} else {
				batch.end();
				if (shapeDrawer == null)
					shapeDrawer = new ShapeDrawer(RoomController.getPolyBatch(), textureRegion());
				shapeDrawer.setTextureRegion(textureRegion());
				setVertices();
				RoomController.getPolyBatch().setProjectionMatrix(batch.getProjectionMatrix().cpy());
				shapeDrawer.setColor(new Color(1.0f, 1.0f, 1.0f, 0.8f));
				RoomController.getPolyBatch().begin();
				shapeDrawer.filledPolygon(polygon);
				RoomController.getPolyBatch().end();
				batch.begin();
			}

			if (getWorldObject() instanceof KitchenElement
					&& !((KitchenElement) getWorldObject()).getDesignObject().getName().contains("L")
					&& !((KitchenElement) getWorldObject()).getDesignObject().getName().contains("Angle")) {
				DrawingHelper.drawDoors(this, batch);
				batch.flush();
			}

			if (getWorldObject().model != null && getWorldObject().getName() != null
					&& getWorldObject().getType().equals(ObjectType.MODELED)
					&& (getWorldObject().getName().contains("Porte"))) {
				DrawingHelper.drawDoorsarc(this, batch);
			}

			if ((getWorldObject().isSelected || getWorldObject().isDrawInnerQuotations()
					|| ActionProcessor.getProcessor().isShowInnerQuotations()) && !(getWorldObject() instanceof Wall)) {
				DrawingHelper.drawQuotationsTopView(this, batch);
			}

			calculateAttachmentPosition();
			if (getWorldObject().isSelected) {
				if (surroundingEdges == null)
					calculateSurroundingEdges();
				Vector2 vertex = new Vector2();
				Edge closest = getClosestEdge(surroundingEdges, vertex);
				Vector2 proj;
				if (closest != null) {
					float[] scales;
					try {
						AbstractScreen screen = (AbstractScreen) Scene.game.getScreen();
						ScreenController controller = screen.getController();
						scales = controller.getScales();
					} catch (Exception e) {
						e.printStackTrace();
						scales = new float[] { 1, 1 };
					}

					proj = MathUtilities.projectPoint(vertex, closest.getV0(), closest.getV1());
					Vector2 vect = new Vector2(vertex).sub(proj);
					String val = (Math.round(vect.len() * scales[0] * 1000)) + "mm";
					Vector2 pos = new Vector2(vertex).add(proj).scl(.5f);
					DrawingHelper.drawCutLine(vertex, new Vector2(proj.x, proj.y), batch, Color.BLACK, 2);
					DrawingHelper.drawText(val, new Vector3(pos, 0), 0, Color.BLACK, .5f, batch);
				}
			}
		}
	}

	protected void drawBorder(Batch batch) {
		batch.end();
		Color borderColor = Color.BLACK;
		if (getWorldObject().isSelected) {
			if (!getWorldObject().isMoveable() || ToolController.getInstance().isFREEZE_MOVE()) {
				borderColor = Color.RED;
			} else {
				borderColor = Color.GREEN;
			}
		}
		DrawingHelper.drawRotatedEdges(this, batch, 5, borderColor);
		batch.begin();
	}

	private void drawBorder(Color color, float thickness) {
		ArrayList<Vector2> border;
		if (getWorldObject().getType() == ObjectType.POLY) {
			border = ((GeometryObject) getWorldObject()).getGeometry().getBorder().getVertices();
		} else {
			border = (ArrayList<Vector2>) getRotatedVertices();
		}
		DrawingHelper.getDebugRenderer().begin(ShapeType.Filled);
		DrawingHelper.getDebugRenderer().setColor(color);
		for (int i = 0; i < border.size() - 1; i++) {
			DrawingHelper.getDebugRenderer().rectLine(border.get(i), border.get(i + 1), thickness);
		}
		DrawingHelper.getDebugRenderer().rectLine(border.get(0), border.get(border.size() - 1), thickness);

		DrawingHelper.getDebugRenderer().setColor(Color.BLACK);
		DrawingHelper.getDebugRenderer().end();

	}

	public ArrayList<Edge> calculateSurroundingEdges() {
		if (surroundingEdges == null)
			surroundingEdges = new ArrayList<Edge>();
		surroundingEdges = new ArrayList<Edge>();
		if (getStage() != null && getStage().getActors() != null) {
			Array<Actor> actors = new Array<Actor>();
			actors.addAll(getStage().getActors());
			for (Actor actor : actors) {
				if (actor instanceof Object2D) {
					Object2D object2dX = (Object2D) actor;
					if (object2dX == this || object2dX instanceof Wall2D || object2dX.getWorldObject().isHidden())
						continue;
					surroundingEdges.addAll(object2dX.calculateEdges());
				}
			}
		}
		return surroundingEdges;
	}

	@Override
	protected void positionChanged() {
		updateOrigin();
		setVertices();
		calculateRotatedVertices();
		calculateAttachmentPosition();
		calculateEdges();
		calculateQuotationPoints();
		if(getWorldObject().isConstraint()) {
			Vector3 position = getWorldObject().getRealWorldPosition();
			Vector4 dims = getWorldObject().getRealWorldDimension();
			Vector3 translation = new Vector3(0,0, dims.z / 2.0f).rotate(Vector3.Y, getWorldObject().rotation);
			Vector3 center = position.cpy().add(translation);
			ProjectManager.getManager().getCurrentScene().addConstraint(getWorldObject(), center);
		}
		((AbstractScreen) Scene.game.getScreen()).getController().updateWorldObject(this);
	}
	//clamp the screen position within the visible range
	private void clampPosition() {
		float x = Math.min(getX(), Gdx.graphics.getWidth() - getWidth()/2.0f);
		x = Math.max(x, 0);
		float y = Math.min(getY(), Gdx.graphics.getHeight() - getHeight()/2.0f);
		y = Math.max(y, 0);
		if (getX() != x || getY() != y)
			setPosition(x, y);
	}

	@Override
	protected void sizeChanged() {
		updateOrigin();
		calculateRotatedVertices();
		calculateEdges();

		calculateQuotationPoints();
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				if (getStage() != null && getStage().getActors() != null) {
					try {
						// getStage().getActors().sort(AltitudeSorter.getSorter());

					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		});
		super.sizeChanged();
	}

	@Override
	public void act(float delta) {
		super.act(delta);
	}

	// getters and setters

	public String getTextureRegion() {
		return textureRegion;
	}

	public void setTextureRegion(String textureRegion) {
		this.textureRegion = textureRegion;
	}

	public <T extends WorldObject> T getWorldObject() {
		return (T) worldObject;
	}

	public Vector2 gethQuotPoint() {
		return hQuotPoint;
	}

	public void sethQuotPoint(Vector2 hQuotPoint) {
		this.hQuotPoint = hQuotPoint;
	}

	public Vector2 getvQuotPoint() {
		return vQuotPoint;
	}

	public void setvQuotPoint(Vector2 vQuotPoint) {
		this.vQuotPoint = vQuotPoint;
	}

	public void setWorldObject(WorldObject worldObject) {
		this.worldObject = worldObject;
	}

	public ArrayList<Edge> getEdges() {
		return edges;
	}

	transient protected PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);

	public void addPropertyChangeListener(PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(listener);
	}

	public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
	}

	public void removePropertyChangeListener(PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(listener);
	}

	public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(propertyName, listener);
	}

	public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
		propertyChangeSupport.firePropertyChange(propertyName, oldValue, newValue);
	}

	public void removeAllLsteners() {
		if (propertyChangeSupport == null) {
			return;
		}
		PropertyChangeListener[] listeners = propertyChangeSupport.getPropertyChangeListeners();
		if (listeners == null) {
			return;
		}
		for (PropertyChangeListener listener : listeners) {
			propertyChangeSupport.removePropertyChangeListener(listener);
		}
	}

	public void setTopView(boolean top) {
		this.topView = top;
		adjustTextureAndColors();
	}

	synchronized public ArrayList<Vector2> getRotatedVertices() {
		if (rotatedVertices.isEmpty())
			calculateRotatedVertices();
		ArrayList<Vector2> cpy = new ArrayList<Vector2>();
		for (Vector2 vertex : rotatedVertices) {
			cpy.add(new Vector2(vertex));
		}
		return cpy;
	}

	public Vector2 getAbsoluteOrigin() {
		updateOrigin();
		return absoluteOrigin;
	}

	public void setAbsoluteOrigin(Vector2 absoluteOrigin) {
		this.absoluteOrigin = absoluteOrigin;
	}

	public Vector2 getNormal() {
		return normal;
	}

	public void setNormal(Vector2 normal) {
		this.normal = normal;
	}

	public Vector2 getDirection() {
		return direction;
	}

	public void setDirection(Vector2 direction) {
		this.direction = direction;
	}

	public boolean isDrawInnerQuotation() {
		return drawInnerQuotation;
	}

	public void setDrawInnerQuotation(boolean drawInnerQuotation) {
		this.drawInnerQuotation = drawInnerQuotation;
	}

	public float getZ() {
		return z;
	}

	public void setZ(float z) {	
		this.z = z;
		positionChanged();
	}

	float getRadius() {
		return (float) Math.sqrt(Math.pow(getWidth() / 2.0f, 2) + Math.pow(getHeight() / 2.0f, 2));
	}

	@Override
	public void clear() {
		// TODO Auto-generated method stub
		super.clear();
		dispose();
	}

	@Override
	protected void finalize() throws Throwable {
		this.clear();
		super.finalize();
	}

	public void delete() {
		remove();
		dispose();
	}

	@Override
	public void dispose() {
		rotatedVertices.clear();
		edges.clear();
		textureRegion = null;
		attachementPoints = null;
		surroundingEdges = null;
	}

	public void hide() {
		setVisible(false);
	}

	/**
	 * do not use this method as while iterator condition car it can cause a
	 * infinite cycle
	 * 
	 * @return
	 */
	public boolean overlap() {
		ArrayList<Object2D> possibleColliders = new ArrayList<>();
		Array<Actor> actors = new Array<Actor>();
		actors.addAll(getStage().getActors());
		for (Actor actor : actors) {
			if (actor == this)
				continue;
			if (actor instanceof Object2D) {
				Object2D object2D = (Object2D) actor;
				if (object2D.getAbsoluteOrigin().dst(getAbsoluteOrigin()) < object2D.getRadius() + getRadius()) {
					possibleColliders.add(object2D);
				}
			}
		}

		for (Object2D possibleCollider : possibleColliders) {
			if (overlap(possibleCollider))
				return true;
		}
		return false;
	}
	public void setPosition(float x, float y , float z) {
		this.z = z;
		setPosition(x, y);
	}

}
