package dressing.ui.Assemblage;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.wb.swt.SWTResourceManager;

import dressing.config.persistence.ResourceManagers;
import dressing.ui.DoubleTextFieldFactory;
import dressing.ui.util.EditorWindow;
import dressing.ui.Assemblage.AssemblyComboSelect;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;

import param.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AssemblySystemEditorWindow extends EditorWindow {

    private final AssemblySystem assemblySystem;
    private Text txtName;
    private Text txtMinRange;
    private Text txtMaxRange;
    private Spinner groupCountSpinner;
    private Spinner assembliesPerGroupSpinner;
    private Composite containerArea;
    private ScrolledComposite root;
    private Composite container;

	private Group groupPositions;
	private Map<AssemblyContainer,ComboViewer> containersAssemblies=new HashMap<AssemblyContainer, ComboViewer>();
	private Map<AssemblyContainer, Label> containerslabels=new HashMap<AssemblyContainer, Label>();
	private ComboViewer calculPreferenceMethodviewer;
	private Text multiplierText;
	private Text entraxeText;
	private ComboViewer viewerSecondAssembly;
	private ComboViewer viewerMainAssembly;
	private Canvas drawingCanvas;
	
	public static final int DRAW_AREA_MARGIN_WIDTH=100;
	public static final int DRAW_AREA_HEIGH=200;
	public static final int DRAW_AREA_ARROW_WIDTH=10;
	public static final int DRAW_AREA_PIECE_WIDTH=36;
	public static final int COTATION_GROUP_Y =30;
	public static final int COTATION_TEXT_GROUP_Y =10;
	public static final int COTATION_ASSEMBLY_Y =60;
	public static final int COTATION_TEXT_ASSEMBLY_Y =40;
	public AssemblySystemEditorWindow(Shell parentShell, AssemblySystem system, int mode) {
        super(parentShell, mode);
        this.assemblySystem = system;
    }

    @Override
    protected Composite createClientArea(Composite parent) {
    	

        container = new Composite(parent, SWT.NONE);
        container.setLayout(new GridLayout(4, false));

        Label lblName = new Label(container, SWT.NONE);
        lblName.setText("Nom:");
        txtName = new Text(container, SWT.BORDER);
        txtName.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        txtName.setText(assemblySystem.getName() != null ? assemblySystem.getName() : "");
        new Label(container, SWT.NONE);
        new Label(container, SWT.NONE);

        createAssembliesAndSelection(container);
        createRangeAndStructure(container);
        createRepatitions(container);

        createControls(container);

        return container;
    }

    private void createControls(Composite parent) {

        root = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL);
        GridData rootData =new GridData(SWT.FILL, SWT.FILL, true, true,4,1);
        rootData.widthHint=100;
        rootData.heightHint=100;
        root.setLayoutData(rootData);
        
        root.setExpandHorizontal(true);
        root.setExpandVertical(true);
        
        containerArea = new Composite(root, SWT.NONE);
        GridData areaData = new GridData(SWT.FILL, SWT.TOP, true, false);
        areaData.widthHint = root.getClientArea().width; // hint to help width sizing
        containerArea.setLayoutData(areaData);
        containerArea.setLayout(new GridLayout(1, false));

        regenerateAssemblyContainers(true); // Initial build
        root.setContent(containerArea);
        containerArea.pack(true);
        containerArea.layout(true, true); // <-- ensure layout before sizing
        Point size = containerArea.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
        root.setMinSize(size.x + 5, size.y + 5); // small buffer
        root.setExpandHorizontal(true);
        root.setExpandVertical(true);
        
    }
    /** Left-top panel */
    private void createAssembliesAndSelection(Composite parent) {
        Group group = new Group(parent, SWT.NONE);
        group.setText("Assemblages & Sélection");
        group.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false,2,2));
        group.setLayout(new GridLayout(2, false));

        // Main assembly
        AssemblyComboSelect mainCombo = new AssemblyComboSelect(group, SWT.NONE);
        mainCombo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
        if (assemblySystem.getMainAssembly() != null) {
            mainCombo.getComboViewer()
                     .setSelection(new StructuredSelection(assemblySystem.getMainAssembly()));
        }
        
        viewerMainAssembly = mainCombo.getComboViewer();
        
        // Secondary assembly
        AssemblyComboSelect secCombo = new AssemblyComboSelect(group, SWT.NONE);
        secCombo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
        if (assemblySystem.getSecondaryAssembly() != null) {
            secCombo.getComboViewer()
                    .setSelection(new StructuredSelection(assemblySystem.getSecondaryAssembly()));
        }
        viewerSecondAssembly = secCombo.getComboViewer();
        createSystemAssembliesControls();
    }

	/**
	 * 
	 */
	public void createSystemAssembliesControls() {
		viewerMainAssembly.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
            	IStructuredSelection selection = viewerMainAssembly.getStructuredSelection();
            	if(selection!=null) {
            		Object selection2 =selection.getFirstElement();
                	if(selection2!=null && selection2 instanceof Assembly)
                    {
                		assemblySystem.setMainAssembly((Assembly) selection2);
    					
                    }else {
                    	assemblySystem.setMainAssembly(null);
                    }
            	}
            	updateContainersAssemblies();
            }
        });
		
		viewerSecondAssembly.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
            	IStructuredSelection selection = viewerSecondAssembly.getStructuredSelection();
            	if(selection!=null) {
            		Object selection2 =selection.getFirstElement();
                	if(selection2!=null && selection2 instanceof Assembly)
                    {
                		assemblySystem.setSecondaryAssembly((Assembly) selection2);
    					
                    }else {
                    	assemblySystem.setSecondaryAssembly(null);
                    }
            	}
            	updateContainersAssemblies();
            }
        });
	}
    
    private void createRangeAndStructure(Composite parent) {
        Group group = new Group(parent, SWT.NONE);
        group.setText("Plage et Structure");
        group.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false,2,4));
        group.setLayout(new GridLayout(2, false));

        // Min Range
        Label minLabel = new Label(group, SWT.NONE);
        minLabel.setText("Plage Min (mm) :");
        txtMinRange = DoubleTextFieldFactory.createDoubleTextField(group, SWT.BORDER);
        txtMinRange.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        txtMinRange.setText(String.valueOf(assemblySystem.getMinRange()));

        // Max Range
        Label maxLabel = new Label(group, SWT.NONE);
        maxLabel.setText("Plage Max (mm) :");
        txtMaxRange = DoubleTextFieldFactory.createDoubleTextField(group, SWT.BORDER);
        txtMaxRange.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        txtMaxRange.setText(String.valueOf(assemblySystem.getMaxRange()));

        // Number of Groups
        Label groupCountLabel = new Label(group, SWT.NONE);
        groupCountLabel.setText("Nombre de groupes :");
        groupCountSpinner = new Spinner(group, SWT.BORDER);
        groupCountSpinner.setMinimum(1);
        groupCountSpinner.setMaximum(10);
        groupCountSpinner.setSelection(assemblySystem.getNumberOfGroups());
        groupCountSpinner.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        
        // Assemblies per Group
        Label assembliesPerGroupLabel = new Label(group, SWT.NONE);
        assembliesPerGroupLabel.setText("Assemblages / groupe :");
        assembliesPerGroupSpinner = new Spinner(group, SWT.BORDER);
        assembliesPerGroupSpinner.setMinimum(1);
        assembliesPerGroupSpinner.setMaximum(10);
        assembliesPerGroupSpinner.setSelection(assemblySystem.getAssembliesPerGroup());
        assembliesPerGroupSpinner.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        createRangeAndStructureControls();
    }

	/**
	 * 
	 */
	public void createRangeAndStructureControls() {
        
        Listener refreshListener = e -> {
            try {           	
                assemblySystem.setNumberOfGroups(groupCountSpinner.getSelection());
                assemblySystem.setAssembliesPerGroup(assembliesPerGroupSpinner.getSelection());
                regenerateAssemblyContainers(true);
            } catch (NumberFormatException ex) {
                // Ignore invalid input silently
            }
        };
        Listener refreshPositionsListener = e -> {
            try {
            	assemblySystem.setMinRange(Double.parseDouble(txtMinRange.getText()));
                assemblySystem.setMaxRange(Double.parseDouble(txtMaxRange.getText()));
                
                updateContainersPositions();
            } catch (NumberFormatException ex) {
                // Ignore invalid input silently
            }
        };
        txtMinRange.addListener(SWT.Modify, refreshPositionsListener);
        txtMaxRange.addListener(SWT.Modify, refreshPositionsListener);
        
        groupCountSpinner.addListener(SWT.Modify, refreshListener);
        assembliesPerGroupSpinner.addListener(SWT.Modify, refreshListener);
        
	}
	private void createRepatitions(Composite parent) {
        Group group = new Group(parent, SWT.NONE);
        group.setText("Répartition");
        group.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false,2,3));
        group.setLayout(new GridLayout(2, false));

        // Entraxe
        new Label(group, SWT.NONE).setText("Entraxe (mm) :");
        entraxeText = DoubleTextFieldFactory.createDoubleTextField(group, SWT.BORDER);
        entraxeText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        entraxeText.setText(String.valueOf(assemblySystem.getEntraxe()));

        // Multiplicateur
        new Label(group, SWT.NONE).setText("Multiplicateur :");
        multiplierText = DoubleTextFieldFactory.createDoubleTextField(group, SWT.BORDER);
        multiplierText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        multiplierText.setText(String.valueOf(assemblySystem.getMultiplier()));
        // Methode Du Calcul
        new Label(group, SWT.NONE).setText("Méthode du répartion \n des Assemblages:");
        calculPreferenceMethodviewer = new ComboViewer(group, SWT.READ_ONLY);
        calculPreferenceMethodviewer.getCombo().setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        calculPreferenceMethodviewer.setContentProvider(ArrayContentProvider.getInstance());
        calculPreferenceMethodviewer.setLabelProvider(new LabelProvider());
        calculPreferenceMethodviewer.setInput(AssemblySystemCalculMethod.values());
        
        if (assemblySystem.getCalculMethod() != null) {
        	calculPreferenceMethodviewer.setSelection(new StructuredSelection(assemblySystem.getCalculMethod()));
        }
        createRepatitionsControls();
    }
	/**
	 * 
	 */
	public void createRepatitionsControls() {
		calculPreferenceMethodviewer.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
            	IStructuredSelection selection = calculPreferenceMethodviewer.getStructuredSelection();
            	if(selection!=null) {
            		Object selection2 =selection.getFirstElement();
                	if(selection2!=null && selection2 instanceof AssemblySystemCalculMethod)
                    {
                		assemblySystem.setCalculMethod((AssemblySystemCalculMethod) selection2);
    					
                    }else {
                    	assemblySystem.setCalculMethod(AssemblySystemCalculMethod.NORMAL);
                    }
                	updateContainersPositions();
            	}
            	
            }
        });
        
        Listener refreshPositionsListener = e -> {
            try {
            	
                assemblySystem.setEntraxe(Double.parseDouble(entraxeText.getText()));
                assemblySystem.setMultiplier(Double.parseDouble(multiplierText.getText()));
                
                updateContainersPositions();
            } catch (NumberFormatException ex) {
                // Ignore invalid input silently
            }
        };
        
        entraxeText.addListener(SWT.Modify, refreshPositionsListener);
        multiplierText.addListener(SWT.Modify, refreshPositionsListener);
	}
    private void createPositionsGroup(Composite parent) {
        groupPositions = new Group(parent, SWT.NONE);
        groupPositions.setText("Positions");
        groupPositions.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false,assemblySystem.getNumberOfGroups(),1));
        GridLayout containerData= new GridLayout((assemblySystem.getGroups().size()-1)*2+4, false);
        containerData.horizontalSpacing=10;
        containerData.verticalSpacing=10;
        groupPositions.setLayout(containerData);

        new Label(groupPositions, SWT.NONE).setText("Retrait 1: ");
        Text retrait1Text = DoubleTextFieldFactory.createDoubleTextField(groupPositions, SWT.BORDER);
        retrait1Text.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        retrait1Text.setText(String.valueOf(assemblySystem.getRetrait1()));
        retrait1Text.addListener(SWT.Modify, e -> {
            try {
                double retrait1 = Double.parseDouble(retrait1Text.getText());
                assemblySystem.setRetrait1(retrait1);
                updateContainersPositions();
            } catch (NumberFormatException ignored) {}
        });

        List<AssemblyContainerGroup> containers = assemblySystem.getGroups();
        for (int i = 1; i < containers.size(); i++) {
        	AssemblyContainerGroup container = containers.get(i);
            if (i > 0) {
                Button autoCheck = new Button(groupPositions, SWT.CHECK);
                autoCheck.setText("Auto");
                autoCheck.setSelection(container.isDistanceAuto());

                Text distanceText = DoubleTextFieldFactory.createDoubleTextField(groupPositions, SWT.BORDER);
                distanceText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
                distanceText.setText(String.valueOf(container.getDistance()));
                distanceText.setVisible(!container.isDistanceAuto());

                autoCheck.addListener(SWT.Selection, e -> {
                    boolean selected = autoCheck.getSelection();
                    container.setDistanceAuto(selected);
                    distanceText.setVisible(!selected);
                    updateContainersPositions();
                    groupPositions.layout(); 
                });

                distanceText.addListener(SWT.Modify, e -> {
                    try {
                        double distance = Double.parseDouble(distanceText.getText());
                        container.setDistance(distance);
                        updateContainersPositions();
                    } catch (NumberFormatException ignored) {}
                });
            }
        }
        new Label(groupPositions, SWT.NONE).setText("Retrait 2: ");
        Text retrait2Text = DoubleTextFieldFactory.createDoubleTextField(groupPositions, SWT.BORDER);
        retrait2Text.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        retrait2Text.setText(String.valueOf(assemblySystem.getRetrait2()));
        retrait2Text.addListener(SWT.Modify, e -> {
            try {
                double retrait2 = Double.parseDouble(retrait2Text.getText());
                assemblySystem.setRetrait2(retrait2);
                updateContainersPositions();
            } catch (NumberFormatException ignored) {}
        });
    }
    /**
     * Instantiates and lays out the canvas, and hooks up the paint listener.
     */
    private void createDrawingCanvas(Composite parent) {
        drawingCanvas = new Canvas(parent, SWT.BORDER);
        GridData gd = new GridData(SWT.FILL, SWT.TOP, true, false, assemblySystem.getNumberOfGroups(), 1);
        gd.heightHint = DRAW_AREA_HEIGH;        
        gd.minimumHeight=DRAW_AREA_HEIGH;// fixed height; adjust as you like
        drawingCanvas.setLayoutData(gd);
        drawingCanvas.setBackground(SWTResourceManager.getColor(255, 255, 255));
        drawingCanvas.addPaintListener(e -> {
            drawAssemblies(e.gc);
        });
    }

    /**
     * Call this method any time your assembly data / ranges / positions change
     * and you want to repaint the little “+” markers.
     */
    private void refreshDrawingCanvas() {
        if (drawingCanvas != null && !drawingCanvas.isDisposed()) {
            drawingCanvas.redraw();
        }
    }

	/**
	 * Does the actual drawing into the canvas's GC.
	 */
	private void drawAssemblies(GC gc) {
		Rectangle area = drawingCanvas.getClientArea();
		double maxX = assemblySystem.getMaxRange();
		double span = maxX;
		double scaleX = (area.width - DRAW_AREA_MARGIN_WIDTH) / span;
		int midY = area.height / 2;
		int underlineGroupY = DRAW_AREA_HEIGH * 3 / 4;
		int scaledWidth = (int) (span * scaleX);
		int R1Scale = (int) (assemblySystem.getRetrait1() * scaleX);
		int R2Scale = (int) (assemblySystem.getRetrait2() * scaleX);

		// Draw base piece rectangle
		int baseX = DRAW_AREA_MARGIN_WIDTH / 2;
		int baseY =midY- DRAW_AREA_PIECE_WIDTH / 2;
		gc.drawRectangle(baseX, baseY, scaledWidth, DRAW_AREA_PIECE_WIDTH);

		double groupSpacing = 0;
		if (assemblySystem.getNumberOfGroups() > 1) {
			int innerSpace = scaledWidth - R1Scale - R2Scale;
			groupSpacing = (double) innerSpace / (assemblySystem.getNumberOfGroups() - 1);
		}

		int mainIndex = assemblySystem.getAssembliesPerGroup() < 3 ? 0 : 1;
		int cotationStart = baseX;
		for (int g = 0; g < assemblySystem.getGroups().size(); g++) {
			AssemblyContainerGroup group = assemblySystem.getGroups().get(g);
			double groupCenter = R1Scale + (groupSpacing * g) + baseX;

			int startX = 0, endX = 0;

			for (int i = 0; i < group.getAssemblies().size(); i++) {
				AssemblyContainer ac = group.getAssemblies().get(i);
				double xval = groupCenter + (i - mainIndex) * (assemblySystem.getEntraxe() * scaleX);
				int xPix = (int) xval;

				if (i == 0)
					startX = xPix;
				if (i == group.getAssemblies().size() - 1)
					endX = xPix;

				drawPlusCircle(gc, xPix, midY);
				drawCenteredText(gc, ac.getName(), xPix-20, midY + 30);

				// Cotation between assemblies
				if (i != mainIndex) {
					gc.setForeground(SWTResourceManager.getColor(SWT.COLOR_BLUE));
					gc.drawLine(xPix, COTATION_ASSEMBLY_Y, (int) groupCenter, COTATION_ASSEMBLY_Y);
					gc.drawLine(xPix, COTATION_ASSEMBLY_Y, xPix, baseY);
					int textX = (int) ((xPix + groupCenter) / 2 - 20);
					gc.drawText(String.valueOf(assemblySystem.getEntraxe()), textX, COTATION_TEXT_ASSEMBLY_Y);
				}
			}

			// Cotation for group
			gc.setForeground(SWTResourceManager.getColor(SWT.COLOR_RED));
			gc.drawLine(cotationStart, COTATION_GROUP_Y, (int) groupCenter, COTATION_GROUP_Y);
			gc.drawLine(cotationStart, COTATION_GROUP_Y, cotationStart, baseY);
			gc.drawLine((int) groupCenter, COTATION_GROUP_Y, (int) groupCenter, baseY);

			String textCotationGroup = "Auto";
			if (!group.isDistanceAuto()) {
				if (assemblySystem.getCalculMethod().equals(AssemblySystemCalculMethod.MULTIPLE32)) {
					textCotationGroup = group.getDistance() + " X " + assemblySystem.getMultiplier() + " ( "
							+ (group.getDistance() * assemblySystem.getMultiplier()) + ")";
				} else {
					textCotationGroup = String.valueOf(group.getDistance());
				}
			}
			if (g == 0)
				textCotationGroup = String.valueOf(assemblySystem.getRetrait1());

			drawCenteredText(gc, textCotationGroup, (int) ((cotationStart + groupCenter) / 2 - 20),
					COTATION_TEXT_GROUP_Y);
			cotationStart = (int) groupCenter;

			// Group underline + arrow
			gc.setForeground(SWTResourceManager.getColor(SWT.COLOR_BLACK));
			gc.drawLine(startX - 20, underlineGroupY, endX + 20, underlineGroupY);
			int groupMidX = (startX + endX) / 2;
			drawArrow(gc, groupMidX, underlineGroupY, DRAW_AREA_HEIGH);
		}

		// Final retrait2 cotation
		gc.setForeground(SWTResourceManager.getColor(SWT.COLOR_RED));
		gc.drawLine(cotationStart, COTATION_GROUP_Y, baseX + scaledWidth, COTATION_GROUP_Y);
		gc.drawLine(baseX + scaledWidth, COTATION_GROUP_Y, baseX + scaledWidth, baseY);
		drawCenteredText(gc, String.valueOf(assemblySystem.getRetrait2()),
				(cotationStart + baseX + scaledWidth) / 2 - 20, COTATION_TEXT_GROUP_Y);
	}

	private void drawPlusCircle(GC gc, int x, int y) {
		gc.setForeground(SWTResourceManager.getColor(SWT.COLOR_BLACK));
		gc.drawLine(x - 4, y, x + 4, y);
		gc.drawLine(x, y - 4, x, y + 4);
		gc.drawOval(x - 8, y - 8, 16, 16);
	}

	private void drawCenteredText(GC gc, String text, int x, int y) {
		gc.drawText(text, x, y, true); // `true` enables transparent background
	}

	private void drawArrow(GC gc, int xMid, int startY, int endY) {
		gc.drawLine(xMid, startY, xMid, endY);
		gc.drawLine(xMid - DRAW_AREA_ARROW_WIDTH, endY - 10, xMid, endY);
		gc.drawLine(xMid + DRAW_AREA_ARROW_WIDTH, endY - 10, xMid, endY);
	}
	
    private void regenerateAssemblyContainers(boolean generateAssemblyContainers) {
        for (Control child : containerArea.getChildren()) {
            child.dispose();
        }
        containersAssemblies.clear();
        containerslabels.clear();
        if(generateAssemblyContainers)
        {
        	generateAssemblyContainers();
        }
        containerArea.pack(true);
        containerArea.layout(true, true);

        Point size = containerArea.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
        root.setMinSize(size.x + 5, size.y + 5); // small buffer
        container.layout(true, true); // This is the outer container layout
        root.setContent(containerArea);

        // Layout and redraw
        root.layout(true, true);
        root.redraw();
		
    }
    /**
	 * 
	 */
	public void generateAssemblyContainers() {
		int groupCount =assemblySystem.getNumberOfGroups();
        int assembliesPerGroup = assemblySystem.getAssembliesPerGroup();
        GridLayout containerData= new GridLayout(groupCount, true);
        containerData.horizontalSpacing=10;
        containerData.verticalSpacing=10;

        containerArea.setLayout(containerData);
        AssemblySystemController.updateAssemblyContainerGroups(groupCount, assembliesPerGroup,assemblySystem);
        //
        AssemblySystemController.updateAssembliesContainers(assembliesPerGroup,assemblySystem);
        //
        updateContainersPositions();
        //
        createPositionsGroup(containerArea);
        //
        createDrawingCanvas(containerArea);

        for(AssemblyContainerGroup containerGroup:assemblySystem.getGroups()) {
        	Composite group = new Composite(containerArea, SWT.NONE);
            group.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
            group.setLayout(new GridLayout(1, false));
            for (AssemblyContainer container:containerGroup.getAssemblies()) {
                createContainerUI(group, container);
            }
        }
       
	}

	
 

    private void createContainerUI(Composite parent, AssemblyContainer container) {
        Group group = new Group(parent, SWT.NONE);
        GridData data = new GridData(SWT.FILL, SWT.TOP, true, false);
        data.minimumWidth = 200;  // Minimum width
        group.setLayoutData(data);
        group.setLayout(new GridLayout(2, false));
        group.setText(container.getName());
        
        new Label(group, SWT.WRAP).setText("Position (X, Y, Z):");

        Label posLabel = new Label(group, SWT.WRAP);
        posLabel.setText(AssemblySystemController.getContainerPositionLabel(container));

        AssemblyComboSelect combo = new AssemblyComboSelect(group, SWT.NONE);
        combo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
        ComboViewer viewer = combo.getComboViewer();

        if (container.getAssembly() != null) {
            viewer.setSelection(new StructuredSelection(container.getAssembly()));
        }

        viewer.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
            	Object selection =viewer.getStructuredSelection().getFirstElement();
            	if(selection!=null && selection instanceof Assembly)
                {
					container.setAssembly((Assembly) selection);
					container.getPosition().getY().setExpression(((Assembly) selection).getDefaultPosY());
					posLabel.setText(AssemblySystemController.getContainerPositionLabel(container));
                }else {
                	container.setAssembly(null);
                }
            }

        });
        containersAssemblies.put(container, viewer);
        containerslabels.put(container, posLabel);
    }
    private void updateContainersAssemblies() {
    	Assembly mainAssembly =assemblySystem.getMainAssembly();
    	Assembly secondaryAssembly =assemblySystem.getSecondaryAssembly();
    	for(AssemblyContainerGroup group:assemblySystem.getGroups()) {
    		 for (int a=0;a<group.getAssemblies().size();a++) {
    			 AssemblyContainer container=group.getAssemblies().get(a);
    			 boolean ismain=assemblySystem.getAssembliesPerGroup()<3 && a==0 || assemblySystem.getAssembliesPerGroup()>=3 && a==1;
    			 
    			 container.setAssembly(ismain?mainAssembly:secondaryAssembly);
    			 ComboViewer viewer= containersAssemblies.get(container);
    			 if(viewer!=null) {
    				 StructuredSelection selection=new StructuredSelection(container.getAssembly()!=null?container.getAssembly():((List) viewer.getInput()).get(0));
    				 viewer.setSelection(selection);
    				 viewer.refresh();
    			 }
    		 }
    	}
    }
    
	private void updateContainersPositions() {
		AssemblySystemController.calculateAssembliesContainersPositions(assemblySystem);
		for(AssemblyContainer container:assemblySystem.getAssemblies()) {
			Label posLabel = containerslabels.get(container);
			if (posLabel != null) {
				posLabel.setText(String.format("(%s, %s, %s)", 
						container.getPosition().getX().getExpression(),
						container.getPosition().getY().getExpression(),
						container.getPosition().getZ().getExpression()));
				posLabel.requestLayout();
			}
		}
		refreshDrawingCanvas();
	}
    @Override
    public boolean check() {
        String name = txtName.getText().trim();
        String minStr = txtMinRange.getText().trim();
        String maxStr = txtMaxRange.getText().trim();

        if (name.isEmpty()) {
            errorMessage = "Name cannot be empty.";
            return false;
        }

        double min, max;
        try {
            min = Double.parseDouble(minStr);
            max = Double.parseDouble(maxStr);
        } catch (NumberFormatException e) {
            errorMessage = "Min and Max must be valid numbers.";
            return false;
        }

        if (min >= max) {
            errorMessage = "Min must be less than Max.";
            return false;
        }

        assemblySystem.setName(name);
        assemblySystem.setMinRange(min);
        assemblySystem.setMaxRange(max);
        return true;
    }

    @Override
    public boolean terminer() {
        if (!super.terminer()) return false;
        if (!check()) return false;

        ResourceManagers.getIntance().saveApplication();
        finish();
        return true;
    }

    @Override
    protected void cancel() {
        // No rollback needed
    }
}