package dressing.model;

import java.util.List;
import java.util.Optional;

import dressing.config.persistence.ResourceManagers;
import dressing.model.types.Orientation;
import dressing.model.types.PieceType;
import param.AssemblySystemsPreference;
import param.Direction;
import param.Face;
import param.MechanicDesign;
import param.ParamFactory;
import param.cam.MachineSides;

public class AssemblySystemsPreferenceResolver {

    private AssemblySystemsPreferenceResolver() {
        // Private constructor to prevent instantiation
    }

    /**
     * Resolves the assembly system preference for a specific face side.
     * Priority: Local preference -> Local allSides -> Imported preference -> Imported allSides -> Default
     *
     * @param design MechanicDesign where the lookup happens
     * @param faceSide The specific face side to resolve
     * @return The matching AssemblySystemsPreference
     */
    public static AssemblySystemsPreference resolvePreference(MechanicDesign design, MachineSides faceSide) {
        if (design == null || faceSide == null) {
//            return AssemblySystemsPreference.defaultPreference();
        	return null;
        }

        // 1. Check local preferences
        AssemblySystemsPreference preference = findPreferenceInGroup(
            design.getAssemblySystemsPreferences() != null ? design.getAssemblySystemsPreferences().getPreferences() : null,
            faceSide
        );
        if (preference != null) {
            return preference;
        }

        // 2. Check imported preferences
        preference = findPreferenceInGroup(design.getInportedAssemblySystemsPreferences(), faceSide);
        if (preference != null) {
            return preference;
        }

        // 3. Default fallback
//        return AssemblySystemsPreference.defaultPreference();
        return null;
    }

    /**
     * Find a preference in a list for the faceSide, first by exact match, then allSides.
     *
     * @param preferences List of preferences to search
     * @param faceSide Target face side
     * @return Matching preference or null
     */
    private static AssemblySystemsPreference findPreferenceInGroup(List<AssemblySystemsPreference> preferences, MachineSides faceSide) {
        if (preferences == null || preferences.isEmpty()) {
            return null;
        }

        // 1. Exact faceSide match
        Optional<AssemblySystemsPreference> exact = preferences.stream()
            .filter(pref -> faceSide.equals(pref.getFaceSide()) && !pref.isAllSides())
            .findFirst();
        if (exact.isPresent()) {
            return exact.get();
        }

        // 2. AllSides match
        Optional<AssemblySystemsPreference> allSides = preferences.stream()
            .filter(AssemblySystemsPreference::isAllSides)
            .findFirst();
        return allSides.orElse(null);
    }
    
    /**
	 * @param piece
	 */
	public static void fixFaceAssemblage(Piece2D piece,Face face) {
		
		face.setImportMethod(true);
		face.setExistExpression("Assembly_Method=='assemblageDynamique'");
		AssemblySystemsPreference assemblySystemsPreference =AssemblySystemsPreferenceResolver.resolvePreference(piece.getMechanicDesignDefinition(), face.getSide());
		if(assemblySystemsPreference!=null) {
			face.setAssemblySystem(assemblySystemsPreference.getAssemblySystem());	
		}
		face.setFamily(ResourceManagers.getIntance().getApplication().getAssemblyFamilyGroup().getFamilies().get(0));
	};
	
    public static void updateDirections(Face face,Piece2D parentPiece) {
    	MachineSides side = face.getSide();
        Orientation orient = parentPiece.getPieceOrientation();

        Direction dx = Direction.XPLUS;
        Direction dy = Direction.XPLUS;
        Direction dz = Direction.XPLUS;

        // orientation → dx, dy
        switch (orient) {
            case HORIZONTAL:
            	
            	if(parentPiece.getPiecetype().isBase()) {
            		dy=Direction.YMINUS;
            	}else {
            		dy=Direction.YPLUS;
            	}
            	
            	switch (side) {
				case BACK:
					dx = Direction.XPLUS;
	                dz = Direction.ZPLUS;
					break;
				
				case FACE:
					dx = Direction.XPLUS;
	                dz = Direction.ZMINUS;
					break;
				case LEFT:
					dx = Direction.ZMINUS;
	                dz = Direction.XPLUS;
					break;
				case RIGHT:
					dx = Direction.ZMINUS;
	                dz = Direction.XMINUS;
					break;
				case BOTTOM:
					dx = Direction.XPLUS;
	                dy = Direction.ZMINUS;
	                dz = Direction.YPLUS;
					break;
				case TOP:
					dx = Direction.XPLUS;
	                dy = Direction.ZMINUS;
	                dz = Direction.YMINUS;
					break;	
				default:
					break;
				}
                
                break;
            case VERTICAL:
            	if(parentPiece.getPiecetype().equals(PieceType.COTE_GAUCHE)) {
            		dy=Direction.XMINUS;
            	}else {
            		dy=Direction.XPLUS;
            	}
            	
            	switch (side) {
				case BACK:
					dx = Direction.YMINUS;
	                dz = Direction.ZPLUS;
					break;
				
				case FACE:
					dx = Direction.YMINUS;
	                dz = Direction.ZMINUS;
					break;
				case LEFT:
					dx = Direction.YPLUS;
	                dz = Direction.XPLUS;
	                dy = Direction.ZMINUS;
					break;
				case RIGHT:
					dx = Direction.YPLUS;
	                dz = Direction.XMINUS;
	                dy = Direction.ZMINUS;
					break;
				case BOTTOM:
					dx = Direction.ZMINUS;
	                dz = Direction.YPLUS;
					break;
				case TOP:
					dx = Direction.ZMINUS;
	                dz = Direction.YMINUS;
					break;	
				default:
					break;
				}
                break;
            case PROUFOUND:
            	if(parentPiece.getZpos()>0) {
            		dy=Direction.ZMINUS;
            	}else {
            		dy=Direction.ZPLUS;
            	}
            	
            	switch (side) {
				case BACK:
					dx = Direction.XPLUS;
	                dz = Direction.ZPLUS;
	                dy = Direction.YPLUS;
					break;
				
				case FACE:
					dx = Direction.XPLUS;
	                dz = Direction.ZMINUS;
	                dy = Direction.YPLUS;
					break;
				case LEFT:
					dx = Direction.YMINUS;
	                dz = Direction.XPLUS;
					break;
				case RIGHT:
					dx = Direction.YMINUS;
	                dz = Direction.XMINUS;
					break;
				case BOTTOM:
					dx = Direction.XPLUS;
	                dz = Direction.YPLUS;
					break;
				case TOP:
					dx = Direction.XPLUS;
	                dz = Direction.YMINUS;
					break;	
				default:
					break;
				}
                break;
            default:
               break;
        }

        face.setDirection_X(dx);
        face.setDirection_Y(dy);
        face.setDirection_Z(dz);

        
    }
}

