package dressing.ui;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.ConnectException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import javax.imageio.ImageIO;

import org.apache.commons.lang.math.NumberUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.frs.supercad.config.ConfigurationManager;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;

import dressing.config.WorkspaceConfiguration;
import dressing.config.persistence.ResourceManagers;
import dressing.ui.util.SingleImageItemLabelProvider;
import dressing.util.UtilDTO;
import dressing.util.dto.TextureCombination;
import dressing.util.dto.TextureCombinationResponse;
import dressing.util.dto.TextureComboView;
import dressing.util.dto.TextureComboViewResponse;
import param.Finition;
import param.Fournisseur_fabricant;
import param.Material;
import param.Nature_panneaux;
import param.impl.FinitionImpl;
import param.impl.Fournisseur_fabricantImpl;
import param.impl.MaterialImpl;
import param.impl.Nature_panneauxImpl;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Text;

public class ImportMaterialWindow {
	private static String BASE_URL = "https://api.catalogue.supercad.frsdev.ovh/api/v1";
	private final static boolean verbose = true;
	protected Shell shell;
	private Text minSizeText;
	private Text maxSizeText;
	private Combo combo;
	/**
	 * Launch the application.
	 * 
	 * @param args
	 */

	public static void main(String[] args) {
		try {
			ImportMaterialWindow window = new ImportMaterialWindow();
			window.open();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Open the window.
	 */
	public void open() {
		Display display = Display.getDefault();
		TextureComboViewResponse textureComboViewResponse = new TextureComboViewResponse();

		BundleContext ctx = FrameworkUtil.getBundle(getClass()).getBundleContext();
		ServiceReference<ConfigurationManager> ref = ctx.getServiceReference(ConfigurationManager.class);
		ConfigurationManager configManager = ctx.getService(ref);
		BASE_URL = configManager.getString("URL_CAD_STORE", "https://api.catalogue.supercad.frsdev.ovh/api/v1");
		try {

			textureComboViewResponse = UtilDTO.parseUrlResposne(new URL(BASE_URL + "/texture-combinations/summary"),
					TextureComboViewResponse.class);

		} catch (ConnectException e) {
			e.printStackTrace();

			MessageDialog.openError(new Shell(display), "Connexion échouée timeout",
					"Impossible de se connecter au supercad store.Veuillez vérifier l’URL de configuration ou votre connexion Internet.");
			return;
		} catch (IOException e) {
			e.printStackTrace();

			MessageDialog.openError(new Shell(display), "Connexion échouée",
					"Impossible de se connecter au supercad store.Veuillez vérifier l’URL de configuration ou votre connexion Internet.");
			return;
		}

		createContents(textureComboViewResponse);
		shell.open();
		shell.layout();
		while (!shell.isDisposed()) {
			if (!display.readAndDispatch()) {
				display.sleep();
			}
		}
	}

	/**
	 * Create contents of the window.
	 */
	protected void createContents(TextureComboViewResponse textureComboViewResponse) {
		shell = new Shell();
		shell.setSize(1200, 720);
		shell.setText("Import Material");
		shell.setLayout(new GridLayout(1, false));
		
		Composite composite = new Composite(shell, SWT.NONE);
		composite.setLayout(new GridLayout(8, false));
		GridData gd_composite = new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1);
		gd_composite.heightHint = 32;
		composite.setLayoutData(gd_composite);
		
		Label lblNewLabel = new Label(composite, SWT.NONE);
		lblNewLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		lblNewLabel.setText("Type de correction");
		
		combo = new Combo(composite, SWT.NONE);
		combo.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true, false, 1, 1));
		
		Label lblNewLabel_1 = new Label(composite, SWT.NONE);
		lblNewLabel_1.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		lblNewLabel_1.setText("dimension minimale");
		
		minSizeText = new Text(composite, SWT.BORDER);
		minSizeText.setText("512");
		GridData gd_minSizeText = new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1);
		gd_minSizeText.widthHint = 71;
		minSizeText.setLayoutData(gd_minSizeText);
		
		Label lblNewLabel_2 = new Label(composite, SWT.NONE);
		lblNewLabel_2.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		lblNewLabel_2.setText("dimension maximale");
		
		maxSizeText = new Text(composite, SWT.BORDER);
		maxSizeText.setText("2048");
		GridData gd_maxSizeText = new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1);
		gd_maxSizeText.widthHint = 59;
		maxSizeText.setLayoutData(gd_maxSizeText);
		
		Button btnResizeMaterial = new Button(composite, SWT.NONE);
		btnResizeMaterial.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				runResizeTask(new File(WorkspaceConfiguration.MATERIALS_FOLDER), "Resizing Materials...");
			}
		});
		btnResizeMaterial.setText("Resize Materials");
		
		Button btnResize = new Button(composite, SWT.NONE);
		btnResize.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
		        runResizeTask(new File(WorkspaceConfiguration.TEXTURES_FOLDER), "Resizing Textures...");
			}
		});
		btnResize.setText("Resize Textures");

		ScrolledComposite scrolledComposite = new ScrolledComposite(shell, SWT.V_SCROLL | SWT.H_SCROLL);
		scrolledComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);

		Composite container = new Composite(scrolledComposite, SWT.NONE);
		container.setLayout(new GridLayout(1, false));

		for (TextureComboView comboView : textureComboViewResponse) {
			Composite itemComposite = new Composite(container, SWT.BORDER);
			itemComposite.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
			itemComposite.setLayout(new GridLayout(2, false));
			itemComposite.setBackgroundMode(SWT.INHERIT_FORCE);

			Composite contentGrid = new Composite(itemComposite, SWT.NONE);
			contentGrid.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
			contentGrid.setLayout(new GridLayout(3, true));
			createInfoLabel(contentGrid, "Type: " + comboView.comboType);
			createInfoLabel(contentGrid, "Source: " + comboView.comboSource);
			createInfoLabel(contentGrid, "Support: " + comboView.comboSupport);
			createInfoLabel(contentGrid, "Count: " + comboView.comboCount);

			Button downloadBtn = new Button(itemComposite, SWT.PUSH);
			downloadBtn.setText("Importer");
			downloadBtn.setLayoutData(new GridData(SWT.RIGHT, SWT.TOP, false, false));
			downloadBtn.setData(comboView);

			downloadBtn.addListener(SWT.Selection, e -> {
				TextureComboView selected = (TextureComboView) downloadBtn.getData();
				runDownloadTask(selected);
			});
		}
		
		
		combo.setItems(new String[] {"FixNP2-ToMain","FixAll-ToSub"});
		combo.select(0);
		
		container.pack();
		scrolledComposite.setContent(container);
		scrolledComposite.setMinSize(container.computeSize(SWT.DEFAULT, SWT.DEFAULT));

	}

	private void createInfoLabel(Composite parent, String text) {
		Label label = new Label(parent, SWT.NONE);
		label.setText(text);
		label.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
	}

	
	private void runResizeTask(File folder, String taskName) {
		Shell shell = Display.getDefault().getActiveShell();
		ProgressMonitorDialog dialog = new ProgressMonitorDialog(shell);
		final ScaleMode scaleMode = ScaleMode.fromString(combo.getText());
		final int minSize = NumberUtils.toInt(minSizeText.getText().strip().trim(),512);
		final int maxSize = NumberUtils.toInt(maxSizeText.getText().strip().trim(),2048);
		final int scaleFactor = 2;

		try {
			dialog.run(true, true, monitor -> {
				monitor.beginTask(taskName, IProgressMonitor.UNKNOWN);
				try {
					resize(folder ,scaleMode, scaleFactor, minSize,maxSize, monitor);
				} catch (IOException | InterruptedException e) {
					throw new InvocationTargetException(e);
				} finally {
					monitor.done();
				}
			});
			MessageDialog.openInformation(shell, "Succès", "Toutes les images ont été traitées avec succès.");

		} catch (InvocationTargetException | InterruptedException e) {
		    Throwable cause = e.getCause();
		    if (cause instanceof InterruptedException) {
			    MessageDialog.openWarning(shell, "Annulé", "L'action a été annulée. Elle sera reprise la prochaine fois, si la reprise est prise en charge.");
		    }else {
				MessageDialog.openError(shell, "Error", "Failed to resize textures:\n" + e.getCause().getMessage());
				e.printStackTrace();
		    }
		    
		}
	}
	
	private void resize(File inputDir, ScaleMode scaleMode, int scaleFactor, int minSize, int maxSize,IProgressMonitor monitor) throws IOException, InterruptedException {
		File tempUpscaleDir = new File(inputDir, "__upscaled");
		File outputDir = new File(inputDir, "__cropped");
		tempUpscaleDir.mkdirs();
		outputDir.mkdirs();

		upscaleImages(inputDir, tempUpscaleDir, scaleMode, scaleFactor, minSize, monitor, verbose);
		cropImages(inputDir, tempUpscaleDir, outputDir, scaleMode, maxSize, monitor, verbose);
		
		if(scaleMode == ScaleMode.FIX_NP2_TO_MAIN) {
	        deleteDirectoryRecursively(tempUpscaleDir);
		}

	}

	private void upscaleImages(File inputDir, File tempUpscaleDir,ScaleMode scaleMode, int scaleFactor, int minSize, IProgressMonitor monitor,boolean verbose) throws IOException, InterruptedException {
	    String realesrganExe = new File(WorkspaceConfiguration.RESIZE_TOOLS, "realesrgan-ncnn-vulkan.exe").getAbsolutePath();
		List<File> imageFiles = Files.walk(inputDir.toPath())
			    .filter(path -> {
			        String pathStr = path.toString().replace("\\", "/");
			        return !pathStr.contains("/__upscaled/") && !pathStr.contains("/__cropped/");
			    })
				.filter(Files::isRegularFile)
				.map(Path::toFile)
				.filter(f -> f.getName().toLowerCase().matches(".*\\.(jpg|jpeg|png)"))
				.collect(Collectors.toList());
		
		int skippedUpscale = 0;
		for (int i = 0; i < imageFiles.size(); i++) {
			
		    if (monitor.isCanceled()) {
		        System.out.println("❌ Upscaling cancelled.");
		        throw new InterruptedException("Upscaling cancelled by user.");
		    }

			File inputFile = imageFiles.get(i);
	        Path relativePath = inputDir.toPath().relativize(inputFile.toPath());
	        File outputFile = new File(tempUpscaleDir, relativePath.toString());
	        outputFile.getParentFile().mkdirs();

	        String folderName = (relativePath.getParent() != null) ? relativePath.getParent().toString() : "(root)";
			
			if (outputFile.exists()) {
				if(verbose)
					System.out.println("⏩ Skipping already upscaled image: " + inputFile.getName() + " in folder: " + folderName);
		        skippedUpscale++;
		        monitor.subTask("Upscaling " + (i + 1) + "/" + imageFiles.size() + " (Skipped: " + skippedUpscale + ")");
				continue;
			}

			
			BufferedImage img ;
			try {
				img = ImageIO.read(inputFile);
				if (img == null) {
					System.err.println("❌ Skipping unreadable image (null): " + inputFile.getAbsolutePath());
			        skippedUpscale++;
					continue;
				}
			} catch (IOException e) {
				System.err.println("❌ Failed to read image: " + inputFile.getAbsolutePath());
		        skippedUpscale++;
				e.printStackTrace(); 
				continue;
			}

			outputFile.getParentFile().mkdirs();
			
			
	        boolean isLargeEnough = img.getWidth() >= minSize && img.getHeight() >= minSize;
	        boolean isPowerOfTwo = isPowerOfTwo(img.getWidth()) && isPowerOfTwo(img.getHeight());


	        if (isLargeEnough) {
	            if (scaleMode == ScaleMode.FIX_NP2_TO_MAIN && isPowerOfTwo) {
		            if(verbose)
		            	System.out.println("⏩ Skipping upscale COMPLETELY for : " + inputFile.getName() + " in folder: " + folderName + " (Image is larger than " + minSize + "x" + minSize + ")");
		            skippedUpscale++;
			        monitor.subTask("Upscaling " + (i + 1) + "/" + imageFiles.size() + " (Skipped: " + skippedUpscale + ")");
		            continue;
	        	}else {
		            Files.copy(inputFile.toPath(), outputFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
		            if(verbose)
		            	System.out.println("⏩ Skipping upscale for copied instead : " + inputFile.getName() + " in folder: " + folderName + " (Image is larger than " + minSize + "x" + minSize + ")");
		            skippedUpscale++;
			        monitor.subTask("Upscaling " + (i + 1) + "/" + imageFiles.size() + " (Skipped: " + skippedUpscale + ")");
		            continue;
	        	}
	        }

	        monitor.subTask("Upscaling " + (i + 1) + "/" + imageFiles.size() + " (Skipped: " + skippedUpscale + ")");
			ProcessBuilder upscalePb = new ProcessBuilder(realesrganExe, "-i", inputFile.getAbsolutePath(), "-o",outputFile.getAbsolutePath(), "-s", String.valueOf(scaleFactor));
			Process process = upscalePb.start();
			process.waitFor();
			if(verbose)
				System.out.println("✅ Upscaled: " + inputFile.getName() + " in folder: " + folderName + " (" + (i + 1) + "/"+ imageFiles.size() + ")");
		}
	}
	
	
	private void cropImages(File inputDir,File tempUpscaleDir, File outputDir,ScaleMode scaleMode,int maxSize, IProgressMonitor monitor,boolean verbose) throws IOException, InterruptedException {
		String resizeExe = new File(WorkspaceConfiguration.RESIZE_TOOLS, "resize.exe").getAbsolutePath();
		List<File> upscaledFiles = Files.walk(tempUpscaleDir.toPath())
				.filter(Files::isRegularFile)
				.map(Path::toFile)
				.filter(f -> f.getName().toLowerCase().matches(".*\\.(jpg|jpeg|png)"))
				.collect(Collectors.toList());

	    for (int i = 0; i < upscaledFiles.size(); i++) {
	        if (monitor.isCanceled()) {
	            System.out.println("❌ Cropping cancelled.");
	            throw new InterruptedException("Cropping cancelled by user.");
	        }

	        File upscaledFile = upscaledFiles.get(i);
	        Path relativePath = tempUpscaleDir.toPath().relativize(upscaledFile.toPath());

	        
	        File finalOutputFile;
	        if (scaleMode == ScaleMode.FIX_NP2_TO_MAIN) {
	            finalOutputFile = new File(inputDir, relativePath.toString());
	        } else {
	            finalOutputFile = new File(outputDir, relativePath.toString());
	        }

	        finalOutputFile.getParentFile().mkdirs();

	        monitor.subTask("Cropping " + (i + 1) + "/" + upscaledFiles.size());

			ProcessBuilder resizePb = new ProcessBuilder(resizeExe, upscaledFile.getAbsolutePath(),
					finalOutputFile.getAbsolutePath(), "--max-size", String.valueOf(maxSize));
	        Process resizeProc = resizePb.start();
	        resizeProc.waitFor();

	        String folderName = (relativePath.getParent() != null) ? relativePath.getParent().toString() : "(root)";
	        if(verbose)
	        	System.out.println("✅ Cropped: " + upscaledFile.getName() + " in folder: " + folderName + " (" + (i + 1) + "/" + upscaledFiles.size() + ")");
		}
	}
	
	
	private void deleteDirectoryRecursively(File directory) throws IOException {
	    if (directory.exists()) {
	        Files.walk(directory.toPath())
	            .sorted(Comparator.reverseOrder())
	            .map(Path::toFile)
	            .forEach(File::delete);
	    }
	}

	private boolean isPowerOfTwo(int x) {
	    return (x > 0) && ((x & (x - 1)) == 0);
	}

	private void runDownloadTask(TextureComboView comboView) {
		Shell shell = Display.getDefault().getActiveShell();
		ProgressMonitorDialog dialog = new ProgressMonitorDialog(shell);

		try {
			dialog.run(true, true, monitor -> {
				monitor.beginTask("Téléchargement des textures...", comboView.getComboCount().intValue());
				try {
					performDownload(comboView, monitor);
				} catch (IOException | InterruptedException e) {
					throw new InvocationTargetException(e);
				} finally {
					monitor.done();
				}
			});
		} catch (InvocationTargetException | InterruptedException e) {
			Throwable cause = e.getCause();
			if (cause instanceof InterruptedException) {
				MessageDialog.openWarning(shell, "Annulé", "L'action a été annulée. Elle sera reprise la prochaine fois, si la reprise est prise en charge.");
			} else {
				MessageDialog.openError(shell, "Erreur", "Échec de l'importation :\n" + cause.getMessage());
				cause.printStackTrace();
			}
		}
	}

	
	private void performDownload(TextureComboView comboView, IProgressMonitor monitor) throws IOException, InterruptedException {

		String requestUrl = buildTextureCombinationRequestUrl(BASE_URL, comboView);
		int skipped = 0;
		int downloaded = 0;

		TextureCombinationResponse textureCombinationResponse = UtilDTO.parseUrlResposne(new URL(requestUrl), TextureCombinationResponse.class);

		for (int i = 0; i < textureCombinationResponse.size(); i++) {
			if (monitor.isCanceled())
				throw new InterruptedException("Téléchargement annulé par l'utilisateur");
			
			TextureCombination textureCombination = textureCombinationResponse.get(i);
			monitor.subTask("Téléchargement " + (i + 1) + "/" + textureCombinationResponse.size() + " : "+ textureCombination.label);

			if (materialExists(textureCombination)) {
				skipped++;
				continue;
			}
			Material newMaterial = createMaterial(textureCombination);
			addMaterialToSystem(newMaterial);
			downloaded++;
			monitor.worked(1);
		}

		ResourceManagers.getIntance().saveLibraryResource();
		showDownloadSummary(downloaded, skipped);

	}

	private boolean materialExists(TextureCombination textureCombination) {
		return ResourceManagers.getIntance().getMaterials().stream().anyMatch(m -> {
			boolean matched = m.getImportID() == textureCombination.getId();
//			if (matched)
//				System.out.println("Matched material: " + m.getName() + " with label: " + textureCombination.getLabel());
			return matched;
		});
	}

	private void addMaterialToSystem(Material material) {
//		boolean isDev = ReportingPreferences.getInstance().getProperty("user.dev", false);
//		if (isDev) {
		ResourceManagers.getIntance().getSystemMaterialGroup().getMaterial().add(material);
//		} else {
//			ResourceManagers.getIntance().getMaterialGroup().getMaterial().add(material);
//		}

	}

	private Material createMaterial(TextureCombination textureCombination) {

		Material materialWeb = new MaterialImpl();

		if (textureCombination.texture.getImage() != null) {
			materialWeb.setImage(downloadImage(textureCombination).toString());
		}
		materialWeb.setEpaisseur(textureCombination.thickness.value);
		materialWeb.setName(textureCombination.label);
		materialWeb.setSensFil(true);
		materialWeb.setApplyTexture(true);
		materialWeb.setImportID(textureCombination.getId());
		materialWeb.setFabricant(getOrCreateFabricant(textureCombination));
		materialWeb.setFinition(getOrCreateFinition(textureCombination));
		materialWeb.setNature_type(getOrCreateNature(textureCombination));

		return materialWeb;

	}

	private Fournisseur_fabricant getOrCreateFabricant(TextureCombination textureCombination) {
		return ResourceManagers.getIntance().getGestion().getFabricants().getFabricants().stream()
				.filter(f -> f.getName().equalsIgnoreCase(textureCombination.texture.source)).findFirst()
				.orElseGet(() -> {
					Fournisseur_fabricantImpl fournisseur = new Fournisseur_fabricantImpl();
					fournisseur.setName(textureCombination.texture.source);
					ResourceManagers.getIntance().getGestion().getFabricants().getFabricants().add(fournisseur);
					ResourceManagers.getIntance().saveLibraryResource();
					return fournisseur;
				});
	}

	private Finition getOrCreateFinition(TextureCombination textureCombination) {
		return ResourceManagers.getIntance().getGestion().getFinitions().getFinitions().stream()
				.filter(f -> f.getName().equalsIgnoreCase(textureCombination.texture.type)).findFirst()
				.orElseGet(() -> {
					FinitionImpl finition = new FinitionImpl();
					finition.setName(textureCombination.texture.type);
					ResourceManagers.getIntance().getGestion().getFinitions().getFinitions().add(finition);
					ResourceManagers.getIntance().saveLibraryResource();
					return finition;
				});
	}

	private Nature_panneaux getOrCreateNature(TextureCombination textureCombination) {
		return ResourceManagers.getIntance().getGestion().getNaturePanneaux().getNatures().stream()
				.filter(t -> t.getName().equalsIgnoreCase(textureCombination.support.name)).findFirst()
				.orElseGet(() -> {
					Nature_panneauxImpl type = new Nature_panneauxImpl();
					type.setName(textureCombination.support.name);
					ResourceManagers.getIntance().getGestion().getNaturePanneaux().getNatures().add(type);
					ResourceManagers.getIntance().saveLibraryResource();
					return type;
				});
	}

	private void showDownloadSummary(int downloaded, int skipped) {
		String message = String.format("Download complete!\nDownloaded: %d\nSkipped: %d", downloaded, skipped);
		Display.getDefault().asyncExec(() -> {
			MessageDialog.openInformation(Display.getDefault().getActiveShell(), "Download Summary", message);
		});
	}

	private Path downloadImage(TextureCombination textureCombination) {
		String imageUrl = textureCombination.getTexture().getImage().getRowUrl();
		try {
			String fileName = imageUrl.substring(imageUrl.lastIndexOf("/") + 1);
			URL url = new URL(imageUrl);
			InputStream in = url.openStream();
			Path destinationPath = Paths.get(WorkspaceConfiguration.TEXTURES_FOLDER, fileName);
			Files.createDirectories(destinationPath.getParent());
			Files.copy(in, destinationPath, StandardCopyOption.REPLACE_EXISTING);

			final String label = textureCombination.getTexture().getName() + " "
					+ textureCombination.getTexture().getSource();
			SingleImageItemLabelProvider.getInstance().setFileLabelProperty(destinationPath, label);
			in.close();
			return destinationPath;
		} catch (IOException e) {
			e.printStackTrace();
			System.out.println("Failed to download image: " + e.getMessage());
			return null;
		}

	}

	public String buildTextureCombinationRequestUrl(String baseUrl, TextureComboView textureComboView) {
		StringBuilder requestUrlBuilder = new StringBuilder();
		requestUrlBuilder.append(baseUrl).append("/texture-combinations/filter?").append("supportId=")
				.append(textureComboView.comboSupportId).append("&textureType=").append(textureComboView.comboType)
				.append("&textureSource=").append(textureComboView.comboSource);

		return requestUrlBuilder.toString();
	}

	private enum ScaleMode {
	    FIX_NP2_TO_MAIN,
	    FIX_ALL_TO_SUB;

	    public static ScaleMode fromString(String mode) {
	        return switch (mode) {
	            case "FixNP2-ToMain" -> FIX_NP2_TO_MAIN;
	            case "FixAll-ToSub" -> FIX_ALL_TO_SUB;
	            default -> throw new IllegalArgumentException("Unknown scale mode: " + mode);
	        };
	    }
	}

}
