package gdxapp.assets;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;

import org.eclipse.swt.widgets.Display;
import org.json.simple.JSONObject;
import org.lwjgl.util.vector.Matrix;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.g3d.Material;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.model.Node;
import com.badlogic.gdx.graphics.g3d.model.NodePart;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import dressing.io.IOUtilities;
import dressing.model.ModelProvider;
import dressing.ui.ProgressBarMonitor;
import gdxapp.object3d.DesignModelFactory;
import gdxapp.object3d.KitchenElement;
import gdxapp.object3d.ModelInfo;
import gdxapp.object3d.Object3D;
import gdxapp.object3d.Object3DFactory;
import gdxapp.object3d.ObjectModel;
import gdxapp.object3d.WorldObject;
import gdxapp.shaders.PbrMaterial;

public class ModelExporter {

	private Model model;
	private Matrix4 transform;

	public static final String header = "# Supercad Wavefront OBJ Exporter v1.0 - (c)2020 FRS";
	private HashMap<Mesh, float[][]> meshsData;
	private ArrayList<NodePartData> nodesData;
	private ArrayList<String> materialDeclarations;
	
	private final StringBuilder strPosition = new StringBuilder();
	private final StringBuilder strNormal = new StringBuilder();
	private final StringBuilder strTexture = new StringBuilder();
	private final StringBuilder strIndices = new StringBuilder();

	private String dotObj = "";
	private String dotMtl = "";
	private String transformString = "";
	private Map<File, String> textures = new HashMap<File, String>();
	
	private String directory;
	private String textureDirectory;
	private String objectName;
	
	public void exportModel(Model model, String directory, String modelName, String textureDirectory) {
		this.model = model;
		this.directory = directory;
		this.objectName = modelName;
		this.textureDirectory = textureDirectory;
		try {
		meshsData = new HashMap<Mesh, float[][]>();
		for (Mesh mesh : model.meshes) {
			int vertexSize = mesh.getVertexSize() / 4;
			int verticesCount = mesh.getNumVertices();
			float[] vertexData = new float[vertexSize * verticesCount];
			mesh.getVertices(vertexData);
			float[][] verticesData = new float[verticesCount][vertexSize];
			for (int i = 0; i < verticesCount; i++) {
				for (int j = 0; j < vertexSize; j++) {
					verticesData[i][j] = vertexData[vertexSize * i + j];
				}
			}
			meshsData.put(mesh, verticesData);
		}

		nodesData = new ArrayList<ModelExporter.NodePartData>();
		for (Node node : model.nodes) {
			if(node.id.equals("croquis"))
				continue;
			processNode(node, new Matrix4());
		}
		encode();
		createDotObjContent(modelName);
		createDotMTLContent();
		createTransformContent();
		saveFiles();
		} catch (Exception e) {
			exportTransform(this.directory, this.objectName);
			e.printStackTrace();
		}finally {
			clear();
			System.gc();
		}
	}
	
	
	public void exportMechanicKitchenElement(KitchenElement element,String dir, String modelName, String textureDir) {
		CountDownLatch latch = new CountDownLatch(1);
		Gdx.app.postRunnable(() -> {
				try {
					var object3D = Object3DFactory.create3DObject(element);
					exportModel(object3D.model, dir, modelName, textureDir);
					HashMap<Model, ArrayList<Matrix4>> instances = new HashMap<Model, ArrayList<Matrix4>>();
					for(Object3D addon: object3D.getAddons()) {
						ArrayList<Matrix4>  list = instances.get(addon.model);
						if(list == null) {
							list = new ArrayList<Matrix4>();
							instances.put(addon.model, list);
						}
						list.add(addon.transform);			
					}
					int c = 0;
					for(Model model: instances.keySet()) {
						String folder =  dir + File.separator + c++;
						exportModel(model, folder, "model", textureDir);
						String transform = "";
						for(var mat4: instances.get(model)) {
							transform += Arrays.toString(mat4.getValues()) + System.lineSeparator();
						}
						File file = new File(folder + File.separator + "transforms.txt");
						FileOutputStream fos;
						try {
							file.createNewFile();
							fos = new FileOutputStream(file);
							fos.write(transform.getBytes());
							fos.close();
						} catch (IOException e) {
							e.printStackTrace();
						}
						
					}
					
				}finally {
					latch.countDown();
				}
			});
		try {
			latch.await();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		
		
	}
	
	
	public void exportModel(String directory, String modelName, UUID id) {

		WorldObject object = ModelProvider.getModeledObjectById(id);
		ModelInfo info = object.getModel().getInfo();
		String path = (String) info.getProperties().get("path_to_HP");
		createTransformContent();
		JSONObject content = new JSONObject();
		content.put("path", path);
		content.put("transform", this.transformString);
		String out = content.toJSONString();
		String filePath = directory + File.separator + modelName + File.separator + modelName + ".json";
		File file = new File(filePath);
		if(!file.exists())
			file.getParentFile().mkdirs();
		try {
		file.createNewFile();
		FileOutputStream fos = null;
		fos = new FileOutputStream(file);
		fos.write(out.getBytes());
		fos.close();
		Display.getDefault().syncExec(new Runnable() {
			@Override
			public void run() {
				ProgressBarMonitor.advance(1);
				ProgressBarMonitor.labelTask("exporting object: " + modelName);
			}					
		});

		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		clear();
	}
	
	private void clear() {
		meshsData.clear();
		nodesData.clear();
		if(materialDeclarations != null)
			materialDeclarations.clear();
		
		strPosition.setLength(0);
		strIndices.setLength(0);
		strNormal.setLength(0);
		strTexture.setLength(0);
			
		dotObj = "";
		dotMtl = "";
		transformString = "";
	}
	
	public void clearRessources() {
		if(textures == null) {
			textures = new HashMap<File, String>();
		}else {
			textures.clear();
		}
	}

	public void exportTransform(String directory, String modelName) {
		this.directory = directory + File.separator + "3d scene";
		this.objectName = modelName;
		createTransformContent();
		try {
			File transformFile = new File(directory + File.separator + objectName + File.separator + objectName + ".txt");

	        if(!transformFile.exists()){
	        	transformFile.getParentFile().mkdirs();
	        	transformFile.createNewFile();
	        }
			FileOutputStream fos = null;

			fos = new FileOutputStream(transformFile);
			fos.write(transformString.getBytes());
			fos.close();
			} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	private void createTransformContent() {
		if(this.transform != null) {
			this.transformString = "";
			for(int i = 0; i < 16; i++) {
				this.transformString += String.valueOf(this.transform.getValues()[i]) + System.lineSeparator();
			}
		}
	}

	private void saveFiles() throws IOException {
		File objFile = new File(directory + File.separator + objectName + ".obj");
		File mtlFile = new File(directory + File.separator + objectName + ".mtl");
		

		if (!objFile.exists()) {
			objFile.getParentFile().mkdirs();
			objFile.createNewFile();
		}
        if(!mtlFile.exists()){
            mtlFile.getParentFile().mkdirs();
            mtlFile.createNewFile();
        }
        
		FileOutputStream fos = null;
		fos = new FileOutputStream(objFile);
		fos.write(dotObj.getBytes());
		fos.close();
		fos = new FileOutputStream(mtlFile);
		fos.write(dotMtl.getBytes());
		fos.close();
		if(transform != null) {
			File transformFile = new File(directory + File.separator + objectName + ".txt");
			if(!transformFile.exists()){
	        	transformFile.getParentFile().mkdirs();
	        	transformFile.createNewFile();
	        	fos = new FileOutputStream(transformFile);
	    		fos.write(transformString.getBytes());
	    		fos.close();
	        }
		}
		
		if(textures != null) {
			String folder = directory; 
			if(textureDirectory!= null &&  !textureDirectory.equals(""))
				folder += File.separator + textureDirectory;
			for (File original : textures.keySet()) {
				File destination = new File(folder + File.separator + textures.get(original));
				IOUtilities.copyFileUsingChannel(original, destination);
			}
		}
	}

	private void processNode(Node node, Matrix4 parentTransform) {
		for(Node nodeChild: node.getChildren()) {
			processNode(nodeChild, node.globalTransform.cpy().mul(nodeChild.localTransform));
		}

		for(NodePart part: node.parts) {
			processNodePart(part, parentTransform);
		}
	}

	private void processNodePart(NodePart nodePart, Matrix4 localTransform) {
		int size = Math.min(nodePart.meshPart.size, nodePart.meshPart.mesh.getNumVertices());
		int offset = nodePart.meshPart.offset;
		boolean indexedMesh = false;
		short[] partIndices = null;
		if(nodePart.meshPart.mesh.getNumIndices() > 0) {
			indexedMesh = true;
			partIndices = new short[nodePart.meshPart.size];
			nodePart.meshPart.mesh.getIndices(offset,nodePart.meshPart.size,partIndices,0);
		}
		
		float[][] mesh = meshsData.get(nodePart.meshPart.mesh);
		float[][] nodeData = new float[size][mesh[0].length];

		for (int i = offset; i < offset + size; i++) {
			nodeData[i - offset] = mesh[i];
		}
		NodePartData data = new NodePartData(nodeData, localTransform, nodePart.meshPart.mesh.getVertexAttributes(),
				nodePart.material, nodePart.meshPart.primitiveType, partIndices);
		this.nodesData.add(data);
		try {
			data.process();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private void encode() {
		int[] indexes = new int[this.nodesData.size()];
		indexes[0] = 0;
		for (int i = 1; i < nodesData.size(); i++) {
			indexes[i] = nodesData.get(i - 1).size + indexes[i-1];
		}
		for (int i = 0; i < nodesData.size(); i++) {
			try {
				encodePart(nodesData.get(i), indexes[i]);
			}catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	private void encodePart(NodePartData nodePartData, int offset) {
		for (int i = 0; i < nodePartData.size; i++) {
			strPosition.append(encodePosition(nodePartData.getPositions().get(i)));
			strNormal.append(encodeNormal(nodePartData.getNormals().get(i)));
			strTexture.append(encodeTextureCoords(nodePartData.getTextureCoords().get(i)));
		}
		strIndices.append("usemtl " + nodePartData.material.id + System.lineSeparator());
		if(nodePartData.getPositions().size() == nodePartData.indices.length) {
			int[][] indices = assemblePrimitives(nodePartData, offset);
			for (int j = 0; j < indices.length; j++) {
				strIndices.append("f  " + indices[j][0] + "/" + indices[j][0] + "/" + indices[j][0] + " " + indices[j][1]
						+ "/" + indices[j][1] + "/" + indices[j][1] + " " + indices[j][2] + "/" + indices[j][2] + "/"
						+ indices[j][2] + System.lineSeparator());
			}
		}else {
			for(int i = 0; i < nodePartData.indices.length; i++) {
				nodePartData.indices[i] += offset + 1;
			}
			for (int j = 0; j < nodePartData.indices.length / 3; j++) {
				strIndices.append("f  " + nodePartData.indices[3 * j]  + "/" + nodePartData.indices[3 * j] + "/"
								+ nodePartData.indices[3 * j] +  " " + nodePartData.indices[3 * j + 1]
						+ "/" + nodePartData.indices[3 * j + 1] + "/" + nodePartData.indices[3 * j + 1] + " " + nodePartData.indices[3 * j + 2] + "/" + nodePartData.indices[3 * j + 2] + "/"
						+ nodePartData.indices[3 * j + 2] + System.lineSeparator());
			}
		}
	}

	private int[][] assemblePrimitives(NodePartData nodePartData, int offset) {
		int[][] faces = null;
		if (nodePartData.getPrimitivetype() == GL20.GL_TRIANGLES) {
			faces = assembleTriangles(offset, offset + nodePartData.getSize());
		}
		return faces;
	}

	private int[][] assembleTriangles(int offset, int last) {
		int trianglesCount = (last - offset) / 3;
		int[][] triangles = new int[trianglesCount][3];
		for (int i = 0; i < trianglesCount; i++) {
			for (int j = 0; j < 3; j++) {
				triangles[i][j] = offset + 3 * i + j + 1;
			}
		}
		return triangles;
	}

	String encodePosition(Vector3 position) {
		return "v  " + valueOfFloat(position.x, 4) + "  " + valueOfFloat(position.y, 4) + "  "
				+ valueOfFloat(position.z, 4) + System.lineSeparator();
	}

	String encodeNormal(Vector3 normal) {
		return "vn  " + valueOfFloat(normal.x, 4) + "  " + valueOfFloat(normal.y, 4) + "  " + valueOfFloat(normal.z, 4)
				+ System.lineSeparator();
	}

	String encodeTextureCoords(Vector2 textureCoords) {
		return "vt  " + valueOfFloat(textureCoords.x, 4) + "  " + valueOfFloat(textureCoords.y, 4)
				+ System.lineSeparator();
	}

	private String encodeMaterial(Material material) throws IOException {
		String code = "newmtl " + material.id + System.lineSeparator();
		// default values
		code += "\t illum 2" + System.lineSeparator();
		if(material instanceof PbrMaterial ) {
			PbrMaterial pbrMtl = (PbrMaterial) material;
			String dir =  "";
			if(textureDirectory != null && !textureDirectory.trim().equals(""))
				dir =  textureDirectory + File.separator;
			if(pbrMtl.getAlbedo() != null)
				code += "\t" + String.format("Kd %s %s %s",pbrMtl.getAlbedo().x, pbrMtl.getAlbedo().y, pbrMtl.getAlbedo().z) + System.lineSeparator();
			code += "\t" + String.format("Pm %s",pbrMtl.getMetalness()) + System.lineSeparator();
			code += "\t" + String.format("Pr %s",pbrMtl.getRoughness()) + System.lineSeparator();
			code += "\t" + String.format("d %s",pbrMtl.getOpacity()) + System.lineSeparator();
			if(pbrMtl.getAlbedoMapPath() != null) {
				File textureFile = new File(pbrMtl.getAlbedoMapPath());
				if(!textures.keySet().contains(textureFile)) {
					String[] parts = textureFile.getAbsolutePath().split("\\.");
					String textureAlias =  "texture" + textures.size() + "." + parts[parts.length - 1] ;
					textures.put(textureFile, textureAlias);
				}
				code += "\t" + String.format("map_Kd %s", dir +  textures.get(textureFile)) + System.lineSeparator();
			}
			if(pbrMtl.getNormalMapPath() != null) {
				File textureFile = new File(pbrMtl.getNormalMapPath());
				if(!textures.keySet().contains(textureFile)) {
					String[] parts = textureFile.getAbsolutePath().split("\\.");
					String textureAlias =  "texture" + textures.size() + "." + parts[parts.length - 1] ;
					textures.put(textureFile, textureAlias);
				}
				code += "\t" + String.format("map_Kn %s",dir + textures.get(textureFile)) + System.lineSeparator();
			}
			if(pbrMtl.getMetalnessMapPath() != null) {
				File textureFile = new File(pbrMtl.getMetalnessMapPath());
				if(!textures.keySet().contains(textureFile)) {
					String[] parts = textureFile.getAbsolutePath().split("\\.");
					String textureAlias =  "texture" + textures.size() + "." + parts[parts.length - 1] ;
					textures.put(textureFile, textureAlias);
				}
				code += "\t" + String.format("map_Pm %s", dir + textures.get(textureFile)) + System.lineSeparator();
			}
			if(pbrMtl.getRoughnessMapPath() != null) {
				File textureFile = new File(pbrMtl.getRoughnessMapPath());
				if(!textures.keySet().contains(textureFile)) {
					String[] parts = textureFile.getAbsolutePath().split("\\.");
					String textureAlias =  "texture" + textures.size() + "." + parts[parts.length - 1] ;
					textures.put(textureFile, textureAlias);
				}
				code += "\t" + String.format("map_Pr %s", dir + textures.get(textureFile)) + System.lineSeparator();
			}
			if(pbrMtl.getAoMapPath() != null) {
				File textureFile = new File(pbrMtl.getAoMapPath());
				if(!textures.keySet().contains(textureFile)) {
					String[] parts = textureFile.getAbsolutePath().split("\\.");
					String textureAlias =  "texture" + textures.size() + "." + parts[parts.length - 1] ;
					textures.put(textureFile, textureAlias);
				}
				code += "\t" + String.format("map_Ka %s", dir + textures.get(textureFile)) + System.lineSeparator();
			}
			if(pbrMtl.getHeightMapPath() != null) {
				File textureFile = new File(pbrMtl.getHeightMapPath());
				if(!textures.keySet().contains(textureFile)) {
					String[] parts = textureFile.getAbsolutePath().split("\\.");
					String textureAlias =  "texture" + textures.size() + "." + parts[parts.length - 1] ;
					textures.put(textureFile, textureAlias);
				}
				code += "\t" + String.format("map_bump %s", dir + textures.get(textureFile)) + System.lineSeparator();
			}
			
			if (pbrMtl.getEmissive() != null) {
				code += "\t" + String.format("Ke  %s  %s  %s \n", pbrMtl.getEmissive().x, pbrMtl.getEmissive().y,
						pbrMtl.getEmissive().z)+ System.lineSeparator();
			}
		}
		return code;
	}





	private void createDotObjContent(String modelName) {
		dotObj = header + System.lineSeparator();
		dotObj += "mtllib " + modelName + ".mtl" + System.lineSeparator();
		dotObj += strPosition.toString();
		dotObj += strNormal.toString();
		dotObj += strTexture.toString();
		dotObj += strIndices.toString();
	}
	
	private void createDotMTLContent() {
		if (materialDeclarations == null)
			materialDeclarations = new ArrayList<String>();
		materialDeclarations.clear();
		ArrayList<String> materialsUsed = new ArrayList();
		dotMtl = "";
		for (Material material : model.materials) {
			if (!materialsUsed.contains(material.id)) {
				String code;
				try {
					code = encodeMaterial(material);
					materialDeclarations.add(code);
					materialsUsed.add(material.id);
					dotMtl += code;
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		
	}

	static String valueOfFloat(float f, int precision) {
		float value = (float) (Math.round(f * Math.pow(10, precision)) * Math.pow(10, -precision));
		return String.valueOf(value);
	}

	public Model getModel() {
		return model;
	}

	public void setModel(Model model) {
		this.model = model;
		
	}
	
	public void setTransform(Matrix4 transform) {
		this.transform = transform;
	}

	class NodePartData {

		float[][] vertices;
		ArrayList<Vector3> positions;
		ArrayList<Vector3> normals;
		ArrayList<Vector2> textureCoords;
		VertexAttributes attributes;
		Matrix4 localtransform;
		Material material;
		int primitivetype;
		short[] indices;

		int size;

		public NodePartData() {
		}

		public NodePartData(float[][] vertices, Matrix4 transform, VertexAttributes attributes, Material material,
				int primtive, short[] indices) {
			super();
			this.vertices = vertices;
			this.localtransform = transform;
			this.attributes = attributes;
			this.material = material;
			this.size = vertices.length;
			this.primitivetype = primtive;
			this.indices = indices;
		}

		public void process() throws Exception {
			this.positions = new ArrayList<Vector3>();
			boolean hasNormal = (this.attributes.findByUsage(VertexAttributes.Usage.Normal) != null);
			boolean hasTextureCoords = (this.attributes.findByUsage(VertexAttributes.Usage.TextureCoordinates) != null);
			Matrix3 normalMatrix = new Matrix3().set(this.localtransform).inv().transpose();
			int vertexSize = 3;
			if (hasNormal) {
				vertexSize += 3;
				this.normals = new ArrayList<Vector3>();
			}
			if (hasTextureCoords) {
				vertexSize += 2;
				this.textureCoords = new ArrayList<>();
			}
			if (vertices[0].length != vertexSize)
				throw new Exception("wrong format of vertex data ");
			for (int i = 0; i < vertices.length; i++) {
				Vector3 position = new Vector3(vertices[i][0], vertices[i][1], vertices[i][2]);
				this.positions.add(position.mul(localtransform));
				if (hasNormal) {
					Vector3 normal = new Vector3(vertices[i][3], vertices[i][4], vertices[i][5]);
					this.normals.add(normal.mul(normalMatrix));
				}
				if (hasTextureCoords) {
					Vector2 textcoords = new Vector2(vertices[i][6], vertices[i][7]);
					this.textureCoords.add(textcoords);
				}
			}
		}

		public ArrayList<Vector3> getPositions() {
			return positions;
		}

		public ArrayList<Vector3> getNormals() {
			return normals;
		}

		public ArrayList<Vector2> getTextureCoords() {
			return textureCoords;
		}

		public int getSize() {
			return size;
		}

		public int getPrimitivetype() {
			return primitivetype;
		}

		public void setPrimitivetype(int primitivetype) {
			this.primitivetype = primitivetype;
		}

	}
	

}
