package gdxapp.shaders;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Objects;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.g3d.Attribute;
import com.badlogic.gdx.graphics.g3d.Material;
import com.badlogic.gdx.graphics.g3d.attributes.BlendingAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.TextureAttribute;
import com.badlogic.gdx.math.Vector3;

import dressing.config.WorkspaceConfiguration;
import dressing.config.persistence.ResourceManagers;
import dressing.io.IOUtilities;
import dressing.model.persistence.dpos.PbrMaterialDPO;
import dressing.model.persistence.mappers.Persistable;
import gdxapp.assets.AssetsTextures;
import jakarta.xml.bind.annotation.XmlRootElement;
import param.MaterialType;

@XmlRootElement(name = "material")
@Persistable(persistableForm = PbrMaterialDPO.class)
public class PbrMaterial extends Material {

	public final static int BASECOLOR_MAP = 0;
	public final static int NORMAL_MAP = 1;
	public final static int METALNESS_MAP = 2;
	public final static int ROUGHNESS_MAP = 3;
	public final static int AMBIANT_OCCLUSION_MAP = 4;
	public final static int OPACITY_MAP = 5;

	private String name = "";
	transient private MaterialType materialType;

	private Vector3 albedo;
	private Vector3 emissive;
	private float metalness;
	private float roughness;
	private float ambiantOcclusion;
	private float opacity = 1.0f;

	private String albedoMapPath;
	private String normalMapPath;
	private String metalnessMapPath;
	private String roughnessMapPath;
	private String aoMapPath;
	private String opacityMapPath;

	transient private Texture albedoMap;
	transient private Texture normalMap;
	transient private Texture metalnessMap;
	transient private Texture roughnessMap;
	transient private Texture aoMap;

	private boolean ready = false;
	transient boolean saveLibrary = false;

	public PbrMaterial(String name) {
		super();
		this.name = name;
	}

	public PbrMaterial(MaterialType materialType) {
		super();
		this.name = materialType.getName();
		this.materialType = materialType;
		try {
			if (materialType.getAlbedo() != null)
				this.albedo = new Vector3(materialType.getAlbedo().getR(), materialType.getAlbedo().getG(),
						materialType.getAlbedo().getB());
			
			this.metalness = materialType.getMetalness();
			this.roughness = materialType.getRoughness();
			this.ambiantOcclusion = materialType.getAmbiantOcclusion();
			this.opacity = materialType.getOpacity();
			this.albedoMapPath = materialType.getAlbedoMapPath();
			this.normalMapPath = materialType.getNormalMapPath();
			this.metalnessMapPath = materialType.getMetalnessMapPath();
			this.roughnessMapPath = materialType.getRoughnessMapPath();
			this.aoMapPath = materialType.getAoMapPath();
		} catch (Exception e) {
			System.err.println("failed to load Obr material from materialtype");
		}
	}

	public PbrMaterial() {
		super();
	}

	public PbrMaterial(final Attribute... attributes) {
		super(attributes);
	}

	public PbrMaterial(String name, File albedoFile, File normalFile, File roughnessFile, File metalnessFile,
			File ambiantOcclusionFile) {
		this.name = name;
		if (albedoFile != null)
			this.albedoMapPath = albedoFile.getAbsolutePath();
		if (normalFile != null)
			this.normalMapPath = normalFile.getAbsolutePath();
		if (roughnessFile != null)
			this.roughnessMapPath = roughnessFile.getAbsolutePath();
		if (metalnessFile != null)
			this.metalnessMapPath = metalnessFile.getAbsolutePath();
		if (ambiantOcclusionFile != null)
			this.aoMapPath = ambiantOcclusionFile.getAbsolutePath();
	}

	// must be called when an opengl context is available not thread safe
	public void load() {
		validate();
		if(materialType != null) {
			cacheMaps();
		}
		try {
			if (albedoMapPath != null && albedoMap == null) {
				albedoMap = AssetsTextures.getInstance().getTexture(albedoMapPath);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private void validate() {
		if(albedoMapPath != null && !albedoMapPath.contentEquals("")) {
			if(!new File(albedoMapPath).exists())
				albedoMap = null;
		}
		
		if(normalMapPath != null) {
			if(!new File(normalMapPath).exists())
				normalMapPath = null;
		}
		
		if(roughnessMapPath != null) {
			if(!new File(roughnessMapPath).exists())
				roughnessMapPath = null;
		}
		
		if(metalnessMapPath != null) {
			if(!new File(metalnessMapPath).exists())
				metalnessMapPath = null;
		}
		
		if(aoMapPath != null) {
			if(!new File(aoMapPath).exists())
				aoMapPath = null;
		}
	}

	public void prepare() {
		if(ready)
			return;
		try {
			if (albedoMapPath != null) {
				albedoMap = AssetsTextures.getInstance().getTexture(albedoMapPath);
			}
			toGdxMaterial();
			ready = true;
		} catch (Exception e) {
			e.printStackTrace();
			ready = false;
		}
	}

	public void toGdxMaterial() {
		BlendingAttribute ba = new BlendingAttribute(this.opacity);
		set(ba);
		if (albedo != null)
			set(ColorAttribute.createDiffuse(new Color(albedo.x, albedo.y, albedo.z, 1.0f)));
		if (albedoMap != null) {
			set(TextureAttribute.createDiffuse(albedoMap));
		}
	}

	public void cacheMaps() {
		validate();
		if (albedoMapPath != null) {
			cacheMap(BASECOLOR_MAP);
		}
		if(normalMapPath != null) {
			cacheMap(NORMAL_MAP);
		}
		if(metalnessMapPath != null) {
			cacheMap(METALNESS_MAP);
		}
		if(roughnessMapPath != null) {
			cacheMap(ROUGHNESS_MAP);
		}
		if(aoMapPath != null) {
			cacheMap(AMBIANT_OCCLUSION_MAP);
		}
		if(saveLibrary)
			ResourceManagers.getIntance().saveLibraryResource();
			
	}

	public void cacheMap(int type) {
		Path path = Paths.get(getMapPath(type));
		if (!path.startsWith(WorkspaceConfiguration.MATERIALS_FOLDER)) {
			String destination = WorkspaceConfiguration.MATERIALS_FOLDER + File.separator + name + File.separator
					+ path.getFileName();
			cacheMap(getMapPath(type), destination);
			setMapPath(type, destination);
			if (materialType != null) {
				switch (type) {
				case BASECOLOR_MAP:
					materialType.setAlbedoMapPath(this.albedoMapPath);
					break;
				case NORMAL_MAP:
					materialType.setNormalMapPath(this.normalMapPath);
					break;
				case METALNESS_MAP:
					materialType.setMetalnessMapPath(this.metalnessMapPath);
					break;
				case ROUGHNESS_MAP:
					materialType.setRoughnessMapPath(this.roughnessMapPath);
					break;
				case AMBIANT_OCCLUSION_MAP:
					materialType.setAoMapPath(this.aoMapPath);
					break;
				}
				saveLibrary = true;
			}
		}
	}

	public void cacheMap(String source, String location) {
		File sourceFile = new File(source);
		File destFile = new File(location);
		try {
			IOUtilities.copyFileUsingChannel(sourceFile, destFile);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return;
		}
	}

	public String getMapPath(int map) {
		switch (map) {
		case BASECOLOR_MAP:
			return albedoMapPath;
		case NORMAL_MAP:
			return normalMapPath;
		case METALNESS_MAP:
			return metalnessMapPath;
		case ROUGHNESS_MAP:
			return roughnessMapPath;
		case AMBIANT_OCCLUSION_MAP:
			return aoMapPath;
		default:
			return "";
		}
	}
	public void setMapPath(int map, String path) {
		switch (map) {
		case BASECOLOR_MAP:
			 albedoMapPath = path;
		case NORMAL_MAP:
			normalMapPath = path;
		case METALNESS_MAP:
			metalnessMapPath = path;
		case ROUGHNESS_MAP:
			roughnessMapPath = path;
		case AMBIANT_OCCLUSION_MAP:
			aoMapPath = path;
			break;
		}
	}

	public static String extractFileNameFromPath(String filePath) {
		if (filePath == null || filePath == "")
			return null;
		File file = new File(filePath);
		String fileNameWithExtension = file.getName();
		int length = fileNameWithExtension.length();
		return (String) fileNameWithExtension.subSequence(0, length - 4);
	}

	public PbrMaterial cpy() {
		PbrMaterial clone = new PbrMaterial();
		if (this.name != null)
			clone.name = new String(this.name);
		if (this.albedo != null)
			clone.albedo = new Vector3(this.albedo);
		if(this.emissive != null)
			clone.setEmissive(emissive);
		clone.metalness = this.metalness;
		clone.roughness = this.roughness;
		clone.ambiantOcclusion = this.ambiantOcclusion;
		clone.opacity = this.opacity;
		if(albedoMapPath != null)
			clone.albedoMapPath = new String(this.albedoMapPath);
		if(normalMapPath != null)
			clone.normalMapPath = new String(this.normalMapPath);
		if(metalnessMapPath != null)
			clone.metalnessMapPath = new String(this.metalnessMapPath);
		if(roughnessMapPath != null)
			clone.roughnessMapPath = new String(this.roughnessMapPath);
		if(aoMapPath != null)
			clone.aoMapPath = new String(this.aoMapPath);
		return clone;
	}

	// getter and setters
	public String getName() {
		return name!=null?name:"";
	}

	public void setName(String name) {
		this.name = name;
	}

	public Vector3 getAlbedo() {
		return albedo;
	}

	public void setAlbedo(Vector3 albedo) {
		this.albedo = albedo;
		ready = false;
	}

	public Vector3 getEmissive() {
		return emissive;
	}

	public void setEmissive(Vector3 emissive) {
		this.emissive = emissive;
	}

	public float getMetalness() {
		return metalness;
	}

	public void setMetalness(float metalness) {
		this.metalness = metalness;
		ready = false;
	}

	public float getRoughness() {
		return roughness;
	}

	public void setRoughness(float roughness) {
		this.roughness = roughness;
		ready = false;
	}

	public float getAmbiantOcclusion() {
		return ambiantOcclusion;
	}

	public void setAmbiantOcclusion(float ambiantOcclusion) {
		this.ambiantOcclusion = ambiantOcclusion;
		ready = false;
	}

	public float getOpacity() {
		return opacity;
	}

	public void setOpacity(float opacity) {
		this.opacity = opacity;
		ready = false;
	}

	public String getAlbedoMapPath() {
		return albedoMapPath;
	}

	public void setAlbedoMapPath(String albedoMapPath) {
		this.albedoMapPath = albedoMapPath;
		ready = false;
	}

	public String getNormalMapPath() {
		return normalMapPath;
	}

	public void setNormalMapPath(String normalMapPath) {
		this.normalMapPath = normalMapPath;
		ready = false;
	}

	public String getMetalnessMapPath() {
		return metalnessMapPath;
	}

	public void setMetalnessMapPath(String metalnessMapPath) {
		this.metalnessMapPath = metalnessMapPath;
		ready = false;
	}

	public String getRoughnessMapPath() {
		return roughnessMapPath;
	}

	public void setRoughnessMapPath(String roughnessMapPath) {
		this.roughnessMapPath = roughnessMapPath;
		ready = false;

	}

	public String getAoMapPath() {
		return aoMapPath;
	}

	public void setAoMapPath(String aoMapPath) {
		this.aoMapPath = aoMapPath;
		ready = false;
	}

	public String getOpacityMapPath() {
		return opacityMapPath;
	}

	public void setOpacityMapPath(String opacityMapPath) {
		this.opacityMapPath = opacityMapPath;
		ready = false;
	}

	public Texture getAlbedoMap() {
		if(albedoMap == null && albedoMapPath != null)
			prepare();
		return albedoMap;
	}

	public void setAlbedoMap(Texture albedoMap) {
		this.albedoMap = albedoMap;
	}

	public boolean isReady() {
		return ready;
	}
	

	@Override
	public String toString() {
		return name;
	}
	
	
	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + Objects.hash(albedo, albedoMapPath, ambiantOcclusion, aoMapPath, metalness,
				metalnessMapPath, name, normalMapPath, opacity, opacityMapPath, ready, roughness, roughnessMapPath);
		return result;
	}

	
	public boolean equals(PbrMaterial other) {
		
		return Objects.equals(albedo, other.albedo)
				&& Objects.equals(emissive, other.emissive)
				&& Objects.equals(albedoMapPath, other.albedoMapPath)
				&& Float.floatToIntBits(ambiantOcclusion) == Float.floatToIntBits(other.ambiantOcclusion)
				&& Objects.equals(aoMapPath, other.aoMapPath)
				&& Float.floatToIntBits(metalness) == Float.floatToIntBits(other.metalness)
				&& Objects.equals(metalnessMapPath, other.metalnessMapPath) 
				&& Objects.equals(normalMapPath, other.normalMapPath)
				&& Float.floatToIntBits(opacity) == Float.floatToIntBits(other.opacity)
				&& Objects.equals(opacityMapPath, other.opacityMapPath) 
				&& Float.floatToIntBits(roughness) == Float.floatToIntBits(other.roughness)
				&& Objects.equals(roughnessMapPath, other.roughnessMapPath);
	}
}
