package gdxapp.object3d;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.loader.ObjLoader;
import com.badlogic.gdx.graphics.g3d.model.MeshPart;
import com.badlogic.gdx.graphics.g3d.model.Node;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector3;
import dressing.config.WorkspaceConfiguration;
import dressing.model.Cavity;
import dressing.model.Piece2D;
import dressing.model.Space3DFree;
import dressing.model.types.PoigneeType;

public class Object3DFactory {
	
	private static Model footModel;
	
	public static Object3D create3DObject(KitchenElement element) {
		
		Model elementModel =  DesignModelFactory.getFactory().createModelFromVertices(element);
		ObjectModel objModel = new ObjectModel(elementModel, new ModelInfo());
		element.setModel(objModel);
		Object3D object3D = new Object3D(element);
		//create handlers
		object3D.addOns(createHandlers(element).toArray(new Object3D[0]));
		var family = WorldObjectFactory.getDesignFamily(element);
		if(family == WorldObjectFactory.Bas || family == WorldObjectFactory.colonne ) {
			object3D.addOns(createFeet(element).toArray(new Object3D[0]));
		}
		return object3D;
	}
	
	
	private static List<Object3D> createFeet(KitchenElement element) {
		List<Object3D> feet = new ArrayList<Object3D>();
		Optional<Piece2D> result = element.getDesignObject().getListPieces().stream()
							.filter( (piece) -> piece.getPiecetype().isBase()).findFirst( );
		if(result.isPresent()) {
			Piece2D base = result.get();
			Vector3 size =  base.getSize().scl(0.0004f);
			Vector3 center = base.getCenter().scl(0.001f);
			ArrayList<Vector3> positions = new ArrayList <>();
			positions.add(center.cpy().add(size.cpy().scl(1,0,1)));
			positions.add(center.cpy().add(size.cpy().scl(1,0,-1)));
			float z = element.getDesignObject().getDesignCaissonType().contains("COINS_L")?0:1; 
			positions.add(center.cpy().add(size.cpy().scl(-1,0,z)));
			positions.add(center.cpy().add(size.cpy().scl(-1,0,-1)));
			for(var position: positions) {
				Object3D foot = new Object3D(getFootModel());
				float scale = 0.15f/ foot.getDimension().y;
				position.sub(0,.075f,0);
				Matrix4 transform = new Matrix4(position, new Quaternion(Vector3.Y, 0), new Vector3(1,1,1).scl(scale));
				foot.transform.set(transform);
				feet.add(foot);
			}
		}
		
		return feet;
	}


	public static List<Object3D> createHandlers(KitchenElement element){
		var handlerType =  element.getDesignObject().getPoigneeType();
		if(handlerType == PoigneeType.SANS_POIGNEE) {
			return createProfiles(element);
		}else {
			return createKnobs(element);
		}
	}
	

	private static List<Object3D> createKnobs(KitchenElement element) {
		List<Piece2D> facades = element.getDesignObject().getListPieces().stream().filter((piece) -> {
			return piece.getPiecetype().isFacade();
		}).toList();
		
		List<Object3D> handlers = new ArrayList<Object3D>();
		for(var facade: facades) {
			Matrix4 localTransform = DefaultHandlePositioner.getPositioner().calculateHandlerTransform(facade, element);
			if(localTransform != null) {				
				var handle = element.getDoorHandle(facade.getName());
				Model model = handle.getModel();
				ModelBuilder modelBuilder = new ModelBuilder();
				modelBuilder.begin();
				Node knobNode = modelBuilder.node();
				knobNode.id = "handle_" + UUID.randomUUID();
				MeshPart meshPart = new MeshPart("part" + knobNode.id,
						model.meshParts.first().mesh.copy(false), model.meshParts.first().offset,
						model.meshParts.first().size, GL20.GL_TRIANGLES);
				meshPart.update();
				var mtl = handle.getMaterial();
				if(!mtl.isReady())
					mtl.prepare();
				modelBuilder.part(meshPart, mtl);
				var modelX = modelBuilder.end();
				Object3D handler = new Object3D(modelX);
				handler.transform.set(localTransform);
				handlers.add(handler);
			}
		}
		return handlers;		
	}


	private static List<Object3D> createProfiles(KitchenElement element) {
		Profile profile = element.getProfile();
		if(profile instanceof GolaProfile) {
			return createGolaProfiles(element);
		}else if(profile instanceof JPullProfile) {
			return createJpullProfiles(element);
		}
		return new ArrayList<Object3D>();
		
	}


	private static List<Object3D> createJpullProfiles(KitchenElement element) {
		List<Object3D> profiles = new ArrayList<Object3D>();
		List<Piece2D> facades = element.getDesignObject().getListPieces().stream().filter((piece) -> {
			return piece.getPiecetype().isFacade();
		}).toList();
		
		ProfilePositioner profilePositioner = new ProfilePositioner();
		profilePositioner.init();
		JPullProfile profile = (JPullProfile) element.getProfile();
		for(var facade: facades) {
			var transform = profilePositioner.calculateProfileTransform (facade, element);
			Object3D profileObj = new Object3D(profile.getModel());
			profileObj.transform.set(transform);
			profiles.add(profileObj);
		}
		
		return profiles;
	}


	private static List<Object3D> createGolaProfiles(KitchenElement element) {
		ArrayList<Object3D> profiles = new ArrayList<Object3D>();
		Space3DFree caisson = (Space3DFree) element.getDesignObject();
		HashMap<Cavity, Cavity> cavityPairs = new HashMap<Cavity, Cavity>();
		List<Cavity> cavities =  new ArrayList<Cavity>( caisson.getUsinages(caisson).stream().
				filter( usinage -> usinage instanceof Cavity && usinage.getName().contains("extrution sans poignee"))
				.map(u -> (Cavity) u).toList());
		float epsilon = 0.01f;
		boolean stop;
		do{
			stop = true;
			for(int i = 0; i < cavities.size() - 1; i++) {
				var cavity = cavities.get(i);
				Vector3 size = cavity.getSize();
				Vector3 position = cavity.getPosition();
				Cavity opposite = null;
				for(int j = i+1; j < cavities.size(); j++) {
					var other = cavities.get(j);
					if(Math.abs(other.getSize().y - size.y) < epsilon && 
							Math.abs(other.getPosition().y - position.y) < epsilon) {
						opposite = other;
						break;
					}	
				}
				if(opposite != null) {
					stop = false;
					cavityPairs.put(cavity, opposite);
					cavities.remove(cavity);
					cavities.remove(opposite);
					break;
				}
			}
		}while(!stop);
		Profile profile = element.getProfile();
		profile.createModel();
		Vector3 modelSize = profile.getDimension();
		for(var pair: cavityPairs.entrySet()) {
			var cavity = pair.getKey();
			var opposite = pair.getValue();
			Piece2D cavityPiece = (Piece2D) cavity.getMother();
			Piece2D oppositePiece = (Piece2D) opposite.getMother();
			Vector3 halfSize = cavity.getSize().scl(0.0005f); //the two cavities have the same size
			Vector3 v0 = cavity.getPosition().scl(0.001f).add(halfSize);
			v0.add(cavityPiece.getPosition().scl(0.001f));
			Vector3 v1 = opposite.getPosition().scl(0.001f).add(halfSize);
			v1.add(oppositePiece.getPosition().scl(0.001f));
			Vector3 translation = v1.cpy().sub(v0);
			Vector3 translationX = translation.cpy().scl(1,1,0);
			Vector3 translationZ = translation.cpy().scl(0,1,1);
			Vector3 m = null;
			float thickness = Math.min(halfSize.x, halfSize.z) * 2;
			if(translationX.len() > epsilon) {
				m = v0.cpy().add(translationX);
				Vector3 center = m.cpy().add(v0).scl(0.5f);
				Vector3 scale = new Vector3((translationX.len() + thickness)/modelSize.x, halfSize.y * 2/modelSize.y, halfSize.z * 2/modelSize.z);
				Matrix4 transform = new Matrix4().translate(center).scale(scale.x, scale.y, scale.z);
				Object3D profileX = new Object3D(profile.getModel());
				profileX.transform.set(transform);
				profiles.add(profileX);
			}
			if(translationZ.len() > epsilon) {
				Vector3 m2 = m.cpy().add(translationZ);
				Vector3 center = m2.cpy().add(m).scl(0.5f);
				Vector3 scale = new Vector3((translationZ.len() + thickness)/modelSize.x, halfSize.y * 2/modelSize.y, halfSize.z/modelSize.z);
				Matrix4 transform = new Matrix4().translate(center).scale(scale.x, scale.y, scale.z).rotate(Vector3.Y, -90);
				Object3D profileZ = new Object3D(profile.getModel());
				profileZ.transform.set(transform);
				profiles.add(profileZ);
			}
		}
		return profiles;
	}


	public static void loadFootModel() {
		ObjLoader objLoader = new ObjLoader();
		footModel = objLoader.loadModel(Gdx.files.absolute(WorkspaceConfiguration.ELEMENT_FOOT_PATH));
	}
	
	public static Model getFootModel() {
		if(footModel == null)
			loadFootModel();
		return footModel;
	}


}
