package geometry;

import java.util.ArrayList;
import java.util.Arrays;

import org.frs.svg.SVGWriter;

import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.math.Quaternion;

import dressing.mathutils.MathUtilities;

public class Box {

	private Vector3 center;
	private Vector3 dimensions;
	private Quaternion rotation = new Quaternion();

	private final Vector3 corner000 = new Vector3();
	private final Vector3 corner001 = new Vector3();
	private final Vector3 corner010 = new Vector3();
	private final Vector3 corner011 = new Vector3();
	private final Vector3 corner100 = new Vector3();
	private final Vector3 corner101 = new Vector3();
	private final Vector3 corner110 = new Vector3();
	private final Vector3 corner111 = new Vector3();

	private final Vector3[] corners = new Vector3[] { corner000, corner001, corner010, corner011, corner100, corner101,
			corner110, corner111 };

	private final Matrix4 relativeTransform = new Matrix4();// the transform from world space to the object space

	// the center and dimensions in world coords
	public Box(Vector3 center, Vector3 dimensions) {
		this.center = center;
		this.dimensions = dimensions;
		calculateCorners();
	}

	// min and max and the transform to world coords
	public Box(Vector3 min, Vector3 max, Matrix4 transform) {
		this.center = min.cpy().add(max).scl(0.5f);
		this.dimensions = max.cpy().sub(min);
		calculateCorners();
		transform(transform);
	}

	public void calculateCorners() {
		corner000.set(dimensions.cpy().scl(-0.5f).mul(rotation).add(center));
		corner001.set(dimensions.cpy().scl(0.5f).scl(-1, -1, 1).mul(rotation).add(center));
		corner010.set(dimensions.cpy().scl(0.5f).scl(-1, 1, -1).mul(rotation).add(center));
		corner011.set(dimensions.cpy().scl(0.5f).scl(-1, 1, 1).mul(rotation).add(center));
		corner100.set(dimensions.cpy().scl(0.5f).scl(1, -1, -1).mul(rotation).add(center));
		corner101.set(dimensions.cpy().scl(0.5f).scl(1, -1, 1).mul(rotation).add(center));
		corner110.set(dimensions.cpy().scl(0.5f).scl(1, 1, -1).mul(rotation).add(center));
		corner111.set(dimensions.cpy().scl(0.5f).mul(rotation).add(center));
	}

	private void calculatePropertiesFromCorners() {
		Vector3[] bounds = MathUtilities.getBoundaries(Arrays.asList(corners));
		this.center = bounds[0].cpy().add(bounds[1]).scl(0.5f);
		this.dimensions = bounds[1].cpy().sub(bounds[0]);
	}

	public void transform(Matrix4 transform) {
		this.relativeTransform.mul(transform.cpy().inv());

		for (int i = 0; i < corners.length; i++) {
			corners[i].mul(transform);
		}
		calculatePropertiesFromCorners();
		calculateCorners();
	}

	public void toLocalTransform() {
		Vector3 X = null, Y = null, Z = null;
		float width = Math.max(Math.max(dimensions.x, dimensions.y), dimensions.z);
		float height = Math.min(Math.min(dimensions.x, dimensions.y), dimensions.z);
		float depth = dimensions.dot(1, 1, 1) - width - height;
		ArrayList<Vector3> axes = new ArrayList<Vector3>();
		axes.add(X);
		axes.add(Y);
		axes.add(Z);
		if (dimensions.x == width) {
			X = Vector3.X.cpy();
		} else if (dimensions.y == width) {
			X = Vector3.Y.cpy();
		} else {
			X = Vector3.Z;
		}
		if (dimensions.x == height && !X.epsilonEquals(Vector3.X)) {
			Z = Vector3.X.cpy();
		} else if (dimensions.y == height && !X.epsilonEquals(Vector3.Y)) {
			Z = Vector3.Y.cpy();
		} else {
			Z = Vector3.Z;
		}
		Y = new Vector3(1, 1, 1).sub(X).sub(Z);
		Matrix4 toWolrd = new Matrix4().set(X, Y, Z, center.cpy());
		transform(toWolrd.inv());

	}

	public ArrayList<Box> extract(Box box) {
		toLocalTransform();
		box.transform(this.relativeTransform.cpy().inv().mul(box.getTransform()));

		Vector3 min0 = getCorner000().cpy();
		min0.z = box.getCorner000().z;
		Vector3 max0 = box.getCorner101().cpy();
		max0.x = getCorner100().x;

		Vector3 min1 = box.getCorner010().cpy();
		min1.x = getCorner000().x;
		Vector3 max1 = getCorner111().cpy();
		max1.z = box.getCorner111().z;

		Vector3 min2 = box.getCorner000().cpy();
		min2.x = getCorner000().x;
		Vector3 max2 = box.getCorner011().cpy();

		Vector3 min3 = box.getCorner100().cpy();
		Vector3 max3 = box.getCorner111().cpy();
		max3.x = getCorner111().x;

		Vector3 min4 = getCorner000().cpy();
		Vector3 max4 = getCorner111().cpy();
		max4.z = box.getCorner000().z;

		Vector3 min5 = getCorner000().cpy();
		min5.z = box.getCorner111().z;
		Vector3 max5 = getCorner111().cpy();

		Box box0 = new Box(min0, max0, relativeTransform), box1 = new Box(min1, max1, relativeTransform),
				box2 = new Box(min2, max2, relativeTransform), box3 = new Box(min3, max3, relativeTransform),
				box4 = new Box(min4, max4, relativeTransform), box5 = new Box(min5, max5, relativeTransform);

		ArrayList<Box> boxes = new ArrayList<Box>();

		if (MathUtilities.boxIsValid(box0))
			boxes.add(box0);
		if (MathUtilities.boxIsValid(box1))
			boxes.add(box1);
		if (MathUtilities.boxIsValid(box2))
			boxes.add(box2);
		if (MathUtilities.boxIsValid(box3))
			boxes.add(box3);
		if (MathUtilities.boxIsValid(box4))
			boxes.add(box4);
		if (MathUtilities.boxIsValid(box5))
			boxes.add(box5);
		return boxes;
	}

	public Vector3 getCenter() {
		return center.cpy().mul(relativeTransform);
	}

	public void setCenter(Vector3 center) {
		this.center = center;
	}

	public Vector3 getDimensions() {
		return dimensions;
	}

	public void setDimensions(Vector3 dimensions) {
		this.dimensions = dimensions;
	}

	public Quaternion getRotation() {
		return rotation;
	}

	public void setRotation(Quaternion rotation) {
		this.rotation = rotation;
	}

	public Matrix4 getTransform() {
		return relativeTransform;
	}

	public Vector3 getCorner001() {
		return corner001.cpy();
	}

	public void setCorner001(Vector3 corner001) {
		this.corner001.set(corner001);
	}

	public Vector3 getCorner010() {
		return corner010.cpy();
	}

	public void setCorner010(Vector3 corner010) {
		this.corner010.set(corner010);
	}

	public Vector3 getCorner011() {
		return corner011.cpy();
	}

	public void setCorner011(Vector3 corner011) {
		this.corner011.set(corner011);
	}

	public Vector3 getCorner100() {
		return corner100.cpy();
	}

	public void setCorner100(Vector3 corner100) {
		this.corner100.set(corner100);
	}

	public Vector3 getCorner101() {
		return corner101.cpy();
	}

	public void setCorner101(Vector3 corner101) {
		this.corner101.set(corner101);
	}

	public Vector3 getCorner110() {
		return corner110.cpy();
	}

	public void setCorner110(Vector3 corner110) {
		this.corner110.set(corner110);
	}

	public Vector3 getCorner111() {
		return corner111.cpy();
	}

	public void setCorner111(Vector3 corner111) {
		this.corner111.set(corner111);
	}

	public Vector3 getCorner000() {
		return corner000.cpy();
	}

	public void setCorner000(Vector3 corner) {
		this.corner000.set(corner);
	}
	
	

	public Vector3[] getCorners() {
		return corners;
	}

	@Override
	protected void finalize() throws Throwable {
		super.finalize();

	}

	public void dispose() {

	}

	public static void main(String[] args) {
		Vector3 center = new Vector3(0,0,5);
		Vector3 dims = new Vector3(2,2,2);
		Quaternion q = new Quaternion(Vector3.Y.cpy(), 45);
		Box box = new Box(center, dims);
		box.setRotation(q);
		box.calculateCorners();

		for(Vector3 vector: box.getCorners())
			System.out.println(vector);
		
	}

}
