package gdxapp.ui;

import java.util.ArrayList;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.g2d.GlyphLayout;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.ui.Button;
import com.badlogic.gdx.scenes.scene2d.ui.Image;
import com.badlogic.gdx.scenes.scene2d.ui.Label;
import com.badlogic.gdx.scenes.scene2d.ui.Label.LabelStyle;
import com.badlogic.gdx.scenes.scene2d.ui.ScrollPane;
import com.badlogic.gdx.scenes.scene2d.ui.Skin;
import com.badlogic.gdx.scenes.scene2d.ui.Table;
import com.badlogic.gdx.scenes.scene2d.utils.ClickListener;
import com.badlogic.gdx.utils.Align;
import com.badlogic.gdx.utils.Scaling;

public class ObjectMenuList extends ScrollPane {

	public static final String COPY = "Copier";
	public static final String PASTE = "Coller";
	public static final String CREATE_HABILLAGE = "Ajouter Habillage";
	public static final String DELETE = "Supprimer";
	public static final String FINITION = "Finition Portes";
	public static final String EDIT = "Modifier";
	public static final String FIT_SPACE = "Largeur Auto";
	public static final String AJUSTER_POSITION = "Ajuster Position";
	public static final String AJUSTER_ROTATION = "Ajuster Rotation";

	public static final String[] WALL2D_MENU = { EDIT, DELETE };
	public static final String[] SCENE_OBJECT_MENU = { FINITION, COPY, DELETE };

	private static final float MIN_WIDTH = 220;
	private static final float MAX_HEIGHT = 300;
	private static final float SHORTCUT_PADDING = 40f;
	private static final float ICON_WIDTH = 24f;
	private static final float ICON_PADDING = 2f;
	private static final float INTERNAL_PADDING = 20f;
	private static final float SCROLL_WIDTH = 12;

	private final Table contentTable;

	public ObjectMenuList(Object subject, Skin skin, String styleName, ArrayList<ActionItem> extraActions) {
		super(null, skin);
		contentTable = new Table();
		contentTable.setBackground(skin.newDrawable("white", Color.WHITE));
		contentTable.pad(0);
		contentTable.defaults().fillX().pad(0);
        contentTable.top().left();

		final float buttonWidth = calculateButtonWidth(extraActions, skin);
		final boolean needsScroll = isScrollable(skin, extraActions);
		final float actualButtonWidth = needsScroll ? buttonWidth - SCROLL_WIDTH : buttonWidth;

		for (ActionItem action : extraActions) {
			Button button = createActionButton(skin, styleName, action);
			contentTable.add(button).width(actualButtonWidth).padBottom(2).row();
		}

		contentTable.pack();
		this.setActor(contentTable);

		this.setFadeScrollBars(false);
		this.setScrollingDisabled(true, false);
		this.setScrollbarsOnTop(true);
		this.setOverscroll(false, false);

		float height = Math.min(contentTable.getHeight(), MAX_HEIGHT);
		this.setSize(buttonWidth, height);
	}

	private boolean isScrollable(Skin skin, ArrayList<ActionItem> extraActions) {
		BitmapFont font = skin.get(LabelStyle.class).font;
		float estimatedButtonHeight = font.getLineHeight() + 10f; 
		float estimatedContentHeight = estimatedButtonHeight * extraActions.size();

		boolean needsScroll = estimatedContentHeight > MAX_HEIGHT;
		return needsScroll;
	}
	
	private float maxTitleWidth(ArrayList<ActionItem> extraActions, Skin skin) {
		GlyphLayout layout = new GlyphLayout();
		BitmapFont font = skin.get(LabelStyle.class).font;

		float maxTitleWidth = 0;

		for (ActionItem action : extraActions) {
			layout.setText(font, action.getTitle());
			float titleWidth = layout.width;
			layout.setText(font, action.getShortcut());
			float shortcutWidth = layout.width;

			float totalRowWidth = titleWidth + shortcutWidth;

			maxTitleWidth = Math.max(maxTitleWidth, totalRowWidth);
		}

		return maxTitleWidth;

	}
	
	private float calculateButtonWidth(ArrayList<ActionItem> extraActions, Skin skin) {
		final float maxTitleWidth = maxTitleWidth(extraActions, skin);
		float calculatedWidth = ICON_WIDTH + ICON_PADDING + maxTitleWidth + SHORTCUT_PADDING + INTERNAL_PADDING;
		return Math.max(MIN_WIDTH, calculatedWidth);
	}

	private Button createActionButton(Skin skin, String styleName, ActionItem action) {
		Image iconImage = null;
		if (action.getIcon() != null) {
			iconImage = new Image(action.getIcon());
			iconImage.setScaling(Scaling.fit);
			iconImage.setSize(ICON_WIDTH, ICON_WIDTH);
		}
		Label titleLabel = new Label(action.getTitle(), skin);
		titleLabel.setAlignment(Align.left);
		titleLabel.setEllipsis(true);

		Label shortcutLabel = new Label(action.getShortcut(), skin);
		shortcutLabel.setAlignment(Align.right);

		Table rowTable = new Table();
		rowTable.defaults().padLeft(5).padRight(5);

		if (iconImage != null) {
			rowTable.add(iconImage).size(ICON_WIDTH, ICON_WIDTH).padRight(ICON_PADDING);
		} else {
			rowTable.add().width(ICON_WIDTH + ICON_PADDING);
		}
		rowTable.add(titleLabel).expandX().left();
		rowTable.add(shortcutLabel).width(SHORTCUT_PADDING).right();

		Button button = new Button(skin, styleName);
		if (!action.isEnabled()) {
			button.setDisabled(true);
			button.setColor(1, 1, 1, 0.5f);
		}

		button.add(rowTable).expand().fill();

		if (action.isEnabled()) {
			button.addListener(new ClickListener() {
				@Override
				public void clicked(InputEvent event, float x, float y) {
					action.execute();
					ObjectMenuList.this.remove();
				}
			});
		}
		return button;
	}


}
