package gdxapp.screens.wall;

import java.util.ArrayList;
import java.util.HashMap;

import com.badlogic.gdx.Input;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.InputListener;
import com.badlogic.gdx.scenes.scene2d.Touchable;

import dressing.events.EventDriver;
import dressing.mathutils.EarClipper;
import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Triangle;
import dressing.model.ProjectManager;
import gdxapp.Commun.AbstractScreen;
import gdxapp.assets.DrawingHelper;
import gdxapp.object3d.Object2D;
import gdxapp.object3d.Wall;
import gdxapp.quotation.Quotation;
import gdxapp.scenes.Scene;
import gdxapp.scenes.SceneEvent;
import gdxapp.screens.room.RoomController;
import geometry.Arc;
import geometry.CompoundShape;
import geometry.Polygon;
import geometry.Ray;
import geometry.Shape;

public class Wall2D extends Object2D {

	private CompoundShape geometry;
	private float thickness;
	private ArrayList<Shape> sides;
	float height;
	private ArrayList<Quotation> quotations;
	private ArrayList<Vector2> joints;
	private ArrayList<Ray> bisectors;
	// attributes for 2d rendering
	private Texture texture;
	private ArrayList<Triangle> triangles;
	private ArrayList<float[]> mesh;
	private ArrayList<short[]> indices;

	public Wall2D(CompoundShape compoundShape, float thickness) {
		this.geometry = compoundShape;
		transformVertices();
		this.geometry.calculateProperties();
		this.thickness = thickness;
		calculateFilledContour();
		setWidth(geometry.getWidth());
		setHeight(geometry.getHeight());
		setPosition(geometry.getX(), geometry.getY());
		setZIndex(0);
		eventListener = new Wall2DEventProcessor();
		addListener(eventListener);
	}
	
	public Wall2D(Wall wall) {
		this.worldObject = wall;
		this.geometry = wall.getPerimeter();
		transformVertices();
		this.geometry.calculateProperties();
		calculateFilledContour();
		setWidth(geometry.getWidth());
		setHeight(geometry.getHeight());
		setPosition(geometry.getX(), geometry.getY());
		setZIndex(0);
		eventListener = new Wall2DEventProcessor();
		addListener(eventListener);
	}

	private void transformVertices() {
		if(!geometry.getWorldTransform().equals(RoomController.getInstance().getSceneToWorldTransform())) {
			for(Shape shape: geometry.getNodes()) {
				for(Vector2 vertex: shape.getVertices()) {
					Vector3 v = new Vector3(vertex, 1.0f).mul(geometry.getWorldTransform());
					v.mul(RoomController.getInstance().getWorldToSceneTransform());
					vertex.set(v.x, v.y);
				}
			}
			geometry.setWorldTransform(RoomController.getInstance().getSceneToWorldTransform());
			//updating transform and transforming vertices
		}
	}

	private void calculateFilledContour() {
		if (mesh == null)
			mesh = new ArrayList<>();
		if (indices == null)
			indices = new ArrayList<>();
		mesh.clear();
		indices.clear();
		this.thickness = getWorldObject().getThickness() * RoomController.getInstance().getScaleX();
		
		ArrayList<Shape> nodes = new ArrayList<Shape>();
		nodes.addAll(geometry.getNodes());
		if (geometry.isClosed()) {
			Polygon polygon = new Polygon();
			polygon.getVertices().add(geometry.getNodes().get(nodes.size() - 1).getlastVertex());
			polygon.getVertices().add(geometry.getNodes().get(0).getVertices().get(0));
			nodes.add(polygon);
		}
		for (Shape node : nodes) {
			ArrayList<Vector2> outerContour;
			if (node instanceof Polygon) {
				Polygon poly = (Polygon) node;
				float[] thicknesses = new float[poly.getEdges().size()];
				for (int i = 0; i < poly.getEdges().size(); i++) {
					Edge edge = poly.getEdges().get(i);
					WallFragment fragment = ProjectManager.getManager().getCurrentScene().getFragments().get(edge);
					if (fragment == null) {
						thicknesses[i] = thickness;
					} else {
						thicknesses[i] = fragment.getThickness() * RoomController.getInstance().getScaleX();
					}
				}
				outerContour = MathUtilities.calculateSurfaceContour(poly.getVertices(), poly.getEdges(), thicknesses);
			} else {
				outerContour = MathUtilities.calculateSurfaceContour(node.getVertices(), false, thickness);
			}

			ArrayList<Vector2> innerContour = new ArrayList<Vector2>();
			int size = node.getVertices().size();
			for (int i = 0; i < size; i++) {
				innerContour.add(node.getVertices().get(size - 1 - i));
			}
			outerContour.addAll(innerContour);
			innerContour.clear();
			MathUtilities.setWinding(outerContour, 1);
			MathUtilities.setWinding(innerContour, -1);
			ArrayList<Triangle> triangles = EarClipper.triangulate(outerContour, innerContour);
			if(this.triangles == null) {
				this.triangles = new ArrayList<Triangle>();
			}else {
				this.triangles.clear();
			}
			this.triangles.addAll(triangles);
			short[] partIndices = new short[triangles.size() * 3];
			float[] meshPart = MathUtilities.indexMesh(triangles, partIndices, new float[] { 0.0f, 1.0f, 0.0f, 1.0f }, Color.WHITE_FLOAT_BITS);
			mesh.add(meshPart);
			indices.add(partIndices);
		}
	}
	
	
	

	@Override
	public Actor hit(float x, float y, boolean touchable) {
		if (touchable && this.getTouchable() != Touchable.enabled) return null;
		if (!isVisible()) return null;
		Actor hit = null;
		Vector2 point = new Vector2(x,y).add(getX(), getY());
		for(Triangle triangle: triangles) {
			if(triangle.contains(point)) {
				hit = this;
				break;
			}
		}
		return hit;
	}

	@Override
	public void draw(Batch batch, float parentAlpha) {
		
		batch.end();
		DrawingHelper.getDebugRenderer().begin(ShapeType.Line);
		DrawingHelper.getDebugRenderer().end();
		batch.begin();
		if (texture == null) {
			texture = new Texture(DrawingHelper.createHachedPixmap());
		}

		if (mesh != null ) {
			batch.end();
			RoomController.getPolyBatch().setProjectionMatrix(batch.getProjectionMatrix().cpy());
			RoomController.getPolyBatch().begin();
			for (int i = 0; i < mesh.size(); i++) {
				RoomController.getPolyBatch().draw(texture, mesh.get(i), 0, mesh.get(i).length, indices.get(i), 0,
						indices.get(i).length);
			}
			RoomController.getPolyBatch().end();
			if (bisectors == null) {
				calculateBisectors();
			}
			calculateBisectors();
			if (bisectors != null) {
				float length = RoomController.getInstance().getScaleX();
				for (Ray ray : bisectors) {
					ray.draw(DrawingHelper.getDebugRenderer(), length);
				}
			}
			batch.begin();
		}
		((Wall2DEventProcessor) eventListener).draw(batch);
	}

	private void calculateSides() {
		if (sides == null) {
			sides = new ArrayList<Shape>();
		} else {
			sides.clear();
		}
		for (Shape shape : geometry.getNodes()) {
			if (shape instanceof Polygon) {
				((Polygon) shape).calculateEdges();
				sides.addAll(((Polygon) shape).getEdges());
			}
			if (shape instanceof Arc) {
				sides.add(shape);
			}
		}
		if (geometry.isClosed()) {
			sides.add(new Edge(geometry.getVertices().get(geometry.getVertices().size() - 1),
					geometry.getVertices().get(0)));
		}
	}


	@Override
	public ArrayList<Vector2> calculateAttachmentPosition() {
		ArrayList<Vector2> criticalPoints = new ArrayList<Vector2>();
		for (Shape node : this.geometry.getNodes()) {
			if (node instanceof Polygon) {
				criticalPoints.addAll(node.getVertices());
			} else if (node instanceof Arc) {
				criticalPoints.add(((Arc) node).getV0());
				criticalPoints.add(((Arc) node).getV1());
			}
		}
		return criticalPoints;
	}

	public CompoundShape getGeometry() {
		return geometry;
	}

	public void setGeometry(CompoundShape geometry) {
		this.geometry = geometry;
		geometryChanged();
	}

	public ArrayList<Shape> getSides() {
		if (sides == null)
			calculateSides();
		return sides;
	}

	public void setEdges(ArrayList<Shape> edges) {
		this.sides = edges;
	}

	public ArrayList<Quotation> getQuotations() {
		calculateSides();
		ArrayList<Shape> edges = getSides();
		this.quotations = new ArrayList<Quotation>(edges.size());
		for (Shape e : edges) {
			if (e instanceof Edge) {
				Edge edge = (Edge) e;
				Quotation quotation = new Quotation(edge,
						(int) ((edge.getLength() * 1000) / RoomController.getInstance().getScaleX()) + "");
				quotation.setDistance(35);
				quotations.add(quotation);
			}
		}
		return quotations;
	}

	public void calculateBisectors() {
		if (this.bisectors == null) {
			this.bisectors = new ArrayList<Ray>();
		} else {
			this.bisectors.clear();
		}
		for (Shape shape : geometry.getNodes(Polygon.class)) {
			Polygon polygon = (Polygon) shape;
			ArrayList<Ray> bisectors = polygon.getBisectors();
			if (bisectors != null)
				this.bisectors.addAll(bisectors);
		}
	}

	public void geometryChanged() {
		geometry.calculateProperties();
		calculateFilledContour();
		calculateSides();
		calculateBisectors();
		setWidth(geometry.getWidth());
		setHeight(geometry.getHeight());
		setPosition(geometry.getX(), geometry.getY());
		getWorldObject().setRequireRefrech(true);

	}







	public class Wall2DEventProcessor extends InputListener {

		private Vector2 mouseLocation;
		long lastClickDate;
		private Shape selectedShape;

		@Override
		public boolean touchDown(InputEvent event, float x, float y, int pointer, int button) {
			long currentClickDate = System.currentTimeMillis();
			if(button == Input.Buttons.RIGHT) {
				HashMap<String, Object> eventData = new HashMap<>();
				eventData.put("location", new Vector2(Wall2D.this.getX() + x, Wall2D.this.getY() + y));
				eventData.put("source", Wall2D.this);
				dressing.events.Event sceneEvent = new dressing.events.Event(SceneEvent.OBJECT_RIGHT_CLICK.name(), eventData);
				EventDriver.getDriver().deliverEvent(sceneEvent);
			}
			if (currentClickDate - lastClickDate < 300) {
				RoomController.getInstance().editWall(Wall2D.this);
				return false;
			}
			Vector2 location = new Vector2(x, y).add(new Vector2(getX(), getY()));
			selectEdge(location, 20);
			lastClickDate = currentClickDate;
			return true;
		}

		@Override
		public boolean mouseMoved(InputEvent event, float x, float y) {
			Vector2 point = new Vector2(x, y).add(getX(), getY());
			mouseLocation = point;
			selectEdge(mouseLocation, 140);
			return true;
		}

		@Override
		public boolean keyDown(InputEvent event, int keycode) {
			return true;
		}

		public void draw(Batch batch) {

			if (selectedShape != null) {
				Vector2 project = selectedShape.projectPoint(mouseLocation);
				float scale = ((AbstractScreen) Scene.game.getScreen()).getController().getScales()[0];
				int distance = Math.round(mouseLocation.dst(project) * 1000 / scale);
				DrawingHelper.drawCutLine(mouseLocation, project, batch, Color.BLUE, 2);
				DrawingHelper.drawText("" + distance, new Vector3(mouseLocation, 0), 0, Color.BLACK, .5f, batch);
			}
		}

		public void selectEdge(Vector2 point, float maxDistance) {
			if (sides == null) {
				calculateSides();
			}
			Shape closest = null;
			float closestDistance = 10000000;
			for (Shape shape : sides) {
				float distance = shape.distanceTo(point);
				if (distance < maxDistance) {
					if (closest == null) {
						closest = shape;
						closestDistance = distance;
						continue;
					} else {
						if (distance < closestDistance) {
							closest = shape;
							closestDistance = distance;
						}
					}
				}
			}
			selectedShape = closest;
		}

	}

	@Override
	protected void sizeChanged() {
		
	}

	@Override
	protected void positionChanged() {
		
	}

	@Override
	public Wall getWorldObject() {
		return  (Wall) worldObject;
	}

	@Override
	public void adjustTextureAndColors() {

	}
	
	
	
	
	
	

}
