package gdxapp.shaders;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.Pixmap;
import com.badlogic.gdx.graphics.VertexAttribute;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.utils.Disposable;
import com.badlogic.gdx.utils.GdxRuntimeException;

import dressing.config.WorkspaceConfiguration;

public class EnvironmentCubemap implements Disposable{
	
	protected final Pixmap[] data = new Pixmap[6];
    protected ShaderProgram shader;
    protected int vp;
    protected Mesh quad;
    int cubemap;

    String vertexShader = Gdx.files.absolute(WorkspaceConfiguration.SHADERS_PATH + "/skybox/vertx.glsl").readString();
    String fragmentShader = Gdx.files.absolute(WorkspaceConfiguration.SHADERS_PATH + "/skybox/frag.glsl").readString();

    public String getDefaultVertexShader(){
        return vertexShader;
    }

    public String getDefaultFragmentShader(){
        return fragmentShader;
    }
    
    public EnvironmentCubemap() {
//    	loadCubemaps();
    	init();
    }

    public EnvironmentCubemap(Pixmap positiveX, Pixmap negativeX, Pixmap positiveY, Pixmap negativeY, Pixmap positiveZ, Pixmap negativeZ) {
        data[0]=positiveX;
        data[1]=negativeX;

        data[2]=positiveY;
        data[3]=negativeY;

        data[4]=positiveZ;
        data[5]=negativeZ;

        init();
    }

    private void init(){
        shader = new ShaderProgram(vertexShader, fragmentShader);
        if (!shader.isCompiled())
            throw new GdxRuntimeException(shader.getLog());

        vp = shader.getUniformLocation("vp");;

        quad = createBox();

        initCubemap();
    }

    private void initCubemap(){
    	
    	cubemap = Gdx.gl20.glGenTexture();
        //bind cubemap
        Gdx.gl20.glBindTexture(GL20.GL_TEXTURE_CUBE_MAP, cubemap);
        Gdx.gl20.glTexImage2D(GL20.GL_TEXTURE_CUBE_MAP_POSITIVE_X, 0, GL20.GL_RGB, data[0].getWidth(), data[0].getHeight(), 0, GL20.GL_RGB, GL20.GL_UNSIGNED_BYTE, data[0].getPixels());
        Gdx.gl20.glTexImage2D(GL20.GL_TEXTURE_CUBE_MAP_NEGATIVE_X, 0, GL20.GL_RGB, data[1].getWidth(), data[1].getHeight(), 0, GL20.GL_RGB, GL20.GL_UNSIGNED_BYTE, data[1].getPixels());

        Gdx.gl20.glTexImage2D(GL20.GL_TEXTURE_CUBE_MAP_POSITIVE_Y, 0, GL20.GL_RGB, data[2].getWidth(), data[2].getHeight(), 0, GL20.GL_RGB, GL20.GL_UNSIGNED_BYTE, data[2].getPixels());
        Gdx.gl20.glTexImage2D(GL20.GL_TEXTURE_CUBE_MAP_NEGATIVE_Y, 0, GL20.GL_RGB, data[3].getWidth(), data[3].getHeight(), 0, GL20.GL_RGB, GL20.GL_UNSIGNED_BYTE, data[3].getPixels());
        Gdx.gl20.glTexImage2D(GL20.GL_TEXTURE_CUBE_MAP_POSITIVE_Z, 0, GL20.GL_RGB, data[4].getWidth(), data[4].getHeight(), 0, GL20.GL_RGB, GL20.GL_UNSIGNED_BYTE, data[4].getPixels());
        Gdx.gl20.glTexImage2D(GL20.GL_TEXTURE_CUBE_MAP_NEGATIVE_Z, 0, GL20.GL_RGB, data[5].getWidth(), data[5].getHeight(), 0, GL20.GL_RGB, GL20.GL_UNSIGNED_BYTE, data[5].getPixels());

        Gdx.gl20.glTexParameteri ( GL20.GL_TEXTURE_CUBE_MAP, GL20.GL_TEXTURE_MIN_FILTER,GL20.GL_LINEAR_MIPMAP_LINEAR );
        Gdx.gl20.glTexParameteri ( GL20.GL_TEXTURE_CUBE_MAP, GL20.GL_TEXTURE_MAG_FILTER,GL20.GL_LINEAR );
        Gdx.gl20.glTexParameteri ( GL20.GL_TEXTURE_CUBE_MAP, GL20.GL_TEXTURE_WRAP_S, GL20.GL_CLAMP_TO_EDGE );
        Gdx.gl20.glTexParameteri ( GL20.GL_TEXTURE_CUBE_MAP, GL20.GL_TEXTURE_WRAP_T, GL20.GL_CLAMP_TO_EDGE );

        Gdx.gl20.glGenerateMipmap(GL20.GL_TEXTURE_CUBE_MAP);
    }



    public void render(Camera camera){
        Quaternion q = new Quaternion();
        camera.view.getRotation(q);
        Matrix4 v = new Matrix4().idt().set(q);
        Matrix4 vpMat = camera.projection.cpy().mul(v);
        
        shader.begin();
        Gdx.gl20.glBindTexture(GL20.GL_TEXTURE_CUBE_MAP, cubemap);
        shader.setUniformMatrix(vp, vpMat);
        quad.render(shader, GL20.GL_TRIANGLES);
        shader.end();

    }

    public Mesh createBox(){
        Mesh mesh = new Mesh(true, 8, 36, VertexAttribute.Position());
        mesh.setVertices(new float[]{
                                -0.5f, -0.5f, -0.5f,
                                -0.5f, -0.5f, 5f,
                                -0.5f, .5f, -.5f,
                                -.5f, .5f, .5f,
                                .5f, -.5f, -.5f,
                                .5f, -.5f, .5f,
                                .5f, .5f, -.5f,
                                .5f, .5f, .5f,
                        });
        mesh.setIndices(new short[] {
                                2, 0, 4, 4, 6, 2,
                                1, 0, 2, 2, 3, 1,
                                4, 5, 7, 7, 6, 4,
                                1, 3, 7, 7, 5, 1,
                                2, 6, 7, 7, 3, 2,
                                1, 0, 4, 4, 5, 1
                                    });
        return mesh;
    }

    @Override
    public void dispose() {
        shader.dispose();
        quad.dispose();
        for(int i=0; i<6; i++)
            data[i].dispose();
    }
    


}
