package supercad.p2.update;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.e4.ui.workbench.IWorkbench;
import org.eclipse.equinox.p2.core.IProvisioningAgent;
import org.eclipse.equinox.p2.operations.ProvisioningJob;
import org.eclipse.equinox.p2.operations.ProvisioningSession;
import org.eclipse.equinox.p2.operations.UpdateOperation;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.frs.supercad.config.ConfigurationManager;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;

public class UpdateHandler {

	private String baseUrl;
	private String repo_loc;
	private IWorkbench workbench;
	private List<LogListner> logListners = new ArrayList();
	private ConfigurationManager configManager;
	
	public UpdateHandler() {
	    BundleContext ctx = FrameworkUtil.getBundle(getClass()).getBundleContext();
	    ServiceReference<ConfigurationManager> ref = ctx.getServiceReference(ConfigurationManager.class);
	    this.configManager = ctx.getService(ref);
	}
    
	public void setBuildNumber(String lastbuildNumber) {
		baseUrl = this.configManager.getProperty("URL_P2_SITE_UPDATE");
		if (baseUrl == null || baseUrl.isEmpty()) {
			throw new IllegalStateException(
					"La propriété 'URL_P2_SITE_UPDATE' est manquante dans le fichier de configuration.");
		}
		repo_loc = baseUrl + lastbuildNumber + "/";
		notifyLog("URL de mise à jour configurée : " + repo_loc);
	}

	public void execute(final IProvisioningAgent agent, IWorkbench workbench) {
	    if (agent == null) {
	        notifyLog("Provisioning Agent is null - cannot perform update");
	        return;
	    }
	    this.workbench = workbench;
	    notifyLog("Début du processus de mise à jour");

	    Job updateJob = Job.create("Recherche de mises à jour", defaultmonitor -> {
	        try {
	            Boolean userConfirmed = Display.getDefault().syncCall(() -> 
	                MessageDialog.openQuestion(null, "Mise à jour disponible", "Une mise à jour est disponible. Voulez-vous l'installer ?")
	            );

	            // Wrap the original monitor to log activity
		        LoggingProgressMonitor monitor = new LoggingProgressMonitor(defaultmonitor);
		        for(LogListner lg : logListners) {
		        	monitor.addLogListner(lg);
		        }
		        
	            if (userConfirmed != null && userConfirmed) {
	                // ➔ Après confirmation : afficher "Téléchargement en cours..."
//	                Display.getDefault().syncExec(() -> {
//	                    loadingDialog = new MessageDialog(
//	                        Display.getDefault().getActiveShell(),
//	                        "Mise à jour", 
//	                        null,
//	                        "Téléchargement de la mise à jour en cours...\nVeuillez patienter.",
//	                        MessageDialog.INFORMATION,
//	                        new String[] {}, // pas de bouton pour forcer l'attente
//	                        0
//	                    );
//	                    loadingDialog.create();
//	                    loadingDialog.getShell().setEnabled(false); 
//	                    loadingDialog.open();
//	                    
//	                });
	            	notifyLog("Téléchargement de la mise à jour en cours...\nVeuillez patienter.");
	                IStatus status = performUpdates(agent, monitor);

	                notifyLog("Update process completed with status: " + status);

//	                Display.getDefault().asyncExec(() -> {
//	                    if (loadingDialog != null && loadingDialog.getShell() != null && !loadingDialog.getShell().isDisposed()) {
//	                        loadingDialog.close();
//	                    }
//	                });

	                if (!status.isOK() && status.getSeverity() != IStatus.CANCEL) {
	                    String errorDetails = getStatusDetails(status);
	                    notifyLog("Update failed: " + errorDetails);

	                    Display.getDefault().asyncExec(() ->
	                        MessageDialog.openError(null, "Erreur", "Échec de la mise à jour: " + errorDetails)
	                    );
	                }
	            } else {
	            	notifyLog("Mise à jour annulée par l'utilisateur.");
	            }
	        } catch (Exception e) {
	            System.err.println("Exception during update: " + e.getMessage());
	            e.printStackTrace();
	        }
	    });
	    updateJob.schedule();
	}


	private IStatus performUpdates(final IProvisioningAgent agent, IProgressMonitor monitor) {
		notifyLog("Creating provisioning session");
	    final ProvisioningSession session = new ProvisioningSession(agent);
	    final UpdateOperation operation = new UpdateOperation(session);

	    try {
	        URI uri = new URI(repo_loc);
	        notifyLog("Setting repository: " + uri);
	        operation.getProvisioningContext().setArtifactRepositories(uri);
	        operation.getProvisioningContext().setMetadataRepositories(uri);

	        notifyLog("Resolving updates...");
           
	        final IStatus status = operation.resolveModal(monitor);
	        logStatus("Resolution status", status);

	        if (status.getCode() == UpdateOperation.STATUS_NOTHING_TO_UPDATE) {
	            Display.getDefault().asyncExec(
	                () -> MessageDialog.openInformation(null, "Mise à jour", "Aucune mise à jour disponible.")
	            );
	            return Status.CANCEL_STATUS;
	        }

	        if (!status.isOK()) {
	            return status; // Return the error status
	        }

	        // ➔ SUPPRIMÉ : demande de confirmation

	        ProvisioningJob provisioningJob = operation.getProvisioningJob(monitor);
	        if (provisioningJob == null) {
	        	notifyLog("ProvisioningJob is null - cannot proceed with update");
	            Display.getDefault().asyncExec(() -> {
	                MessageDialog.openWarning(Display.getDefault().getActiveShell(), "Update Error",
	                    "Could not create installation job.\n" + "Resolution result: "
	                    + operation.getResolutionResult());
	            });
	            return new Status(IStatus.ERROR, "supercad.p2.update", "ProvisioningJob is null");
	        }
	        notifyLog("Configuring provisioning job !");
	        configureProvisioningJob(provisioningJob);
	        provisioningJob.schedule();
	        return Status.OK_STATUS;

	    } catch (URISyntaxException e) {
	    	notifyLog("Invalid repository URI: " + e.getMessage());
	        return new Status(IStatus.ERROR, "supercad.p2.update", "Invalid repository URI", e);
	    }
	    catch (Exception e) {
	    	notifyLog("Erreur survenue: " + e.getMessage());
	        return new Status(IStatus.ERROR, "supercad.p2.update", "Erreur survenue", e);
	    }
	}


	private void configureProvisioningJob(ProvisioningJob provisioningJob) {

		provisioningJob.addJobChangeListener(new JobChangeAdapter() {
			@Override
			public void done(IJobChangeEvent event) {
				notifyLog("Provisioning job completed with IStatusCodes : " + event.getResult().getCode());
				notifyLog("Provisioning job completed with Message : " + event.getResult().getMessage());
				notifyLog("Provisioning job completed with Plugin : " + event.getResult().getPlugin());

				if (event.getResult().getException() != null)
					notifyLog("Provisioning job completed with Plugin : "
							+ event.getResult().getException().getMessage());

				if (event.getResult().getSeverity() == IStatus.CANCEL) {
					notifyLog(
							"Provisioning job was cancelled. Possible cause: identical version already installed.");
				}
				if (event.getResult().isOK()) {
					notifyLog("Update installed successfully");	
					notifyLog("Mise à jour terminée avec succès, redémarrage de l'application...");
					Display.getDefault().asyncExec(() -> {
					    // Créer un MessageDialog personnalisé
					    MessageDialog dialog = new MessageDialog(
					            Display.getDefault().getActiveShell(), // Utiliser la fenêtre active
					            "Redémarrage nécessaire",               // Titre
					            null,                                   // Pas d'icône personnalisée
					            "La mise à jour a été installée avec succès.\n"
					            + "L'application doit redémarrer pour appliquer les changements.\n\n"
					            + "Redémarrer maintenant ?",            // Message
					            MessageDialog.QUESTION,                 // Type d'icône
					            new String[] { "Oui", "Non" },           // Boutons en français
					            0                                        // Bouton par défaut ("Oui")
					    );

					    int result = dialog.open();

					    if (result == 0) { // 0 correspond à "Oui"
					        workbench.restart();
					    }
					});
				} else {
					String errorDetails = getStatusDetails(event.getResult());
					notifyLog("Update installation failed: " + errorDetails);

					Display.getDefault().asyncExec(() -> MessageDialog.openError(null, "Erreur",
							"Échec de l'installation de la mise à jour:\n" + event.getResult().getMessage()));
				}
			}
		});
	}
	
	private void logStatus(String prefix, IStatus status) {
		notifyLog(prefix + ": " + status);
		if (status.isMultiStatus()) {
			for (IStatus child : status.getChildren()) {
				System.out.println("  - " + child);
			}
		}
	}

	private String getStatusDetails(IStatus status) {
		StringBuilder sb = new StringBuilder();
		sb.append(status.getMessage());

		if (status.getException() != null) {
			sb.append("\nException: ").append(status.getException().getMessage());
		}

		if (status.isMultiStatus()) {
			for (IStatus child : status.getChildren()) {
				sb.append("\n  - ").append(child.getMessage());
				if (child.getException() != null) {
					sb.append(" [").append(child.getException().getMessage()).append("]");
				}
			}
		}

		return sb.toString();
	}

	public void addLogListner(LogListner loglistner) {
		// TODO Auto-generated method stub
		if(logListners!=null) {
			logListners.add(loglistner);
		}
	}

	private void notifyLog(String msg) {
        System.err.println(msg);
		for(LogListner lisnet : logListners) {
			lisnet.updateLog(msg);
		}
	}
}
