package gdxapp.object3d;

import java.util.List;

import org.frs.rule_engin.FactsRealm;
import org.frs.rule_engin.NaiveRuleExecutor;
import org.frs.rule_engin.Rule;

import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector3;

import dressing.model.DesignObject3D;
import dressing.model.Piece2D;
import dressing.model.types.PieceType;
import dressing.model.types.PoigneeType;

public class DefaultHandlePositioner {
	
	private FactsRealm facts;
	private NaiveRuleExecutor ruleExecutor;
	private Rule existenceRule;
	private Rule orientationRule;
	private Rule singleDoorRule;
	private Rule heightRule;
	private Rule sideRule;
	private Rule centeredRule;
	private Rule shapeLRule;
	
	private static DefaultHandlePositioner positioner;
	
	private DefaultHandlePositioner() {
		init();
	}
		
	public void init() {			
		createExistenceRule();
		createOrientationRule();
		createSingleDoorRule();
		createHeightRule();
		createSideRule();
		createCenteredRule();
		createshapeLRule();
		ruleExecutor = new NaiveRuleExecutor();
		ruleExecutor.addRules(existenceRule, orientationRule, singleDoorRule, heightRule, sideRule, centeredRule, shapeLRule);
	}
	
	public void bind(Piece2D facade, KitchenElement parent) {
		//check the orientation of the piece
		var size = facade.getSize().scl(0.001f);
		var center = facade.getCenter().scl(0.001f);
		Matrix4 pieceTransform = new Matrix4().setToTranslation(center).inv();
		if(size.x < size.z) {
			pieceTransform.set(new Vector3(0,0,-1), Vector3.Y, new Vector3(1,0,0), center.cpy()).inv();
			size.mul(pieceTransform).sub(new Vector3().mul(pieceTransform));
			size.x = Math.abs(size.x);
			size.y = Math.abs(size.y);
			size.z = Math.abs(size.z);
		}
		
		this.facts = new FactsRealm();
		facts.addFact("facade", facade);
		facts.addFact("transform", pieceTransform);
		facts.addFact("element", parent);
		facts.addFact("handler_type", parent.getDesignObject().getPoigneeType());
		facts.addFact("size", size);
		Float rotation = 0.0f;
		facts.addFact("rotation", rotation);
	}
	
	public void calculateRelativeProperties(KitchenElement element) {
		var facades = element.getDesignObject().getListPieces().stream().filter( piece -> piece.getPiecetype().isFacade()).toList();
		for(var facade: facades) {
			bind(facade, element);
			ruleExecutor.execute(facts);
			int[] position = facts.getFact("position", int[].class);
			float rotation = facts.getFact("rotation", Float.class);
			DoorHandle handle = element.getDoorHandle(facade.getName());
			if(position != null) {
				handle.setRelativePosition(position[0], position[1]);
			}
			handle.setRotation((int) rotation);
		}
		
	}
	
	public Matrix4 calculateDefaultHandlerTransform(Piece2D facade, KitchenElement element) {
		bind(facade, element);
		ruleExecutor.execute(facts);
		int[] position = facts.getFact("position", int[].class);
		Matrix4 transform = null;
		if(position != null) {
			float rotation = facts.getFact("rotation", Float.class);
			DoorHandle model = element.getDoorHandle(facade.getName());
			Vector3 modelHalfSize =  model.getDimension().cpy().scl(0.5f).rotate(Vector3.Z, rotation);
			modelHalfSize.x = Math.abs(modelHalfSize.x);
			modelHalfSize.y = Math.abs(modelHalfSize.y);
			modelHalfSize.z = Math.abs(modelHalfSize.z);
			float padding = 0.02f;
			modelHalfSize.add(padding, padding, 0);
			Vector3 facadeHalfSize =  facts.getFact("size", Vector3.class).scl(0.5f);
			Vector3 facadeCenter = new Vector3();// facade.getCenter().scl(0.001f);
			Vector3 translation = facadeCenter.add(facadeHalfSize.scl(position[0], position[1], 1))
					.sub(modelHalfSize.scl(position[0], position[1], 0));
			Quaternion q = new Quaternion(Vector3.Z, rotation);
			transform = new Matrix4(translation, q, new Vector3(1, 1, 1));
			var pieceTransform = facts.getFact("transform", Matrix4.class);
			transform.mulLeft(pieceTransform.inv());
		}
		return transform;
	}
	
	public Matrix4 calculateHandlerTransform(Piece2D facade, KitchenElement element) {
		DoorHandle handle = element.getDoorHandle(facade.getName());
		bind(facade, element);
		ruleExecutor.execute(facts);
		Matrix4 transform = null;
		if(facts.getFact("position", int[].class) == null)
			return null;
		int[] position = handle.getRelativePosition();
		float rotation = handle.getRotation();
	
		Vector3 modelHalfSize =  handle.getDimension().cpy().scl(0.5f).rotate(Vector3.Z, rotation);
		modelHalfSize.x = Math.abs(modelHalfSize.x);
		modelHalfSize.y = Math.abs(modelHalfSize.y);
		modelHalfSize.z = Math.abs(modelHalfSize.z);
		float padding = 0.02f;
		modelHalfSize.add(padding, padding, 0);
		Vector3 facadeHalfSize =  facts.getFact("size", Vector3.class).scl(0.5f);
		Vector3 facadeCenter = new Vector3();// facade.getCenter().scl(0.001f);
		Vector3 translation = facadeCenter.add(facadeHalfSize.scl(position[0], position[1], 1))
				.sub(modelHalfSize.scl(position[0], position[1], 0));
		Quaternion q = new Quaternion(Vector3.Z, rotation);
		transform = new Matrix4(translation, q, new Vector3(1, 1, 1));
		var pieceTransform = facts.getFact("transform", Matrix4.class);
		transform.mulLeft(pieceTransform.inv());
	
		return transform;
	}
	
	
 	public void createExistenceRule() {
		existenceRule =  new Rule() {
				@Override
				public void execute(FactsRealm realm) {
					int[] position = {0,0};
					realm.addFact("position", position);
				}
				@Override
				public boolean appliedWhen(FactsRealm realm) {
					boolean hasKnobe = facts.getFact("handler_type").equals(PoigneeType.AVEC_POIGNEE);
					boolean isShapeL  = realm.getFact("element", KitchenElement.class).getDesignObject().getDesignCaissonType().contains("COINS_L");
					return hasKnobe && (!isShapeL || shapeLRule.appliedWhen(realm));
				}
			};
	}

	public void createOrientationRule() {
		orientationRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				Vector3 size = facts.getFact("size", Vector3.class);
				float rot = size.x> size.y? 0.0f:90.0f;
				facts.addFact("rotation", rot);
			}
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				return existenceRule.appliedWhen(realm);
			}
		};
	}

	public void createSingleDoorRule() {
		singleDoorRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				facts.addFact("rotation", 0.0f);
			    facts.getFact("position", int[].class)[0] = 0;
			}
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				if(!existenceRule.appliedWhen(realm))
					return false;
				DesignObject3D design = facts.getFact("element",KitchenElement.class).getDesignObject();
				Piece2D facade = (Piece2D) facts.getFact("facade");
				List<Piece2D> facades =  design.getListPieces().stream().filter((piece) -> {
					return piece != facade && piece.getPiecetype().isFacade();
					}).toList();
				float epsilon = 0.001f;
				for(var otherFacade: facades) {
					if(Math.abs(otherFacade.getPosition().y - facade.getPosition().y) < epsilon &&
							otherFacade.getSize().epsilonEquals(facade.getSize(), epsilon) );
					return false;
				}
				return true;
			}
		};
	}

	public void createHeightRule() {
		heightRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				Piece2D facade = facts.getFact("facade", Piece2D.class);
				KitchenElement element = facts.getFact("element", KitchenElement.class);
				Vector3 offset = element.getRealWorldPosition().cpy().sub(element.getRealWorldDimension().xyz().scl(0.5f)).scl(1000);
				if(offset.y + facade.getPosition().y > 800) {
					facts.getFact("position", int[].class)[1] = -1;
				}else {
					facts.getFact("position", int[].class)[1] = 1;
				}
			}
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				return existenceRule.appliedWhen(realm);
			}
		};
	}

	public void createSideRule() {
		sideRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				Piece2D facade = realm.getFact("facade", Piece2D.class);
				DesignObject3D design = realm.getFact("element",KitchenElement.class).getDesignObject();
				
				if(facade.getPosition().x < design.getLongeurext()/2.0f) {
					realm.getFact("position", int[].class)[0] = 1;
				}else {
					realm.getFact("position", int[].class)[0] = -1;
				}
			}
			
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				return existenceRule.appliedWhen(realm);
			}
		};
	}
	
	public void createCenteredRule() {
		centeredRule = new Rule() {
			@Override
			public void execute(FactsRealm realm) {
				realm.getFact("position", int[].class)[0] = 0;
			}
			
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				return singleDoorRule.appliedWhen(realm) && Math.abs(facts.getFact("rotation", Float.class)) < 0.01f;
			}
		};
	}
	
	public void createshapeLRule() {
		shapeLRule = new Rule() {
			
			@Override
			public void execute(FactsRealm realm) {
				var facade = realm.getFact("facade", Piece2D.class);
				if(facade.getPiecetype() == PieceType.PORTE_DROITE) { // coins gauche
					realm.addFact("rotation", 90.0f);
				}else {
					facts.getFact("position", int[].class)[0] = -1;
				}
			}
			
			@Override
			public boolean appliedWhen(FactsRealm realm) {
				var parent = realm.getFact("element", KitchenElement.class).getDesignObject();
				var facade = realm.getFact("facade", Piece2D.class);
				var name = parent.getDesignCaissonType();
				if(name.contains("COINS_L")) {
					if( parent.getName().contains("Gauche") && facade.getPiecetype() == PieceType.PORTE_DROITE)
						return true;
					if( parent.getName().contains("Droite") && facade.getPiecetype() == PieceType.PORTE_GAUCHE)
						return true;
				}
				return false;
			}
		};
	}

	public static DefaultHandlePositioner getPositioner() {
		if(positioner == null)
			positioner = new DefaultHandlePositioner();
		return positioner;
	}
	
	
	

}
