package dressing.ui.util;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.widgets.Display;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import javax.imageio.ImageIO;

public class ImageLoaderCache {
    private final Map<String, Image> imageCache;
    private final Device device;
    public static ImageLoaderCache instance;
    
    public static ImageLoaderCache getInstance() {
    	synchronized(ImageLoaderCache.class) {
			if (instance == null) {
				instance = new ImageLoaderCache(Display.getCurrent());
			}
			return instance;
		}
    }
    public ImageLoaderCache(Device device) {
        this.device = device;
        this.imageCache = new HashMap<>();
    }
    
	public Image loadImage(String imagePath, int desiredWidth, int desiredHeight) {
		return loadImage(imagePath, desiredWidth, desiredHeight, true);
	}
	
	public Image loadImage(String imagePath, int desiredWidth, int desiredHeight, boolean cache) {

		// Check if a resized image already exists on disk
		String resizedImagePath = getResizedImagePath(imagePath, desiredWidth, desiredHeight);

		Image resizedImage = imageCache.get(resizedImagePath);
		if (resizedImage != null && !resizedImage.isDisposed()) {
			return resizedImage;
		}

		File resizedImageFile = new File(resizedImagePath);
		if (resizedImageFile.exists()) {
			resizedImage = new Image(device, resizedImagePath);
			if (cache) {
				imageCache.put(resizedImagePath, resizedImage);
			}
			return resizedImage;
		}

		// Load the original image
		Image image = imageCache.get(imagePath);
		if (image == null || image.isDisposed()) {
			image = new Image(device, imagePath);
			if (cache) {
				imageCache.put(imagePath, image);
			}

		}

		// Resize the image to the desired dimensions
		resizedImage = resizeImage(image, desiredWidth, desiredHeight);

		// Save the resized image to disk
		saveImageToFile(resizedImage, resizedImagePath);
		if (cache) {
			imageCache.put(resizedImagePath, resizedImage);
		}

		// Use the resized image
		return resizedImage;
	}

    public Image resizeImage(Image originalImage, int desiredWidth, int desiredHeight) {
    	if(originalImage == null )return null;
        ImageLoader loader = new ImageLoader();
        loader.data = new ImageData[]{originalImage.getImageData()};
        loader.logicalScreenWidth = desiredWidth;
        loader.logicalScreenHeight = desiredHeight;

        ImageData resizedImageData = originalImage.getImageData().scaledTo(desiredWidth, desiredHeight);
        originalImage.dispose();
        return new Image(device, resizedImageData);
    }

    public void saveImageToFile(Image image, String filePath) {
        ImageLoader loader = new ImageLoader();
        loader.data = new ImageData[]{image.getImageData()};
        loader.save(filePath, getImageFormat(filePath));
    }

    public String getResizedImagePath(String imagePath, int width, int height) {
    	File image=new File(imagePath);

        String extension = imagePath.substring(imagePath.lastIndexOf('.'));
        
    	String parentpath=image.getParentFile().getAbsolutePath();
    	String name =image.getName();
    	parentpath+= "_resized";
    	File parentfile=new File(parentpath);
    	if(!parentfile.exists()) {
    		parentfile.mkdirs();
    	}
        return parentpath+File.separator+name.substring(0, name.lastIndexOf('.')) + "_" + width + "x" + height + extension;
    }

    private int getImageFormat(String imagePath) {
        String extension = imagePath.substring(imagePath.lastIndexOf('.')).toLowerCase();
        switch (extension) {
            case ".png":
                return SWT.IMAGE_PNG;
            case ".jpg":
            case ".jpeg":
                return SWT.IMAGE_JPEG;
            // Add support for more image formats if required
            default:
                return SWT.IMAGE_UNDEFINED;
        }
    }
    public static boolean isValidImagePath(String imagePath) {
        // Check if the provided string is a valid local file path
        File file = new File(imagePath);
        return file.exists() && !file.isDirectory();
    }

    public static boolean isValidImageUrl(String imageUrl) {
        // Check if the provided string is a valid URL
        try {
        	URL url= new URL(imageUrl);
            try {
                ImageData imageData = new ImageData(url.openStream());
                Image img=  new Image(Display.getCurrent(), imageData);
                img.dispose();
            } catch (Exception e) {
                // Handle URL-related exceptions
                e.printStackTrace();
                return false;
            }
            return true;
        } catch (MalformedURLException e) {
            return false;
        }
    }
    public static boolean isValidImageURL(URL imageUrl) {
        try {
            return ImageIO.read(imageUrl) != null;
        } catch (IOException e) {
            return false;
        }
    }
    public static Image createImageFromPathOrUrl(String pathOrUrl) {

        Display display = Display.getCurrent();
        if (display == null) {
            throw new IllegalStateException("Method must be called from a valid SWT UI thread");
        }

        if (isValidImagePath(pathOrUrl)) {
        	Image img=new Image(display, pathOrUrl);
            return img;
        } else if (isValidImageUrl(pathOrUrl)) {
            try {
                URL imageUrl = new URL(pathOrUrl);
                ImageData imageData = new ImageData(imageUrl.openStream());
                
                Image img=new Image(display, imageData);
                return img;
            } catch (Exception e) {
                // Handle URL-related exceptions
                e.printStackTrace();
            }
        }

        return null; // Return null if neither path nor URL is valid
    }
    public static String getImageNameFromURL(String imageUrl) {
        try {
            URL url = new URL(imageUrl);
            String path = url.getPath();
            // Extract the filename from the path
            String[] parts = path.split("/");
            return parts[parts.length - 1];
        } catch (MalformedURLException e) {
            return null;
        }
    }
	public Map<String, Image> getImageCache() {
		return imageCache;
	}
    
}
