package gdxapp.object3d;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder.VertexInfo;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import dressing.mathutils.EarClipper;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Triangle;
import gdxapp.fabs3d.BezierCurve;
import geometry.Triangle3D;

public class JPullProfile extends Profile {
	
	private final String ID = "4fd9451d-437a-4411-b015-c0ea1b9018e8";
	
	private transient Vector3[] definingPoints = 
		{ new Vector3(18, 15, 0),
		  new Vector3(15, 15, 0),
		  new Vector3(15, 7, 0),
		  new Vector3(3, 7, 0),
		  new Vector3(3, 25, 0),
		  new Vector3(0, 25, 0),
		  new Vector3(0, 0, 0),
		  new Vector3(18, 0, 0)
		 };
	
		
	public JPullProfile() {
		super();
		name = "JPull";
		previewPath = "C:\\ProgramData\\supercad\\assets\\handlers\\jPull.png";
		modelId = UUID.fromString(ID);
		dimension.set(0.6f,0.025f,0.018f);
	}
	
	@Override
	public Model getModel() {
		if(model == null)
			createModel();
		return model;
	}

	


	@Override
	public JPullProfile cpy() {
		JPullProfile clone = new JPullProfile();
		clone.setModelId(modelId);
		clone.setName(name);
		clone.setDimension(dimension.cpy());
		clone.setPrice(price);
		if(material != null)
			clone.setMaterial(material.cpy());
		clone.setDimension(dimension);
		clone.setPreviewPath(this.previewPath);
		clone.setPreviewImg(previewImg);
		return clone;
	}

	@Override
	public void createModel() {
		clearMeshData();
		createContour();
		triangulate();
		clean();
		build();
	}

	private void clearMeshData() {
		contour.clear();
		triangles.clear();
	}

	private void clean() {
		float epsilon = 0.01f;
		ArrayList<Vector3> vertices = new ArrayList<Vector3>();
		for(var t: triangles) {
			for(var vertex: t.getVertices()) {
				boolean exist = false;
				for(var vertexX: vertices) {
					if(vertexX.epsilonEquals(vertex, epsilon)) {
						exist = true;
						break;
					}
				}
				if(!exist) {
					vertices.add(vertex);
				}
			}
		}
		Vector3[] bounds =  MathUtilities.getBoundaries(vertices);
		Vector3 center = bounds[0].cpy().add(bounds[1]).scl(0.5f);
		for(var t: triangles) {
			for(var vertex: t.getVertices()) {
				vertex.sub(center).rotate(Vector3.Y, 90).scl(0.001f);
			}
		}
		
	}

	private void build() {
		// generate normals and uvs
		ModelBuilder builder = new ModelBuilder();
		builder.begin();
		int attr = VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal
				| VertexAttributes.Usage.TextureCoordinates;
		MeshPartBuilder meshPartBuilder = builder.part("root", GL20.GL_TRIANGLES, attr, material);
		Vector3 reference = new Vector3();
		for (var triangle : triangles) {
			Vector3 p0 = triangle.getVertices()[0];
			Vector3 p1 = triangle.getVertices()[1];
			Vector3 p2 = triangle.getVertices()[2];
			Vector3 N = p1.cpy().sub(p0).crs(p2.cpy().sub(p0)).nor();
			Vector3 T = p1.cpy().sub(p0).nor();
			Vector3 B = N.cpy().crs(T);
			Matrix4 transform = new Matrix4().set(T, B, N, reference);
			Vector3 uv0 = p0.cpy().mul(transform);
			Vector3 uv1 = p1.cpy().mul(transform);
			Vector3 uv2 = p2.cpy().mul(transform);
			VertexInfo i0 = new VertexInfo();
			i0.setPos(p0);
			i0.setNor(N);
			i0.setUV(uv0.x, uv0.y);
			VertexInfo i1 = new VertexInfo();
			i1.setPos(p1);
			i1.setNor(N);
			i1.setUV(uv1.x, uv1.y);
			VertexInfo i2 = new VertexInfo();
			i2.setPos(p2);
			i2.setNor(N);
			i2.setUV(uv2.x, uv2.y);
			meshPartBuilder.triangle(i0, i1, i2);
		}
		model = builder.end();
	}

	public void triangulate() {
		ArrayList<Vector2> planeVertices = new ArrayList<Vector2>();
		for (int i = 0; i < contour.size(); i++) {
			planeVertices.add(new Vector2(contour.get(i) .x, contour.get(i).y));
		}
		float depth = 600;
		ArrayList<Triangle> plane = EarClipper.triangulate(planeVertices);
		ArrayList<Triangle3D> front = new ArrayList<Triangle3D>();
		ArrayList<Triangle3D> back = new ArrayList<Triangle3D>();
		var d = new Vector3(0, 0, depth / 2.0f);
		plane.stream().forEach(v -> {
			var v0 = new Vector3(v.getV0(), 0);
			var v1 = new Vector3(v.getV1(), 0);
			var v2 = new Vector3(v.getV2(), 0);

			front.add(new Triangle3D(v0.cpy().add(d), v1.cpy().add(d), v2.cpy().add(d)));
			back.add(new Triangle3D(v0.cpy().sub(d), v2.cpy().sub(d), v1.cpy().sub(d)));
		});
		triangles = new ArrayList<Triangle3D>();
		triangles.addAll(back);
		triangles.addAll(front);
		// side triangles
		ArrayList<Vector3> frontVertices = new ArrayList<Vector3>();
		ArrayList<Vector3> backVertices = new ArrayList<Vector3>();
		contour.forEach(v -> {
			frontVertices.add(new Vector3(v).add(d));
			backVertices.add(new Vector3(v).sub(d));
		});
		for (int i = 0; i < frontVertices.size(); i++) {
			int nexIndex = (i+1)%frontVertices.size();
			Triangle3D t1 = new Triangle3D(frontVertices.get(i).cpy(), backVertices.get(i).cpy(), frontVertices.get(nexIndex).cpy());
			Triangle3D t2 = new Triangle3D(frontVertices.get(nexIndex).cpy(), backVertices.get(i).cpy(), backVertices.get(nexIndex).cpy());
			triangles.add(t1);
			triangles.add(t2);
		}
	}

	public void createContour() {
		contour = new ArrayList<Vector3>();
		contour.add(definingPoints[0]);
		BezierCurve curve = new BezierCurve(48, definingPoints[1], definingPoints[2], definingPoints[3], definingPoints[4]);
		curve.calculateAxisPoints();
		contour.addAll(curve.getVertices());
		for (int i = 5; i < definingPoints.length; i++) {
			contour.add(definingPoints[i]);
		}
	}

	public Vector3[] getDefiningPoints() {
		return definingPoints;
	}

	public void setDefiningPoints(Vector3[] definingPoints) {
		this.definingPoints = definingPoints;
	}

}
