package org.frs.svg;

import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector3;

public class ProjectionCamera {
	
	private float near;
	private float hfov;
	private float aspectRatio; //projector height by width ratio
	private float screenWidth;
	
	private Vector3 position = new Vector3();
	private Vector3 direction;
	private Vector3 up = new Vector3(0,1,0);
	private Vector3 right = new Vector3(1,0,0);
	
	private Matrix4 view = new Matrix4();
	private Matrix4 projection = new Matrix4();					//use this matrix for perspective projection
	private Matrix4 orthoProjection = new Matrix4();			//use this matrix for orthographic projection

	boolean ortho = true;
	
	public ProjectionCamera(float near, float hfov, float aspectRatio, float screenWidth) {
		this.near = near;
		this.hfov = hfov;
		this.aspectRatio = aspectRatio;
		this.screenWidth = screenWidth;
		calculateProjectionTransform();
	}
	
	
	public void lookAt(Vector3 target) {
		view.setToLookAt(position, target, up);
	}
	
	private void calculateProjectionTransform() {
		
		float visibleWorldWidth = (float) Math.abs((2 * this.near * Math.tan(Math.toRadians(hfov/2.0f))));
		float screenHeight = screenWidth * aspectRatio;
		//perspective projection
		float a = screenWidth /visibleWorldWidth ;
		Vector3 m = new Vector3(a,a,1).scl(near);
		Vector3 t = new Vector3(screenWidth/2.0f, screenHeight/2.0f, 0);
		projection.set(m.cpy().scl(1,0,0), m.cpy().scl(0,1,0), m.cpy().scl(0,0,1),  t);
		//ortho projection
		m.set(a,a,1);
		orthoProjection.set(m.cpy().scl(1, 0, 0), m.cpy().scl(0, 1, 0), m.cpy().scl(0, 0, 1), t);
	}
	
	
	public Vector3 project(Vector3 point) {
		Vector3 eyePoint = point.cpy().mul(view);
		if(ortho) {
			//eyePoint.scl(1, 1, near/eyePoint.z);
			return eyePoint.mul(orthoProjection);
		}else {
			eyePoint.scl(1.0f/eyePoint.z);
			return eyePoint.mul(projection);
		}
	}
	
	public Vector3 toScreenSpace(Vector3 point) {
		Vector3 tmp = project(point);
		tmp.y = getScreenHeight() - tmp.y;
		return tmp;
	}
	
	
	
	public boolean isOrtho() {
		return ortho;
	}


	public void setOrtho(boolean ortho) {
		this.ortho = ortho;
	}


	public Vector3 getPosition() {
		return position;
	}

	public void setPosition(Vector3 position) {
		this.position = position;
	}


	public float getAspectRatio() {
		return aspectRatio;
	}

	public void setAspectRatio(float aspectRatio) {
		this.aspectRatio = aspectRatio;
	}

	public float getScreenWidth() {
		return screenWidth;
	}
	
	public float getScreenHeight() {
		return screenWidth * aspectRatio;
	}

	public void setScreenWidth(float screenWidth) {
		this.screenWidth = screenWidth;
	}
	
	

}
