package org.frs.svg;

import java.util.ArrayList;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.utils.GdxNativesLoader;


public class Triangle {

	static {
		GdxNativesLoader.load();
	}
	
	private Vector3 v0;
	private Vector3 v1;
	private Vector3 v2;
	
	private Vector3 color;
	
	public Triangle(Vector3 v0, Vector3 v1, Vector3 v2, Vector3... color) {
		super();
		this.v0 = v0.cpy();
		this.v1 = v1.cpy();
		this.v2 = v2.cpy();
		if(color.length > 0)
			this.color = color[0];
	}
	
	public void tesselate(int level, ArrayList<Triangle> out) {
		if(--level > 0) {
			Vector3 barycenter = v0.cpy().add(v1).add(v2).scl(1.0f/3.0f);
			Vector3 m0 = v1.cpy().add(v2).scl(0.5f);
			Vector3 m1 = v0.cpy().add(v2).scl(0.5f);
			Vector3 m2 = v1.cpy().add(v0).scl(0.5f);

			Triangle t0 = new Triangle(barycenter.cpy(), v0.cpy(), m2.cpy(), color);
			Triangle t1 = new Triangle(barycenter.cpy(), m2.cpy(), v1.cpy(), color);
			Triangle t2 = new Triangle(barycenter.cpy(), v1.cpy(), m0.cpy(), color);
			
			Triangle t3 = new Triangle(barycenter.cpy(), m0.cpy(), v2.cpy(), color);
			Triangle t4 = new Triangle(barycenter.cpy(), v2.cpy(), m1.cpy(), color);
			Triangle t5 = new Triangle(barycenter.cpy(), m1.cpy(), v0.cpy(), color);
			t0.tesselate(level, out);
			t1.tesselate(level, out);
			t2.tesselate(level, out);
			t3.tesselate(level, out);
			t4.tesselate(level, out);
			t5.tesselate(level, out);
			
		}else {
			out.add(this);
		}
		
	}

	public LineSegment[] getEdges() {
		return new LineSegment[] {new LineSegment(v0, v1), new LineSegment(v1, v2), new LineSegment(v2, v0)};
	}
	
	float getArea() {
		Vector3 base = v1.cpy().sub(v0);
		Vector3 a = v2.cpy().sub(v0);
		Vector3 p = base.cpy().nor().scl(a.dot(base)/base.len());
		a.sub(p);
		return a.len() * base.len();
	}
	
	//sample the points with barycentric coords (1,1,1), (0.8,0.1,0.1), (0.1,0.8,0.1)and (0.1,0.1,0.8)
	public Vector3[] getSmaples() {
		Vector3[] samples = new Vector3[4];
		samples[0] = getIsocenter();
		samples[1] = v0.cpy().scl(0.9f).add(v1.cpy().scl(0.05f)).add(v2.cpy().scl(0.05f));
		samples[2] = v1.cpy().scl(0.9f).add(v0.cpy().scl(0.05f)).add(v2.cpy().scl(0.05f));
		samples[3] = v2.cpy().scl(0.9f).add(v1.cpy().scl(0.05f)).add(v0.cpy().scl(0.05f));
		return samples;
	}

	public Triangle cpy() {
		Triangle clone = new Triangle(v0.cpy(), v1.cpy(), v2.cpy());
		if(color != null)
			clone.setColor(color.cpy());
		return clone;
	}

	public Triangle transform(Matrix4 transform) {
		v0.mul(transform);
		v1.mul(transform);
		v2.mul(transform);
		return this;
	}
	
	public Vector3 getIsocenter() {
		return v0.cpy().add(v1).add(v2).scl(1/3.0f);
	}
	
	
	public boolean contains(Vector3 point, boolean strictly) {
		Vector3 U = v1.cpy().sub(v0);
		Vector3 V = v2.cpy().sub(v0);
		Vector3 P = point.cpy().sub(v0);
		float[] values = new float[] {
				U.len2(), U.dot(V), 0,
				U.dot(V), V.len2(), 0,
				0,		0, 			1
		};
		Matrix3 M = new Matrix3(values);
		if(M.det() == 0)
			return false;
		Vector2 W = new Vector2(P.dot(U), P.dot(V)).mul(M.inv());
		float epsilon = strictly?0.001f:-0.001f;
		return (W.x > epsilon && W.y > epsilon) &&  W.x + W.y  < 1 - epsilon;
		
	}
	
	public boolean hasVertex(Vector3 point) {
		return v0.epsilonEquals(point, 1) || v1.epsilonEquals(point,1) || v2.epsilonEquals(point, 1);
	}
	
	public Vector3 intersect(Line line) {
		Object intersection = line.intersect(getPlane());
		if(intersection != null && intersection instanceof Vector3) {
			Vector3 point = (Vector3) intersection;
			if(contains(point, false))
				return point;
		}
		return null;
	}
	
	public boolean overlap(Triangle other) {
		if(!getPlane().equals(other.getPlane()))
			return false;
		
		for(Vector3 vertex: other.getVertices()) {
			if(contains(vertex, true))
				return true;
		}
		
		for(Vector3 vertex: getVertices()) {
			if(other.contains(vertex, true))
				return true;
		}	
		for(LineSegment edge: other.getEdges()) {
			for(LineSegment edgeX: getEdges()) {
				Vector3 intersecion = edge.intersect(edgeX);
				if(intersecion != null && !edge.startOrEndWith(intersecion, 0.01f) && !edgeX.startOrEndWith(intersecion, 0.01f) )
					return true;
			}
		}
		Triangle large = this.getArea()> other.getArea()?this:other;
		Triangle small = large == this?other:this;
		int i = 0;
		for(Vector3 v: small.getVertices()) {
			if(large.contains(v,false))
				i++;
		}
		
		return i==3;
	}
	public ArrayList<Vector3> intersect(LineSegment seg){
		ArrayList<Vector3> intersectionPoints = new ArrayList<Vector3>();
		for(LineSegment segX: getEdges() ) {
			Vector3 cut = segX.intersect(seg);
			if(cut != null ) {
				boolean newPoint = true;
				for(Vector3 cutPoint: intersectionPoints) {
					if(cutPoint.epsilonEquals(cut, 0.1f)) {
						newPoint = false;
						break;
					}
				}
				if(newPoint)
					intersectionPoints.add(cut);

			}
		}
		return intersectionPoints;
	}
	
	public Triangle[] slice(Plane plane) {
		return null;
	}
	
	public boolean intersect(Triangle other) {
		Plane p0 = new Plane(v0, v1, v2);
		Plane otherPlane = new Plane(other.getIsocenter(), other.getNormal());
		Object planeIntersection = p0.intersects(otherPlane);
		if(planeIntersection == null)
			return false;
		if(planeIntersection instanceof Plane) {
			//handle coplanar triangle intersections
			//one contains the other
			if(contains(other) || other.contains(this))
				return true;
			//they have at least two intersecting segment
			
		}
		for(LineSegment seg: getEdges()) {
			for(var segX: other.getEdges()) {
				if(seg.intersect(segX) != null)
					return true;
			}
		}
		return false;
	}
	
	public boolean contains(Triangle other) {
		return contains(other.getV0(),false) && contains(other.getV1(), false) && contains(other.getV2(), false);
	}
	
	//return the vertex whose image when applyiing the given transform is equal to the provided vertex
	public Vector3 getEquivalentVertex(Vector3 vertex, Matrix4 transform) {
		for(Vector3 v: getVertices()) {
			Vector3 image = v.cpy().mul(transform);
			if(image.epsilonEquals(vertex, 1))
				return v;
		}
		return null;
	}
	
	//return the vertex whose image when applying the given transform is equal to the provided vertex
		public LineSegment getEquivalentEdge(LineSegment edge, Matrix4 transform) {
			for(LineSegment edgeX: getEdges()) {
				LineSegment image = edgeX.cpy().transform(transform);
				if(image.equals(edge))
					return edgeX;
			}
			return null;
		}
	
	public Plane getPlane() {
		return new Plane(v0, v1.cpy().sub(v0).crs(v2.cpy().sub(v0)).nor());
	}

	public Vector3 getV0() {
		return v0;
	}

	public void setV0(Vector3 v0) {
		this.v0 = v0;
	}

	public Vector3 getV1() {
		return v1;
	}

	public void setV1(Vector3 v1) {
		this.v1 = v1;
	}

	public Vector3 getV2() {
		return v2;
	}

	public void setV2(Vector3 v2) {
		this.v2 = v2;
	}
	
	
	public Vector3 getBarycentricCoords(Vector3 point) {
		Vector3 e0 = v1.cpy().sub(v0);
		Vector3 e1 = v2.cpy().sub(v0);
		float values[] = new float[] {
				e0.x, e0.y, 0,
				e1.x, e1.y, 0,
				0   , 0   , 1
		};
		Matrix3 M = new Matrix3(values);
		if(M.det() == 0)
			return null;
		Vector3 relativeCoords = point.cpy().sub(v0).mul(M.inv());
		Vector3 barycentricCoords = new Vector3(1 - relativeCoords.x - relativeCoords.y, relativeCoords.x, relativeCoords.y);
		return barycentricCoords;
	}
	
	public Vector3 getPointFromGeocentricCoords(Vector3 point) {
		float values[] = new float[] {
				v0.x, v0.y, v0.z,
				v1.x, v1.y, v1.z,
				v2.x, v2.y, v2.z
		};
		Matrix3 M = new Matrix3(values);
		return point.cpy().mul(M);
	}
	
	
	


	
	
	public Vector3 getColor() {
		return color;
	}

	public void setColor(Vector3 color) {
		this.color = color;
	}

	@Override
	public String toString() {
		return "Triangle [v0=" + v0 + ", v1=" + v1 + ", v2=" + v2 + "]";
	}

	public void print(ProjectionCamera camera) {
		
	}

	public Vector3 getNormal() {
		return v2.cpy().sub(v0).crs(v1.cpy().sub(v0)).nor();
	}

	public Vector3[] getVertices() {
		
		return new Vector3[] {v0, v1, v2};
	}

	public boolean contains(LineSegment edge) {
		for(LineSegment segment: getEdges()) {
			if(segment.equals(edge)  || segment.contains(edge))
				return true;
		}
		return false;
	}
	
	public boolean contains(Triangle other, boolean strictly) {
		for(Vector3 vertex: other.getVertices()) {
			boolean condition = contains(vertex,strictly);
			if(!condition)
				return false;
		}
		return true;
	}
	
	
	public boolean equals(Triangle other) {
		for(Vector3 vertex: other.getVertices()) {
			if(!hasVertex(vertex))
				return false;
		}
		return true;
	}

	public boolean isTangent(LineSegment edge) {
		if(contains(edge))
			return true;
		ArrayList<Vector3> intersection = intersect(edge);
		if(intersection.size() == 1){
//			System.err.println();
			return !(contains(edge.getV0(), true) || contains(edge.getV1(), true));

		}else {
			for(LineSegment edgeX: getEdges()) {
				if(edgeX.getDirection(false).isOnLine(edge.getDirection(false))) {
					boolean cond1 = edgeX.contains(edge.getV0()) || edgeX.contains(edge.getV1());
					boolean cond2 = edge.contains(edgeX.getV0())  || edge.contains(edgeX.getV1()); 
					if(cond1 || cond2)
						return true;
				}
			}
		}
		return false;
	}

	public boolean isValid() {
		Vector2 a0 = new Vector2(v0.x, v0.y);
		Vector2 a1 = new Vector2(v1.x, v1.y);
		Vector2 a2 = new Vector2(v2.x, v2.y);
		float angle0 = Math.abs(a1.cpy().sub(a0).angle(a2.cpy().sub(a0)));
		float angle1 = Math.abs(a0.cpy().sub(a1).angle(a2.cpy().sub(a1)));
		float angle2 = 180 - angle0 - angle1;
		return Math.min(Math.min(angle0, angle1), angle2) > 1;
	}
	
	
	
	
	public static void main(String...args) {
		Vector3 v0 = new Vector3(-.5f,1,.5f), v1 = new Vector3(.5f,-1,.5f), v2 = new Vector3(.5f,1,.5f);
		Vector3 p = new Vector3(-0.4099099f,1.018018f,0.5f);
		Triangle t = new Triangle(v0, v1, v2);
		System.out.println(t.contains(p, false));
		
	}
	
	
}
