package api.mep;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import org.joml.Matrix3f;
import org.joml.Matrix4f;
import org.joml.Vector2i;
import org.joml.Vector3f;
import org.joml.Vector4f;
import org.lwjgl.opengl.GL45;

import api.backend.ApplicationContext;
import api.events.CustomEvent;
import api.events.EventBus;
import api.events.EventHandler;
import api.events.EventType;
import api.graphics.Camera.CAMERA_MODE;
import api.graphics.Camera.PROJECTION_MODE;
import api.graphics.IDrawable;
import api.graphics.Mesh;
import api.graphics.Model;
import api.graphics.ModelInstance;
import api.graphics.ObjectController;
import api.graphics.RenderingMode;
import api.graphics.Scene;
import api.graphics.SelectionMode;
import api.graphics.geometry.MeshBuilder;
import api.graphics.geometry.Triangle;
import api.input.InputMultiplexer;
import api.provider.ModelProvider;
import api.ui.RightClickMenuManager;
import api.utils.IOUtils;
import api.utils.MathUtilities;
import api.utils.ScreenUtils;

public class EditorScene extends Scene implements EventHandler {

	private final ArrayList<ModelInstance> objects = new ArrayList<ModelInstance>();
	private QuotationDrawer quotsDrawer;
	private int mode;
	private SelectionMode selectionMode;
	private MeasureTaker measureTaker;
	private ObjectController objectController;
	

	public EditorScene() {
		this.init();
	}

	@Override
	public void init() {
		super.init();
		ObjectSelector.getSelector().init();
		ObjectSelector.getSelector().setCamera(camera);
		ObjectSelector.getSelector().setObjects(objects);
		SurfaceController.getController();
		addDrawable(ObjectSelector.getSelector());
		addDrawable(PipeDrawer.getDrawer());
		InputMultiplexer.getInstance().pushEventHandler(ObjectSelector.getSelector());
		InputMultiplexer.getInstance().pushEventHandler(RightClickMenuManager.getManager());
		subscribe(EventType.SELECTION_MODE_CHANGED);
		subscribe(EventType.INVOKE_PIPE_DRAWER);
		subscribe(EventType.OBJECT_SELECTION_TRIGFERED);
		subscribe(EventType.DROP_OBJECT);
		subscribe(EventType.ADD_RENDERABLE);
		subscribe(EventType.HIDE_ELEMENT);
		subscribe(EventType.DELETE_OBJECT);
		subscribe(EventType.INVOKE_RULER, EventType.REVOKE_RULER);
		subscribe(EventType.PROJECTION_MODE_CHANGED);
		subscribe(EventType.OBJECT_SELECTED);
		subscribe(EventType.ISOLATE, EventType.END_ISOLATE);
		this.quotsDrawer = new QuotationDrawer(objects);
		this.measureTaker = new MeasureTaker();
		this.objectController = new ObjectController(camera);
		//addDrawable(this.objectController);
		addDrawable(cameraController);
	}

	public void loadScene(String path) {
		File folder = new File(path);
		File[] modelFiles = folder.listFiles(new FileFilter() {
			@Override
			public boolean accept(File pathname) {
				return pathname.getName().endsWith(".obj");
			}
		});

		for (File modelFile : modelFiles) {
			Model model = new Model(modelFile.getAbsolutePath());
			ModelInstance instance = new ModelInstance(model);
			for(Mesh mesh:instance)
				mesh.calculateSilhouetteEdges();
			instance.setRenderingMode(RenderingMode.WIRED_FACETS);
			// load transform
			String transformFilePath = modelFile.getPath().replace(".obj", ".txt");
			var transform = IOUtils.readTransformFile(transformFilePath);
			instance.setTransform(transform);
			addDrawable(instance);
		}
	}

	@Override
	public void handle(CustomEvent event) {
		EventType type = event.getType();

		switch (type) {

		case SELECTION_MODE_CHANGED:
			SelectionMode mode = (SelectionMode) event.getDetail("mode");
			setRenderingMode(mode);
			break;
		case INVOKE_PIPE_DRAWER:
			PIPE_TYPE pipeType = (PIPE_TYPE) event.getDetail("Type");
			invokePipeDrawer(pipeType);
			break;
		case OBJECT_SELECTION_TRIGFERED:
			InputMultiplexer.getInstance().removeEventHandler(PipeDrawer.getDrawer());
			break;
		case DROP_OBJECT:
			dropObject(event);
			break;
		case ADD_RENDERABLE:
			var renderable = event.getDetail("renderable");
			addDrawable((IDrawable) renderable);
			break;
		case ISOLATE:
			List<Triangle> target;
			try {
				target = (List<Triangle>) event.getDetail("ortho");
			} catch (ClassCastException e) {
				break;
			}
			isolate(target);
			break;
		case HIDE_ELEMENT:
			List<ModelInstance> elements = (List<ModelInstance>) event.getDetail("elements");
			hideDrawable(elements);
			break;
		case REMOVE_DRAWABLE:
			IDrawable drawable = (IDrawable) event.getDetail("drawable");
			if (drawable != null) {
				removeDrawable(drawable);
			}
			break;
		case DELETE_OBJECT:
			ModelInstance object = (ModelInstance) event.getDetail("object");
			deleteObject(object);
			break;
		case INVOKE_RULER:
			InputMultiplexer.getInstance().insertEventHandler(measureTaker, 0);
			addDrawable(measureTaker);
			break;
		case REVOKE_RULER:
			removeDrawable(measureTaker);
			break;
		case PROJECTION_MODE_CHANGED:
			setProjectionMode(event);
			break;
		case OBJECT_SELECTED:
			onObjectSelection(event);
			break;
		case END_ISOLATE:
			endIsolation();
			break;
		default:
			throw new IllegalArgumentException("Unexpected value: " + type);
		}

	}

	private void onObjectSelection(CustomEvent selectionEvent) {
		try {
			ModelInstance selection = (ModelInstance) selectionEvent.getDetail("selection");
			cameraController.getOrbitCamera().setTarget(selection);
			this.objectController.setTarget(selection);
		}catch (Exception e) {
			e.printStackTrace();
		}
		
	}

	private void setProjectionMode(CustomEvent event) {
		try {
			var projMode =  event.getDetail("proj_mode");
			if(projMode != null) {
				if(projMode == PROJECTION_MODE.ORTHOGRAPHIC) {
					cameraController.setToOrtho();
				}else if(projMode == PROJECTION_MODE.PERSPECTIVE) {
					cameraController.setToPerspective();
				}
			}
			
			var cameraMode = event.getDetail("camera_mode");
			if(cameraMode != null) {
				cameraController.setMode((CAMERA_MODE) cameraMode);
			}
		}catch (Exception e) {
			System.err.println("wrong event data for PROJECTION_MODE_CHANGED");
		}
	}

	public void setRenderingMode(SelectionMode mode) {
		this.selectionMode = mode;
	}

	private void deleteObject(ModelInstance object) {
		drawables.remove(object);
		objects.remove(object);
	}

	private void hideDrawable(List<ModelInstance> elements) {
		for (ModelInstance element : elements) {
			element.setHidden(true);
		}
	}

	private void dropObject(CustomEvent event) {
		try {
			String modelName = (String) event.getDetail("model name");
			Vector2i location = (Vector2i) event.getDetail("location");
			var info = ModelProvider.getProvider().findModelByName(modelName);
			float z = ScreenUtils.readDepth(location.x, location.y);
			float y = ApplicationContext.getHeight() - location.y;
			Vector3f position = camera.unproject(new Vector3f(location.x, y, z), ApplicationContext.getViewport());
			Triangle face = ObjectSelector.getSelector().selectPoly(position);
			float rotation = 0;
			Vector3f normal = face.getNormal();
			if (face != null) {
				var dirToCamera = new Vector3f(camera.getPosition()).sub(position);
				if(normal.dot(dirToCamera) < 0.0f)
					normal.mul(-1.0f);
				Vector4f normal4 = new Vector4f(normal, 0).mul(face.transform).normalize();
				normal.set(normal4.x, normal4.y, normal4.z);
				Vector3f director = new Vector3f(normal).rotateAxis((float) Math.PI / 2.0f, 0, 1, 0);
				rotation = (float) Math.atan2(director.z, director.x);
			}
			position.add(normal.mul(info.getScale().z / 2.0f));
			Matrix4f mat = new Matrix4f().translate(position).rotate(-rotation, new Vector3f(0, 1, 0));
			ModelInstance instance = ModelProvider.getProvider().createInstanceOf(modelName);
			instance.transform(mat);
			System.out.println("drop location :" + position);
			addDrawable(instance);
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	@Override
	public void render(long delta) {
		GL45.glPolygonMode(GL45.GL_FRONT_AND_BACK, mode);
		super.render(delta);
		if (selectionMode != null) {
			switch (selectionMode) {
			case VERTICES:
				drawVertices();
				break;
			case EDGES:
				break;
			case SILHOUETTE:
				break;
			case POLY:
				drawPolies();
				break;
			default:
			}
		}
		GL45.glPolygonMode(GL45.GL_FRONT_AND_BACK, GL45.GL_FILL);

	}
	private void drawVertices() {
		debugShader.begin(camera);
		debugShader.setVec3("color", new Vector3f(1,0,1));
		debugShader.setFloat("opacity", 0.5f);
		for (IDrawable drawable : drawables) {
			drawable.drawVertices(camera);
		}
	}

	private void drawPolies() {
		debugShader.begin(camera);
		debugShader.setVec3("color", new Vector3f(1,1,0));
		debugShader.setFloat("opacity", 0.5f);
		for (IDrawable drawable : drawables) {
			drawable.draw(debugShader, GL45.GL_LINE);
		}
	}

	private void invokePipeDrawer(PIPE_TYPE pipeType) {
		PipeDrawer.getDrawer().begin(camera, pipeType);
		InputMultiplexer.getInstance().insertEventHandler(PipeDrawer.getDrawer(), 0);
		InputMultiplexer.getInstance().removeEventHandler(ObjectSelector.getSelector());
		// drawables.remove(ObjectSelector.getSelector());
	}

	@Override
	public void addDrawable(IDrawable drawable) {
		this.drawables.add(drawable);
		this.drawables.sort(new Comparator<IDrawable>() {

			@Override
			public int compare(IDrawable o1, IDrawable o2) {
				if (o1 instanceof ModelInstance && !(o2 instanceof ModelInstance))
					return -1;
				if (!(o1 instanceof ModelInstance) && o2 instanceof ModelInstance)
					return 1;

				// both re model instances
				if (o1 instanceof ModelInstance && o2 instanceof ModelInstance) {
					float d1 = getCamera().getPosition().distanceSquared(((ModelInstance) o1).getCenter());
					float d2 = getCamera().getPosition().distanceSquared(((ModelInstance) o2).getCenter());
					return (int) Math.signum(d1 - d2);
				}
				return 0;
			}
		});
		if (drawable instanceof ModelInstance) {
			objects.add((ModelInstance) drawable);
			HashMap<String, Object> details = new HashMap<String, Object>();
			details.put("object", drawable);
			EventBus.getInstance().notify(new CustomEvent(EventType.OBJECT_ADDED, details));
			if (quotsDrawer != null)
				quotsDrawer.generateQuoations();
		}
	}

	private void removeDrawable(IDrawable drawable) {
		this.drawables.remove(drawable);
	}

	public void isolate(List<Triangle> target) {
		ArrayList<Vector3f> vertices = new ArrayList<Vector3f>();
		for (Triangle face : target) {
			vertices.addAll(Arrays.asList(face.getCorners()));
		}
		Vector3f[] boundaries = MathUtilities.getBoundaries(vertices.toArray(new Vector3f[0]));
		Vector3f halfSize = new Vector3f(boundaries[1]).sub(boundaries[0]).mul(0.5f);

		Vector3f B = new Vector3f(0, 1, 0);

		Vector3f center = new Vector3f(boundaries[0]).add(boundaries[1]).mul(0.5f);
		Vector3f normal = target.get(0).getNormal().normalize();
		Vector3f T = new Vector3f();
		B.cross(normal, T);

		Matrix4f localToWorld = new Matrix4f(new Vector4f(T, 0), new Vector4f(B, 0), new Vector4f(normal, 0),
				new Vector4f(center, 1));
		Vector4f low = new Vector4f(boundaries[0], 1);
		Vector4f max = new Vector4f(boundaries[1], 1);
		Vector3f lowerBound = new Vector3f(low.x, low.y, low.z);
		Vector3f upperBound = new Vector3f(max.x, max.y, max.z);
		HashMap<String, Object> details = new HashMap<String, Object>();
		details.put("transform", localToWorld);
		details.put("lower bound", lowerBound);
		details.put("upper bound", upperBound);
		details.put("objects", this.objects);
		details.put("camera", camera);
		var instance = ObjectSelector.getSelector().getSelectedInstance();
		if (instance != null) {
			details.put("instance", instance);
		}
		EventBus.getInstance().notify(new CustomEvent(EventType.SURFACE_MODE, details));
		//set camera to ortho against the isolated  object
		Vector3f cameraPosition = new Vector3f(center).add(normal);
		camera.setPosition(cameraPosition);
		camera.lookAt(center);
		cameraController.setToOrtho();
		cameraController.freezeCamera(true);;
		
		addDrawable(SurfaceController.getController());

	}
	
	public void endIsolation() {
		for(ModelInstance obj: objects)
			obj.setHidden(false);
		cameraController.setToPerspective();
		cameraController.freezeCamera(false);
		removeDrawable(SurfaceController.getController());
	}
	
	public int getMode() {
		return mode;
	}

	public void setMode(int mode) {
		this.mode = mode;
	}

}
