package geometry;

import java.util.ArrayList;

import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.InputListener;

import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.ui.project.CreateFormWizard;
import dressing.ui.util.EdgeMesureWidow;
import dressing.ui.util.OpenWizardDialog;
import gdxapp.assets.DrawingHelper;
import gdxapp.object3d.GeometryObject;
import gdxapp.object3d.Wall;
import gdxapp.screens.room.RoomController;

public class PolygonDrawer extends Actor implements IDrawer<Polygon> {

	
	private Polygon out;   //the object being drawn
	private GeometryObject editable;
	private final ArrayList<Vector2> vertices = new ArrayList<>();
	private final ArrayList<Edge> edges = new ArrayList<Edge>();
	private Vector2 tempVertex;
	
	private ShapeRenderer shapeRenderer;
	private Vector3[] angles;
	private Vector2 selectedVertex;
	private Edge selectedEdge;
	private boolean closed;
	private Vector2[] bounds;
	
	private PolygonDrawerListenner listener;
	
	private Matrix4 transform;       //the transform where the vertices have been picked
	private boolean createMode;
	
	private static PolygonDrawer instance;
	

	private PolygonDrawer() {
		setWidth(Gdx.graphics.getWidth());
		setHeight(Gdx.graphics.getHeight());
		listener = new PolygonDrawerListenner();
		addListener(listener);
	}
	
	@Override
	public void begin(boolean closed) {
		clear();
		this.closed = closed;
		createMode = true;
		out = new Polygon();
	}
	
	@Override
	public void edit(Polygon shape) {
		Polygon poly = (Polygon) this.editable.getGeometry().getBorder().getNodes().get(0);
		out = poly;
		createMode = false;
		for (Vector2 vertex : poly.getVertices()) {
			this.vertices.add(vertex);
		}
		updateEdgesList();
		calculateAngles();
	}
	
	public void edit(GeometryObject object) {
		clear();
		this.editable = object;
		try {
			Polygon poly = (Polygon) object.getGeometry().getBorder().getNodes().get(0);
			edit(poly);
		}catch (Exception e) {
			System.err.println("unable to fetch polygon from geometry object");
		}
	}
	
	@Override
	public Polygon end() {
		if (this.vertices != null && this.vertices.size() > 1) {
			out.setVertices(this.vertices);
			clear();
			return out;
		}
		clear();
		return null;
	}

	public void clear() {
		vertices.clear();
		bounds = null;
		angles = null;
		edges.clear();
		tempVertex = null;
		selectedEdge = null;
		selectedVertex = null;
	}
	

	private boolean selectVertex(Vector2 point) {
		if (!this.vertices.isEmpty()) {
			float[] distances = new float[this.vertices.size()];
			for (int i = 0; i < distances.length; i++) {
				distances[i] = vertices.get(i).dst(point);
			}
			float min = distances[0];
			int minIndex = 0;
			for (int i = 1; i < distances.length; i++) {
				if (distances[i] < min) {
					min = distances[i];
					minIndex = i;
				}
			}
			if (distances[minIndex] < 10) {
				selectedVertex = vertices.get(minIndex);
				return true;
			}
		}
		return false;
	}

	private boolean selectEdge(Vector2 point) {
		selectedEdge = null;
		if (!this.edges.isEmpty()) {
			Edge nearest = edges.get(0);
			for (int i = 1; i < edges.size(); i++) {
				if (edges.get(i).distanceTo(point) < nearest.distanceTo(point))
					nearest = edges.get(i);
			}
			if (nearest.distanceTo(point) < 10) {
				selectedEdge = nearest;
				tempVertex = null;
				return true;
			}
		}
		return false;
	}

	public void updateEdgesList() {
		edges.clear();
		int count = vertices.size();
		if (count > 1) {
			for (int i = 0; i < count - 1; i++)
				edges.add(new Edge(vertices.get(i), vertices.get(i + 1)));
			if (closed)
				edges.add(new Edge(vertices.get(count - 1), vertices.get(0)));
		}
	}

	private void addVertex(Vector2 point) {
		if (vertices.isEmpty()) {
			vertices.add(point);
		} else if (vertices.size() == 1) {
			alignVH(point, vertices.get(0));
			vertices.add(point);
		} else {
			Edge edge = new Edge(vertices.get(vertices.size() - 1), point);
			for (Edge edgeX : edges) {
				Vector2 intersection = MathUtilities.getIntersectionPoint(edgeX, edge);
				if (!(edgeX.hasVertex(intersection) || edge.hasVertex(intersection))
						&& (edgeX.contains(intersection) && edge.contains(intersection))) {
					System.out.println("plygon is self intersecting!");
					float a = edgeX.getV0().dst(intersection);
					float b = edgeX.getV1().dst(intersection);
					float c = edgeX.getV0().dst(edgeX.getV1());
					System.out.println(a + " " + b + " " + c);
					return;
				}
			}
			vertices.add(point);
			rectifyPosition(point);

		}
		updateBoundaries();
		updateEdgesList();
	}

	private void updateBoundaries() {
		Vector2 lastPosition = vertices.get(vertices.size() - 1);
		if (this.bounds == null) {
			this.bounds = new Vector2[4];
			this.bounds[0] = lastPosition.cpy();
			this.bounds[1] = lastPosition.cpy();
			this.bounds[2] = lastPosition.cpy();
			this.bounds[3] = lastPosition.cpy();

		} else {
			this.bounds[0].x = Math.min(lastPosition.x, bounds[0].x);
			this.bounds[0].y = Math.min(lastPosition.y, bounds[0].y);
			this.bounds[1].x = Math.max(lastPosition.x, bounds[1].x);
			this.bounds[1].y = Math.max(lastPosition.y, bounds[1].y);
			this.bounds[2].set(this.bounds[0].x, this.bounds[1].y);
			this.bounds[3].set(this.bounds[1].x, this.bounds[0].y);
		}
	}

	public boolean link(Vector2 point) {
		Vector2 nearestPoint = null;
		for (Vector2 attachmentPoint : RoomController.getInstance().getCriticalPoints()) {
			if (nearestPoint == null) {
				nearestPoint = attachmentPoint;
				continue;
			}
			if (attachmentPoint.dst(point) < nearestPoint.dst(point)) {
				nearestPoint = attachmentPoint;
			}
		}

		if (nearestPoint != null && nearestPoint.dst(point) < 20) {
			point.set(nearestPoint);
			return true;
		}
		return false;
	}

	private void rectifyPosition(Vector2 point) {

		int index = vertices.indexOf(point);
		if (vertices.size() == 1)
			return;

		if (index == 1 && vertices.size() == 2) {
			alignVH(point, vertices.get(0));
			return;
		}

		Vector2 original = point.cpy();
		Vector2 next = index > vertices.size() - 2 ? null : vertices.get(index + 1);
		Vector2 previous = index > 0 ? vertices.get(index - 1) : null;
		try {
			if (previous == null) {
				Vector2 v0 = vertices.get(index + 2).cpy();
				Vector2 v1 = vertices.get(index + 1).cpy();
				Vector2 v01 = new Vector2(v0).sub(v1);
				Vector2 v21 = new Vector2(point).sub(v1);
				float angle = MathUtilities.signedAngle(v01, v21);
				float rads5 = (float) Math.toRadians(15);
				float rectifiedAngle = Math.round(angle / rads5) * rads5;
				Vector2 rectifiedDirection = v01.nor().rotateRad(rectifiedAngle).scl(v21.len());
				point.set(v1).add(rectifiedDirection);
			} else {
				Vector2 v0 = vertices.get(index - 2).cpy();
				Vector2 v1 = vertices.get(index - 1).cpy();
				Vector2 v01 = new Vector2(v0).sub(v1);
				Vector2 v21 = new Vector2(point).sub(v1);
				float angle = MathUtilities.signedAngle(v01, v21);
				float rads5 = (float) Math.toRadians(15);
				float rectifiedAngle = Math.round(angle / rads5) * rads5;
				Vector2 rectifiedDirection = v01.nor().rotateRad(rectifiedAngle).scl(v21.len());
				point.set(v1).add(rectifiedDirection);
			}
		} catch (IndexOutOfBoundsException e) {
			point.set(original);
		}
	}

	private void alignVH(Vector2 point, Vector2 origin) {
		Vector2 po = point.cpy().sub(origin);
		po.x = Math.abs(po.x);
		po.y = Math.abs(po.y);
		float cm10 = RoomController.getInstance().getScaleX() * 0.3f;
		if (po.x < po.y) {
			if (po.x < cm10) {
				point.x = origin.x;
			}
		} else {
			if (po.y < cm10) {
				point.y = origin.y;
			}
		}
	}

	void calculateAngles() {
		if (this.vertices != null && this.vertices.size() > 2) {
			this.angles = new Vector3[this.vertices.size()];
			int previousIndex, nextIndex;
			Vector2 v01, v21;
			for (int i = 0; i < vertices.size(); i++) {
				previousIndex = (i - 1) % vertices.size();
				if (previousIndex < 0)
					previousIndex += vertices.size();
				nextIndex = (i + 1) % vertices.size();
				v01 = new Vector2(vertices.get(previousIndex)).sub(vertices.get(i));
				v21 = new Vector2(vertices.get(nextIndex)).sub(vertices.get(i));
				float angle = MathUtilities.signedAngle(v01, v21);
				angles[i] = new Vector3(vertices.get(i), angle);
			}
		}
	}
	
	

	@Override
	public void draw(Batch batch, float parentAlpha) {
		getStage().setKeyboardFocus(this);
		boolean batchInterrupted = false;
		if (batch.isDrawing()) {
			batch.end();
			batchInterrupted = true;
		}
		draw(batch.getProjectionMatrix());
		if(batchInterrupted)
			batch.begin();
	}

	public void draw(Matrix4 projection) {
		if (shapeRenderer == null)
			shapeRenderer = new ShapeRenderer();
		shapeRenderer.setProjectionMatrix(projection);
		shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
		int verticesCount = closed ? vertices.size() + 1 : vertices.size();
		int nextIndex;
		for (int i = 0; i < vertices.size(); i++) {
			shapeRenderer.setColor(Color.GREEN);
			if (vertices.get(i) == selectedVertex)
				shapeRenderer.setColor(Color.RED);
			shapeRenderer.circle(vertices.get(i).x, vertices.get(i).y, 5.0f);
		}

		for (int i = 0; i < verticesCount - 1; i++) {
			nextIndex = (i + 1) % (vertices.size());
			shapeRenderer.setColor(Color.GREEN);
			shapeRenderer.rectLine(vertices.get(i), vertices.get(nextIndex), 5);
		}
		if (selectedEdge != null) {
			shapeRenderer.setColor(Color.RED);
			shapeRenderer.rectLine(selectedEdge.getV0(), selectedEdge.getV1(), 4);
			shapeRenderer.setColor(Color.BLACK);
		}
		shapeRenderer.end();

		if (edges != null) {
			for (Edge edge : edges) {
				float length = edge.getLength() / RoomController.getInstance().getScaleX();
				length = Math.round(1000 * length);
				Vector2 position = edge.getMiddle().add(edge.getNormal().scl(25));
				float rotation = edge.getV1().cpy().sub(edge.getV0()).angle();
				DrawingHelper.drawText(String.format("%.0f", length), new Vector3(position, 0.0f), rotation,
						Color.BLACK, 1.0f, RoomController.getInstance().getStage().getBatch());
			}
		}
		if (vertices.size() > 2) {
			for (int i = 0; i < vertices.size(); i++) {
				if (!closed && (i == 0 || i == vertices.size() - 1))
					continue;
				int nextVertex = (i + 1) % vertices.size();
				int previousVertex = (i - 1) % vertices.size();
				if (closed && i == 0) {
					previousVertex = vertices.size() - 1;
				}
				if (closed && i == vertices.size() - 1) {
					nextVertex = 0;
				}
				Vector2 vp = vertices.get(previousVertex).cpy().sub(vertices.get(i));
				Vector2 vn = vertices.get(nextVertex).cpy().sub(vertices.get(i));
				float rotation = (float) Math.toDegrees(MathUtilities.signedAngle(vp, vn));
				Vector2 translation = vp.cpy().nor().add(vn.cpy().nor()).nor().scl(-50);
				DrawingHelper.drawText(String.format("%.0f", rotation),
						new Vector3(translation.cpy().add(vertices.get(i)), 0.0f),
						translation.cpy().scl(-1).rotate(90).angle(), Color.BLUE, 1.0f,
						RoomController.getInstance().getStage().getBatch());
			}
		}

	}


	private void removeVertex(Vector2 vertex) {
		selectedEdge = null;
		vertices.remove(vertex);
	}

	public void setTempVertex(Vector2 tempVertex) {
		this.tempVertex = tempVertex;
	}

	@Override
	public ArrayList<Vector2> getVertices() {
		return vertices;
	}

	public boolean isClosed() {
		return closed;
	}

	public void setClosed(boolean closed) {
		this.closed = closed;
	}
	

	public Matrix4 getTransform() {
		return transform;
	}

	public void setTransform(Matrix4 transform) {
		this.transform = transform;
	}

	public static PolygonDrawer getInstance() {
		if(instance == null)
			instance = new PolygonDrawer();
		return instance;
	}

	public static void setInstance(PolygonDrawer instance) {
		PolygonDrawer.instance = instance;
	}




	class PolygonDrawerListenner extends InputListener {
		
		private float xoffset, yoffset;
		private long lastClickDate;
		private Vector2 currentMouseLocation;
		
		
		
		@Override
		public boolean touchDown(InputEvent event, float x, float y, int pointer, int button) {
			long currentTime = System.currentTimeMillis();
			xoffset = x;
			yoffset = y;
			Vector2 point = new Vector2(xoffset, yoffset);
			if (!selectVertex(point)) {
				if (selectEdge(point) && (currentTime - lastClickDate < 500)) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							
							Wall wall = null;
							if(editable instanceof Wall) {
								wall= (Wall)editable;
							}
							EdgeMesureWidow window = new EdgeMesureWidow(new Shell(),selectedEdge, wall);
							window.setEdges(edges);
							window.open();
						}
					});
				}
			}
			lastClickDate = currentTime;
			return true;
		}

		@Override
		public void touchUp(InputEvent event, float x, float y, int pointer, int button) {
			Vector2 point = new Vector2(x, y);
			if (button == Input.Buttons.LEFT) {
				if (selectedVertex == null && selectedEdge == null) {
					if (Gdx.input.isKeyPressed(Input.Keys.X)) {
						link(point);
						vertices.add(point);
					} else {
						addVertex(point);
					}
				} else {
					if (selectedVertex != null && (!Gdx.input.isKeyPressed(Input.Keys.SHIFT_LEFT)
							&& !Gdx.input.isKeyPressed(Input.Keys.SHIFT_RIGHT))) {
						if (Gdx.input.isKeyPressed(Input.Keys.X)) {
							link(selectedVertex);
						} else {
							rectifyPosition(selectedVertex);
						}
					}
				}
			}
			if (button == Input.Buttons.RIGHT) {
				selectVertex(point);
				removeVertex(selectedVertex);
			}
			updateEdgesList();
			calculateAngles();
			setTempVertex(null);
			selectedVertex = null;
		}

		@Override
		public void touchDragged(InputEvent event, float x, float y, int pointer) {
			setCurrentMouseLocation(x, y);
			float xmov = x - xoffset;
			float ymov = y - yoffset;
			if (vertices.size() > 0 && Gdx.input.isButtonPressed(Input.Buttons.LEFT) && selectedVertex == null
					&& selectedEdge == null) {
				setTempVertex(currentMouseLocation);
			}
			if (selectedVertex != null) {
				selectedVertex.add(xmov, ymov);
			} else if (selectedEdge != null) {
				selectedEdge.translate(xmov, ymov);
			}
			xoffset = x;
			yoffset = y;
		}

		@Override
		public boolean mouseMoved(InputEvent event, float x, float y) {
			setCurrentMouseLocation(x, y);
			return false;
		}

		@Override
		public boolean keyDown(InputEvent event, int keycode) {
			if (keycode == Input.Keys.F) {
				closed = !closed;
			}
			if(keycode ==Input.Keys.ESCAPE) {
				if(createMode) {
					Polygon poly = end();
					CompoundShape border = new CompoundShape(poly);
					CompoundObject geometry = new CompoundObject(border, null);
					geometry.setWorldTransform(transform);
					Display.getDefault().asyncExec(new Runnable() {
						@Override
						public void run() {
							CreateFormWizard formWizard = new CreateFormWizard(geometry);
							WizardDialog formWizarddialogue= new OpenWizardDialog(new Shell(Display.getCurrent()) ,formWizard);
							formWizard.setContainer(formWizarddialogue);
							formWizarddialogue.setModal(true);
							formWizarddialogue.setShellStyle(SWT.APPLICATION_MODAL);
							formWizarddialogue.open();
						}
					});
				}else {
					RoomController.getInstance().updateObject2D(editable);
					clear();
				}
				remove();
			}
			return false;
		}

		public Vector2 getCurrentMouseLocation() {
			return currentMouseLocation;
		}

		public void setCurrentMouseLocation(float x, float y) {
			if (currentMouseLocation == null)
				currentMouseLocation = new Vector2();
			currentMouseLocation.set(x, y);
		}

	}





}
