package gdxapp.quotation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.TreeSet;
import java.util.Vector;

import org.frs.svg.Line;
import org.frs.svg.LineSegment;

import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.ibm.icu.text.PersonNameFormatter.Length;

import dressing.events.Event;
import dressing.events.EventHandler;
import gdxapp.Commun.ActionProcessor;
import gdxapp.fabs3d.Quad;
import gdxapp.object3d.KitchenElement;
import gdxapp.object3d.Object3D;
import gdxapp.scenes.SceneEvent;
import gdxapp.screens.room3d.Room3DController;
import geometry.Box;
import geometry.Ray;
import geometry.Ray3D;

public class Object3DQuotationManager implements EventHandler{
	
	
	private Camera camera;
	private List<Object3D> objects;
	private static ShapeRenderer shapeRenderer;
	private final Vector<Object3DQuotation> quotations = new Vector<Object3DQuotation>();
	private ArrayList<Quad> sceneSides = new ArrayList<Quad>();
	private HashMap<Object3D, List<Object3DQuotation>> sourceGrps= new HashMap<Object3D, List<Object3DQuotation>>();
	private HashMap<Line, List<Object3DQuotation>> alignmentGrps = new HashMap<Line, List<Object3DQuotation>>();
	private static boolean draw = false;
	private static Object3DQuotationManager manager = new Object3DQuotationManager();
	
	private Object3DQuotationManager() {
		subscribe(SceneEvent.GENERATE_QUOTS.name(), SceneEvent.CAMERA_MOVED.name(), SceneEvent.CAMERA_MODE_CHANGED.name());
	}

	public void createQuotations() {
			if(objects == null)
				return;
			Vector<Object3DQuotation> tmpQuotsList = new Vector<Object3DQuotation>();
			alignmentGrps.clear();
			sourceGrps.clear();
			sceneSides.clear();
			for(Object3D object: objects) {
				sceneSides.addAll(object.getRotatedBoundingBox().getSides());
			}
			ArrayList<LineSegment> edges = new ArrayList<LineSegment>();
			ArrayList<Vector3> hiddenCorners = new ArrayList<Vector3>();
			HashMap<LineSegment, Object3D> edgesMap = new HashMap<LineSegment, Object3D>(); 
			for(Object3D object: objects) {
				List<LineSegment> segments = object.getRotatedBoundingBox().getEdges();
				for(LineSegment segment: segments) {
					edgesMap.put(segment, object);
				}
				edges.addAll(segments);
			    List<Vector3> corners = object.getRotatedBoundingBox().getCorners();
			    for(Vector3 corner: corners) {
			    	if(isHiddenCorner(corner, sceneSides, camera ))
			    			hiddenCorners.add(corner);
			    }
			}
			
			//remove edges connected to the hidden corner
			filterEdges(edges, hiddenCorners);
			//prepare quotations
			for(LineSegment segment: edges) {
				Object3D source = edgesMap.get(segment);
				Object3DQuotation quots = new Object3DQuotation(segment.getV0().cpy(), segment.getV1().cpy(), source);
				quots.calculateDrawingPoints(camera);
				//remove quots which would be less than 50 pixel in length
				if(quots.getLengthOnScreen() >50)
					tmpQuotsList.add(quots);
			}
			
			groupBySource(tmpQuotsList, sourceGrps);
			groupByLine(tmpQuotsList, alignmentGrps);
			getQuotations().clear();
			//filter same source quots{
			for(List<Object3DQuotation> sourceGrp :this.sourceGrps.values()) {
				List<Object3DQuotation> filteredList = filterSameSourceQuots(sourceGrp);
				getQuotations().addAll(filteredList);
			}
			//reduce equivalents
			//two quotations are equivalents if they have the same length, sae direction but they re not aligned
			boolean stop = false;
			do {
				stop = true;
				for(Object3DQuotation quot: tmpQuotsList) {
					List<Object3DQuotation> equivalents = findEquivalent(quot, tmpQuotsList);
					if(!equivalents.isEmpty()) {
						equivalents.add(quot);
						stop = false;
						Object3DQuotation survivor = null;
						int survivorScore = -1;
						for(Object3DQuotation equivalent: equivalents) {
							int score = getLineGroup(equivalent).size();
							if(score > survivorScore) {
								survivor = equivalent;
								survivorScore = score;
							}
						}
						tmpQuotsList.removeAll(equivalents);
						tmpQuotsList.add(survivor);
						break;
					}
					
				}
				
			}while(!stop);
			getQuotations().clear();
			getQuotations().addAll(tmpQuotsList);
		
		
		
	}
	List<Object3DQuotation> findEquivalent(Object3DQuotation forQuotation, List<Object3DQuotation> searchGroup){
		List<Object3DQuotation> equivalents = new ArrayList<Object3DQuotation>();
		List<Object3DQuotation> sameLength = findQuotationsByLength(forQuotation.getLength(), searchGroup);
		List<Object3DQuotation> sameDir = findQuotationByDir(forQuotation.getDirection(false), sameLength);
		for(Object3DQuotation quot: sameDir) {
			if(quot == forQuotation)
				continue;
			Vector3 translation = quot.getCenter().cpy().sub(forQuotation.getCenter());
			if(Math.abs(translation.dot(forQuotation.getDirection(false))) < 0.001f){
				equivalents.add(quot);
			}
		}
		return equivalents;
	}

	
	private List<Object3DQuotation> findQuotationByDir(Vector3 direction, List<Object3DQuotation> searchGroup) {
		List<Object3DQuotation> list = new ArrayList<Object3DQuotation>();
		for(Object3DQuotation quotX: searchGroup) {
			Vector3 director = quotX.getDirection(false);
			float dotProduct = direction.dot(director);
			if(Math.abs(dotProduct  * dotProduct - director.len2() * direction.len2()) < 0.001f)
				list.add(quotX);
		}
		return list;
	}

	private List<Object3DQuotation> findQuotationsByLength(float length, List<Object3DQuotation> searchGroup) {
		List<Object3DQuotation> list = new ArrayList<Object3DQuotation>();
		int l = Math.round(length * 100);
		for(Object3DQuotation quotX: searchGroup) {
			if( Math.round(quotX.getLength() * 100) == l)
				list.add(quotX);
		}
		return list;
	}

	private List<Object3DQuotation> filterSameSourceQuots(List<Object3DQuotation> sourceGrp) {
		List<Object3DQuotation> filtered = new ArrayList<Object3DQuotation>();
		HashMap<Integer, List<Object3DQuotation>> lengthGrps = groupByLength(sourceGrp, null);
		for(List<Object3DQuotation> lengthGrp: lengthGrps.values()) {
			if(lengthGrp.size() > 1) {
				HashMap<Vector3, List<Object3DQuotation>> directionGrps = groupByDirection(lengthGrp, null);
				for(List<Object3DQuotation> grp: directionGrps.values()) {
					if(grp.size() > 1) {
						Object3DQuotation selected = null;
						int selectedScore = -1;
						for(Object3DQuotation quotX: grp) {
							List<Object3DQuotation> lineGrp = getLineGroup(quotX);
							if(lineGrp == null) {
								System.err.println("no alignment group was found for " + quotX);
								continue;
							}
							int score = lineGrp.size();
							if(selectedScore < score || (selectedScore == score && selected.getLengthOnScreen() < quotX.getLengthOnScreen())) {
								selected = quotX;
								selectedScore = score;
							}
						}
						if(selected != null)
							filtered.add(selected);
						
					}else {
						filtered.add(grp.get(0));
					}
				}
			}else {
				filtered.add(lengthGrp.get(0));
			}
		}
		return filtered;
	}

	private List<Object3DQuotation> getLineGroup(Object3DQuotation quotX) {
		for(List<Object3DQuotation> lineGrpX: this.alignmentGrps.values()) {
			if(lineGrpX.contains(quotX))
				return lineGrpX;
		}
		return null;
	}

	private static HashMap<Line, List<Object3DQuotation>> groupByLine(List<Object3DQuotation> quots, HashMap<Line, List<Object3DQuotation>> out) {
		if(out == null)
			out = new HashMap<Line, List<Object3DQuotation>>();
		for(Object3DQuotation quot: quots) {
			Line line = quot.getLine();
			List<Object3DQuotation> alignment = null;
			for(Line lineX: out.keySet()) {
				if(line.equals(lineX)) {
					alignment = out.get(lineX);
				}
			}
			if(alignment == null) {
				alignment = new ArrayList<Object3DQuotation>();
				out.put(line, alignment);
			}
			alignment.add(quot);
		}
		return out;
	}
	
	public HashMap<Object3D, List<Object3DQuotation>> groupBySource(List<Object3DQuotation> quots, HashMap<Object3D, List<Object3DQuotation>> out) {
		if(out == null)
			out = new HashMap<Object3D, List<Object3DQuotation>>();
		for(Object3DQuotation quot: quots) {
			Object3D source = quot.getSource();
			List<Object3DQuotation> srcGrp = out.get(source);
			if(srcGrp == null) {
				srcGrp = new ArrayList<Object3DQuotation>();
				out.put(source, srcGrp);
			}
			srcGrp.add(quot);
		}
		return out;
	}
	
	public static HashMap<Vector3, List<Object3DQuotation>> groupByDirection(List<Object3DQuotation> quots, HashMap<Vector3, List<Object3DQuotation>> out) {
		if(out == null)
			out = new HashMap<Vector3, List<Object3DQuotation>>();
		List<Object3DQuotation> quotGrp = null;
		for(Object3DQuotation quot: quots) {
			Vector3 direction =  quot.getDirection(false);
			for(Vector3 director: out.keySet()) {
				float dotProduct = direction.dot(director);
				if(Math.abs(dotProduct  * dotProduct - director.len2() * direction.len2()) < 0.001f) {
					quotGrp = out.get(director);
				}
			}
			if(quotGrp == null) {
				quotGrp = new ArrayList<Object3DQuotation>();
				out.put(direction, quotGrp);
			}
			quotGrp.add(quot);
		}
		return out;
	}
	
	public static HashMap<Integer, List<Object3DQuotation>> groupByLength(List<Object3DQuotation> quots, HashMap<Integer, List<Object3DQuotation>> out ){
		if(out == null)
			out = new HashMap<Integer, List<Object3DQuotation>>();
		List<Object3DQuotation> quotGrp = null;
		for(Object3DQuotation quot: quots) {
			int len =  Math.round(quot.getLength() * 100);
			quotGrp = out.get(len);
			if(quotGrp == null) {
				quotGrp = new ArrayList<Object3DQuotation>();
				out.put(len, quotGrp);
			}
			quotGrp.add(quot);
		}
		return out;
	}
	


	public boolean isHiddenCorner(Vector3 corner, List<Quad> faces, Camera camera){
		boolean hidden = false;
	    	Ray3D ray = new Ray3D(corner, camera.position.cpy().sub(corner).nor());
	    	for(Quad quad: sceneSides) {
	    		if(quad.hasVertex(corner, 0.01f))
	    			continue;
	    		Vector3 intersection =  ray.intersect(quad);
	    		if(intersection != null && quad.contains(intersection, 0.01f)) {
		    		hidden = true;
		    		break;
	    		}
	    	}
		return hidden;
	}
	
	public void filterEdges(List<LineSegment> edges, List<Vector3> exclusions) {
		//remove repeatedEdges
		boolean finished;
		do {
			finished = true;
			for(int i = 0; i < edges.size() - 1; i++) {
				LineSegment current = edges.get(i);
				for(int j = i+1; j < edges.size(); j++) {
					LineSegment comparable = edges.get(j);
					if(Math.abs(current.getLen2() - comparable.getLen2()) > 0.01f)
						continue;
					if(!current.getDir(false).isOnLine(comparable.getDir(false), 0.01f))
						continue;
					if(current.getCenter().epsilonEquals(comparable.getCenter(), 0.01f)) {
						edges.remove(current);
						finished = false;
						break;
					}
				}
			}
		}while(!finished);
		//remove edges connected to the hidden corner
		boolean stop;
		do {
			stop = true;
			for(Vector3 hiddenCorner: exclusions) {
				for(LineSegment lineSegment: edges) {
					if(lineSegment.startOrEndWith(hiddenCorner, 0.01f)) {
						edges.remove(lineSegment);
						stop = false;
						break;
					}
				}
				if(!stop)
					break;
			}
			
		}while(!stop);
	}
	
	public void draw(Batch batch) {
//		if(!draw)
//			return;
		try {
			
	        	if(shapeRenderer == null)
					shapeRenderer = new ShapeRenderer();
	        	for(Object3DQuotation quot: getQuotations()) {

					quot.draw(batch, shapeRenderer);
				}			
		}catch (Exception e) {
			e.printStackTrace();
		}
	}
	


	



	public synchronized Vector<Object3DQuotation> getQuotations() {
		return quotations;
	}

	@Override
	public void handle(Event event) {
		SceneEvent eventType = SceneEvent.valueOf(event.getTopic());
		switch (eventType) {
		case GENERATE_QUOTS:
			handleGenerateQuotsEvent(event);
			break;
		case CAMERA_MOVED:
			handleCameraMoved();
			break;
		case CAMERA_MODE_CHANGED:
			cameraModeChanged();
			break;
		}
		
	}

	private void cameraModeChanged() {
		if(draw) {
			camera = Room3DController.getInstance().getCameraController().getCurrentCamera();
			createQuotations();
		}
	}

	private void handleCameraMoved() {
		createQuotations();
	}

	private void handleGenerateQuotsEvent(Event event) {
		try{
			Map<String, Object> args = (Map<String, Object>) event.getData();
			Camera camera = (Camera) args.get("camera");
			List<Object3D> sources = (List<Object3D>) args.get("sources");
			this.objects = sources;
			this.camera = camera;
			createQuotations();
		}catch (Exception e) {
			System.err.println("failed to handle event " + event.toString());
			e.printStackTrace();
		}		
	}

	public static Object3DQuotationManager getManager() {
		return manager;
	}

	public static void setManager(Object3DQuotationManager manager) {
		Object3DQuotationManager.manager = manager;
	}
	
}
