package dressing.controller;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.time.LocalDateTime;
import java.util.UUID;

import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import dressing.config.UserPreference;
import dressing.model.DesignException;
import dressing.model.Kitchen;
import dressing.model.Materiaux;
import dressing.model.ModelProvider;
import dressing.model.ProjectManager;
import dressing.model.SuperCadProject;
import dressing.model.SuperCadProject.ProjectType;
import dressing.model.evalutor.GeometricEngineException;
import gdxapp.Commun.Preferences;
import gdxapp.object3d.DoorHandle;
import gdxapp.object3d.Wall;
import gdxapp.scenes.ScenePreferences;
import gdxapp.screens.room.RoomController;
import geometry.CompoundShape;
import geometry.Polygon;

public class ProjectCreator implements Creator {

	private String kitchenname = "Cuisine";
	private Materiaux basematerial = ModelProvider.getMateriallist() != null
			&& ModelProvider.getMateriallist().get(0) != null ? ModelProvider.getMateriallist().get(0) : null;
	private Materiaux backmaterial = ModelProvider.getMateriallist() != null
			&& ModelProvider.getMateriallist().get(ModelProvider.getMateriallist().size() - 1) != null
					? ModelProvider.getMateriallist().get(ModelProvider.getMateriallist().size() - 1)
					: null;
	private Materiaux facematerial = ModelProvider.getMateriallist() != null
			&& ModelProvider.getMateriallist().get(1) != null ? ModelProvider.getMateriallist().get(1) : null;

	public enum CornerType {
		Rectancle, Angle, None {
			@Override
			public String toString() {

				return "Aucun";
			}
		}
	}

	private SuperCadProject project = new SuperCadProject();
	private String client;
	private String clientNumber;
	private String site=UserPreference.getPreference().getProperties().getProperty("site", "");;
	private String commercial =UserPreference.getPreference().getProperties().getProperty("commercial", "");
	private String projectName;
	private ProjectType type = ProjectType.KITCHEN;
	private Kitchen cuisine;
	private final KitchenShapeBuilder kitchenShapeBuilder = new KitchenShapeBuilder();
	private ScenePreferences preferences = new ScenePreferences();
	private DoorHandle doorHandle;
	private Wall kitchenWall;
	public static ProjectCreator instance = null;

	public static ProjectCreator getInstance() {
		if (instance == null) {
			instance = new ProjectCreator();
		}
		return instance;
	}

	public Materiaux getBasematerial() {
		return basematerial;
	}

	public void setBasematerial(Materiaux basematerial) {

		Materiaux oldValue = this.basematerial;
		this.basematerial = basematerial;
		firePropertyChange("basematerial", oldValue, this.basematerial);
	}

	public Materiaux getBackmaterial() {
		return backmaterial;
	}

	public void setBackmaterial(Materiaux backmaterial) {

		Materiaux oldValue = this.backmaterial;
		this.backmaterial = backmaterial;
		firePropertyChange("backmaterial", oldValue, this.backmaterial);
	}

	public Materiaux getFacematerial() {
		return facematerial;
	}

	public void setFacematerial(Materiaux facematerial) {
		Materiaux oldValue = this.facematerial;
		this.facematerial = facematerial;
		firePropertyChange("facematerial", oldValue, this.facematerial);
	}

	public String getKitchenname() {
		return kitchenname;
	}

	public void setKitchenname(String kitchenname) {
		String oldValue = this.kitchenname;
		this.kitchenname = kitchenname;
		firePropertyChange("kitchenname", oldValue, kitchenname);
	}

	public SuperCadProject getProject() {
		return project;
	}

	public void setProject(SuperCadProject project) {
		SuperCadProject oldValue = this.project;
		this.project.set(project);
		firePropertyChange("project", oldValue, project);
	}

	public String getClient() {
		return client;
	}

	public void setClient(String client) {
		String oldValue = this.client;
		this.client = client;
		firePropertyChange("client", oldValue, client);

	}

	public String getSite() {
		return site;
	}

	public void setSite(String site) {
		String oldValue = this.site;
		this.site = site;
		firePropertyChange("site", oldValue, site);
	}

	public String getCommercial() {
		return commercial;
	}

	public void setCommercial(String commercial) {
		String oldValue = this.commercial;
		this.commercial = commercial;
		firePropertyChange("commercial", oldValue, commercial);
	}

	public String getProjectName() {
		return projectName;
	}

	public void setProjectName(String projectName) {
		String oldValue = this.projectName;
		this.projectName = projectName;
		firePropertyChange("projectName", oldValue, projectName);
	}

	public ProjectType getType() {
		return type;
	}

	public void setType(ProjectType type) {
		ProjectType oldValue = this.type;
		this.type = type;
		firePropertyChange("type", oldValue, type);
	}

	public Kitchen getCuisine() {
		return cuisine;
	}

	public void setCuisine(Kitchen cuisine) {
		Kitchen oldValue = this.cuisine;
		this.cuisine = cuisine;
		firePropertyChange("cuisine", oldValue, cuisine);
	}














	public ScenePreferences getPreferences() {
		return preferences;
	}

	public void setPreferences(ScenePreferences preferences) {
		ScenePreferences oldValue = this.preferences;
		this.preferences = preferences;
		firePropertyChange("preferences", oldValue, preferences);
	}

	public PropertyChangeSupport getPropertyChangeSupport() {
		return propertyChangeSupport;
	}

	public void setPropertyChangeSupport(PropertyChangeSupport propertyChangeSupport) {
		this.propertyChangeSupport = propertyChangeSupport;
	}

	transient protected PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);

	public void addPropertyChangeListener(PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(listener);
	}

	public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
	}

	public void removePropertyChangeListener(PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(listener);
	}

	public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(propertyName, listener);
	}

	protected void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
		propertyChangeSupport.firePropertyChange(propertyName, oldValue, newValue);
	}

	public SuperCadProject getProjectInstance() {
		return project;
	}

	public void check() throws DesignException {

		checkProject();
		checkKitchen();
	}

	public void checkProject() throws DesignException {
		
		String regex = "^[a-zA-Z0-9]+(?:\\s[a-zA-Z0-9]+)*$";
		String phoneNumberRegx = "^(\\+\\d{1,4})?\\s?[0-9\\s-]+$";
		String projectName = getProjectName();
		String clientNumber = getClientNumber();
		
		if (projectName == null || getProjectName().isEmpty())
			throw new DesignException("Le nom de projet est obligatoire");
		if(!projectName.matches(regex))
			throw new DesignException("Les caractères spéciaux et les espaces en fin de ligne sont interdits.");
		if (clientNumber == null || clientNumber.isEmpty())
			throw new DesignException("Le numéro de téléphone du client est obligatoire");
		if(!clientNumber.matches(phoneNumberRegx))
			throw new DesignException("Numéro de téléphone invalide!");
		
		if(ProjectManager.getManager().getProjects()!=null) {
			for(SuperCadProject proj:ProjectManager.getManager().getProjects()) {
				if(getProjectName()!=null && getProjectName().contentEquals(proj.getName())) {
					throw new DesignException("Ce nom de projet est déjà utilisé.");
				}
			}
		}
	}

	public void checkKitchen() throws DesignException {

		if (getKitchenname() == null || getKitchenname().isEmpty())
			throw new DesignException("Le nom de cuisine est obligatoire");

		if (getBasematerial() == null)
			throw new DesignException("Le matériau de base doit être renseigné");
		if (getFacematerial() == null)
			throw new DesignException("Le matériau de façade doit être renseigné");
		if (getBackmaterial() == null)
			throw new DesignException("Le matériau de dos doit être renseigné");

		if (getBasematerial().getEpaisseur() <= 0)
			throw new DesignException("L'épaisseur de matériaux de base doit être superieur à 1 mm");

		if (getBackmaterial().getEpaisseur() <= 0)
			throw new DesignException("L'épaisseur de matériaux de dos doit être superieur à 1 mm");

	}

	public void create() throws DesignException, GeometricEngineException {
		check(); // On met le controle dans une seule fonction
		this.project=new SuperCadProject();
		this.project.setName(getProjectName());
		this.project.setClient(getClient());
		this.project.setId(UUID.randomUUID());
		this.project.setCommercial(getCommercial());
		this.project.setSite(getSite());
		this.project.setType(getType());
		this.project.setCreationDate(LocalDateTime.now());
		createSubElements();
		this.project.setLoaded(true);
	}

	public void createSubElements() throws DesignException, GeometricEngineException {
		
		this.project.setCurrentKitchen(null);
		
		this.cuisine = new Kitchen(this.project);
		this.cuisine.setWidth(kitchenShapeBuilder.getWidth());
		this.cuisine.setHeight(kitchenShapeBuilder.getHeight() );
		this.cuisine.setBackMaterial(getBackmaterial());
		this.cuisine.setPrimaryMaterial(getBasematerial());
		this.cuisine.setFaceMaterial(getFacematerial());
		this.cuisine.setName(getKitchenname());
		this.cuisine.setUuid(UUID.randomUUID());
		this.cuisine.getScene().setPreferences(preferences);
		this.project.addElement(this.cuisine);
		UserPreference.getPreference().setScenePreferences(getPreferences().clone());
		UserPreference.getPreference().getProperties().put("site", this.site);
		UserPreference.getPreference().getProperties().put("commercial", this.commercial);

		createKitchenRoom();
	}
	
	
	//to redo
	public void createKitchenRoom() {
		var polies = kitchenShapeBuilder.createWalls();
		
		if(!polies.isEmpty()) {
			CompoundShape shape = new CompoundShape();
			//transform vertices to screen space
			float[] transform = new float[] { 1, 0, 0   ,0, 0, 1,  0, 1, 0};
			Matrix3 worldTransform = new Matrix3(transform);
			Matrix4 screenTransform = RoomController.getInstance().getWorldToSceneTransform();
			Vector3 translation = new Vector3(Preferences.WORLD_WIDTH /2.0f, 0, Preferences.WORLD_HEIGHT /2.f);
			for(Polygon poly: polies) {
				poly.removeDuplicateVertices();
				for(Vector2 vertex: poly.getVertices()) {
					Vector3 screenCoords = new Vector3(vertex, 0).mul(worldTransform).add(translation).mul(screenTransform);
					vertex.set(screenCoords.x, screenCoords.y);
				}
				poly.setWinding(1);
				shape.addNode(poly);
			}
			this.kitchenWall = new Wall();
			this.kitchenWall.setPerimeter(shape);
			createWall(shape);
		}
	}

	public void createWall(CompoundShape compound) {
		ScenePreferences preferences = ProjectManager.getManager().getCurrentScene().getPreferences();
		Wall wallX = new Wall(preferences.getWallHeight(), preferences.getWallThickness());
		wallX.setPerimeter(compound);
		wallX.setMaterial(this.preferences.getWallMtl());
		this.cuisine.getScene().addActor(wallX, false);
	}

	public DoorHandle getDoorHandle() {
		return doorHandle;
	}

	public void setDoorHandle(DoorHandle doorHandle) {
		this.doorHandle = doorHandle;
	}

	public String getClientNumber() {
		return clientNumber;
	}

	public void setClientNumber(String clientNumber) {
		String oldValue = this.clientNumber;
		this.clientNumber = clientNumber;
		firePropertyChange("clientNumber", oldValue, clientNumber);
		System.out.println("client number changed to: " + clientNumber);
	}

	public KitchenShapeBuilder getKitchenShapeBuilder() {
		return kitchenShapeBuilder;
	}	
	
}
