package dressing.model.persistence.mappers;

import java.lang.reflect.Field;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;

import org.eclipse.core.databinding.observable.value.LocalDateTimeObservableValue;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Quaternion;
import dressing.model.persistence.dpos.ColorDPO;
import dressing.model.persistence.dpos.DPO;
import dressing.model.persistence.dpos.Matrix4DPO;
import supercad.persistence.databinding.BooleanValueMapper;
import supercad.persistence.databinding.CustomValueMapper;
import supercad.persistence.databinding.DateTimeValueMapper;
import supercad.persistence.databinding.DoubleValueMapper;
import supercad.persistence.databinding.EnumMapper;
import supercad.persistence.databinding.FloatValueMapper;
import supercad.persistence.databinding.IntegerValueMapper;
import supercad.persistence.databinding.NaiveValueMapper;
import supercad.persistence.databinding.UUIDValueMapper;
import supercad.persistence.databinding.XmlMapper;
import supercad.runtime.AnnotationScanner;
import supercad.runtime.ReflectionUtils;

public class MapperProvider {

	private Map<Class, Class<? extends DPO>> conversionMap = new HashMap<Class, Class<? extends DPO>>();
	private Map<Class, Field> reductionMap = new HashMap<Class, Field>();
	private Map<Class, CustomValueMapper> valueMappers = new HashMap<Class, CustomValueMapper>();
	// classes that are transient
	private ArrayList<Class> forbiddenClasses = new ArrayList<Class>();
	private boolean ready = false;
	
	private static MapperProvider instance;

	private MapperProvider() {
		registerMappersAndModule();
	}
	
	public void registerMappersAndModule() {
		if (!ready) {
			scanAnnotation();
			conversionMap.put(Color.class, ColorDPO.class);
			conversionMap.put(Matrix4.class, Matrix4DPO.class);
			registerCustomValueMappers();
			// add transient classes
			forbiddenClasses.add(Quaternion.class);
			ready = true;
		}
	}

	private void registerCustomValueMappers() {
		valueMappers.put(int.class, new IntegerValueMapper());
		valueMappers.put(Integer.class, new IntegerValueMapper());
		valueMappers.put(float.class, new FloatValueMapper());
		valueMappers.put(Float.class, new FloatValueMapper());
		valueMappers.put(double.class, new DoubleValueMapper());
		valueMappers.put(Double.class, new DoubleValueMapper());
		valueMappers.put(boolean.class, new BooleanValueMapper());
		valueMappers.put(Boolean.class, new BooleanValueMapper());
		valueMappers.put(UUID.class, new UUIDValueMapper());
		valueMappers.put(LocalDateTime.class, new DateTimeValueMapper());
		valueMappers.put(Enum.class, new EnumMapper());
	}

	private void scanAnnotation() {
		// scanning for @persistable
		ArrayList<Class> persistables = AnnotationScanner.findClassesWithAnnotations(Persistable.class, "dressing",
				"gdxapp");
		for (Class clazz : persistables) {
			Persistable persistable = (Persistable) clazz.getAnnotation(Persistable.class);
			conversionMap.put(clazz, persistable.persistableForm());
		}
		// scanning for @PersistanceValue
		for (Class clazz : ReflectionUtils.getSrcClasses()) {
			ArrayList<Field> persistenceFields = AnnotationScanner.findAnnotatedFields(PersistenceValue.class, clazz);
			if (!persistenceFields.isEmpty()) {
				reductionMap.put(clazz, persistenceFields.get(0));
			}
		}
	}

	public Class<? extends DPO> getPersistableType(Class type) {
		if(type == null)
			return null;
		registerMappersAndModule();
		Class<? extends DPO> persistenceType = conversionMap.get(type);
		if (persistenceType == null && type.isAnnotationPresent(Persistable.class)) {
			persistenceType = ((Persistable) type.getAnnotation(Persistable.class)).persistableForm();
		}
		return persistenceType;
	}
	
	public Class getRuntimeType(Class<? extends DPO> persistenceForm) {
		Class runtimeClass = persistenceForm;
		for(Entry<Class, Class<? extends DPO>> entry: conversionMap.entrySet()) {
			if(entry.getValue() == persistenceForm) {
				runtimeClass = entry.getKey();
			}
		}
		return runtimeClass;
	}

	public static XmlMapper getXmlMapper() {
		return new XmlMapper();
	}

	public CustomValueMapper getValueTypeMapper(Class<? extends Object> clazz) {
		registerCustomValueMappers();
		CustomValueMapper mapper = valueMappers.get(clazz);
		if(mapper == null) {
			Class assignable = null;
			for(Class type: valueMappers.keySet()) {
				if(type.isAssignableFrom(clazz)){
					assignable = type;
					break;
				}
			}
			if(assignable != null)
				mapper = valueMappers.get(assignable);
		}
		return mapper == null?NaiveValueMapper.getInstance():mapper;
	}

	public static MapperProvider getInstance() {
		if(instance == null)
			instance = new MapperProvider();
		return instance;
	}

}
