package geometry;

import java.util.ArrayList;

import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import dressing.mathutils.MathUtilities;
import dressing.mathutils.Triangle;
import jakarta.xml.bind.annotation.XmlRootElement;

@XmlRootElement(name="compound_object")
public class CompoundObject {
	
	private CompoundShape border;
	private CompoundShape extrusion;
	private boolean perfored;
	private Matrix4 worldTransform;
	private transient ArrayList<Triangle> trianglesMesh;
	
	public CompoundObject() {
		super();
	}

	public CompoundObject(CompoundShape border, CompoundShape extrusion) {
		this.border = border;
		this.extrusion = extrusion;
	}

	public ArrayList<Triangle> triangulate(){
		trianglesMesh =  EarClipper.triangulate(border.getVertices(), (extrusion==null)?new ArrayList<Vector2>():extrusion.getVertices());
		return getTrianglesMesh();
	}
	
	public CompoundShape getBorder() {
		return border;
	}
	public void setBorder(CompoundShape border) {
		border.setWinding(GL20.GL_CCW);
		this.border = border;
	}
	public CompoundShape getExtrusion() {
		return extrusion;
	}
	public void setExtrusion(CompoundShape extrusion) {
		extrusion.setWinding(GL20.GL_CW);
		this.extrusion = extrusion;
	}
	
	public Matrix4 getWorldTransform() {
		return worldTransform;
	}

	public void setWorldTransform(Matrix4 worldTransform) {
		this.worldTransform = worldTransform;
	}
	
	public boolean hasHole() {
		return this.extrusion!=null;
	}
	
	public ArrayList<Triangle> getTrianglesMesh() {
		if(trianglesMesh == null) {
			triangulate();
		}
		return trianglesMesh;
	}

	public void setTrianglesMesh(ArrayList<Triangle> trianglesMesh) {
		this.trianglesMesh = trianglesMesh;
	}

	public Vector3 getRealWorldDimension(float depth) {
		ArrayList<Vector3> verticesInRealWorld = new ArrayList<Vector3>();
		for(int i = 0; i < border.getVertices().size(); i++) {
			verticesInRealWorld.add(new Vector3(border.getVertices().get(i),depth/2.0f).mul(worldTransform));
			verticesInRealWorld.add(new Vector3(border.getVertices().get(i),-depth/2.0f).mul(worldTransform));
		}
		Vector3[] boundaries = MathUtilities.getBoundaries(verticesInRealWorld);
		Vector3 dimension = boundaries[1].sub(boundaries[0]);
		return dimension;
	}
	
	public  ArrayList<Vector2> getExtrusionVertices() {
		if(hasHole())
		{	
			float rotation=-1;
			ArrayList<Vector2> vertices =extrusion.getVertices();
			if(vertices!=null && vertices.size()>2) {
				Vector2 vp = vertices.get(0).cpy().sub(vertices.get(1));
        		Vector2 vn = vertices.get(2).cpy().sub(vertices.get(1));
        		rotation = (float) Math.toDegrees(MathUtilities.signedAngle(vp, vn));
        		System.err.println(rotation);
        		if(rotation>0) {
        			ArrayList<Vector2> vertices1=new ArrayList<Vector2>();
        			for(int i=vertices.size()-1;i>=0;i--) {
        				vertices1.add(vertices.get(i));
        			}
        			vertices=vertices1;
        		}
			}
			
			return vertices;
		}
		return new ArrayList<Vector2>();
	}

	public CompoundObject cpy() {
		CompoundObject clone = new CompoundObject();
		clone.setBorder(border.cpy());
		if(hasHole()) {
			clone.setExtrusion(extrusion.cpy());
		}
		clone.setWorldTransform(worldTransform.cpy());
		clone.setPerfored(perfored);
		return clone;
	}

	public void clear() {
		border = null;
		extrusion = null;
		
	}

	public boolean isValid() {
		if(border.getVertices().size() < 3)
			return false;
		if(hasHole() && extrusion.getVertices().size() < 3)
			return false;
		return true;
	}

	public boolean isPerfored() {
		return perfored;
	}

	public void setPerfored(boolean perfored) {
		this.perfored = perfored;
	}
	
}
