package api.mep;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import org.jogamp.java3d.Jogl2es2Context.GL_State;
import org.joml.Matrix4f;
import org.joml.Vector2i;
import org.joml.Vector3f;
import org.joml.Vector4f;
import org.lwjgl.opengl.GL45;

import api.backend.ApplicationContext;
import api.event.EventBus;
import api.event.EventHandler;
import api.event.EventType;
import api.event.MepEvent;
import api.graphics.Camera;
import api.graphics.IDrawable;
import api.graphics.Model;
import api.graphics.ModelInstance;
import api.graphics.ObjectController;
import api.graphics.ObjectPositioner;
import api.graphics.ObjectRotator;
import api.graphics.Scene;
import api.graphics.Shader;
import api.graphics.geometry.Triangle;
import api.input.InputMultiplexer;
import api.provider.ModelProvider;
import api.ui.CameraController;
import api.ui.RightClickMenuManager;
import api.utils.IOUtils;
import api.utils.MathUtilities;
import api.utils.ScreenUtils;

public class EditorScene extends Scene implements EventHandler {
	
	private final ArrayList<ModelInstance> objects = new ArrayList<ModelInstance>();
	private final ArrayList<ModelInstance> hiddenElement = new ArrayList<ModelInstance>();
	private QuotationDrawer quotsDrawer;
	private int mode;
	
	public EditorScene() {
		this.init();
	}

	@Override
	public void init() {
		super.init();
		InputMultiplexer.getInstance().pushEventHandler(cameraController);
		ObjectPositioner.getPositioner().setCamera(camera);
		ObjectRotator.getRotator().setCamera(camera);
		ObjectController.getController().setCamera(camera);
		ObjectSelector.getSelector().init();
		ObjectSelector.getSelector().setCamera(camera);
		ObjectSelector.getSelector().setObjects(objects);
		SurfaceController.getController();
		addDrawable(ObjectSelector.getSelector());
		addDrawable(PipeDrawer.getDrawer());
		
		
		InputMultiplexer.getInstance().pushEventHandler(ObjectSelector.getSelector());
		InputMultiplexer.getInstance().pushEventHandler(RightClickMenuManager.getManager());

		subscribe(EventType.INVOKE_PIPE_DRAWER);
		subscribe(EventType.OBJECT_SELECTION_TRIGFERED);
		subscribe(EventType.DROP_OBJECT);
		subscribe(EventType.ADD_RENDERABLE);
		subscribe(EventType.ORTHO_PROJECTION);
		subscribe(EventType.HIDE_ELEMENT);
		this.quotsDrawer = new QuotationDrawer(objects);
	}
	
	public void loadScene(String path) {
		File folder = new File(path);
		File[] modelFiles = folder.listFiles(new FileFilter() {
			@Override
			public boolean accept(File pathname) {
				return pathname.getName().endsWith(".obj");
			}
		});
		
		for(File modelFile: modelFiles) {
			Model model = new Model(modelFile.getAbsolutePath());
			ModelInstance instance = new ModelInstance(model);
			//load transform
			String transformFilePath = modelFile.getPath().replace(".obj", ".txt");
			var transform = IOUtils.readTransformFile(transformFilePath);
			instance.setTransform(transform);
			addDrawable(instance);
		}
	}
	

	@Override
	public void handle(MepEvent event) {
		EventType type = event.getType();
		
		switch (type) {
		case INVOKE_PIPE_DRAWER:
			PIPE_TYPE pipeType = (PIPE_TYPE) event.getDetail("Type");
			invokePipeDrawer(pipeType);
			break;
		case OBJECT_SELECTION_TRIGFERED:
			InputMultiplexer.getInstance().removeEventHandler(PipeDrawer.getDrawer());
			break;
		case DROP_OBJECT:
			dropObject(event);
			break;
		case ADD_RENDERABLE:
			var renderable = event.getDetail("renderable");
			addDrawable((IDrawable) renderable);
			break;
		case ORTHO_PROJECTION:
			List<Triangle> target;
			try {
				 target = (List<Triangle>) event.getDetail("ortho");
			}catch (ClassCastException e) {
				break;
			}
			setToOrthoAgainst(target);
			break;
		case HIDE_ELEMENT:
			List<ModelInstance> elements =  (List<ModelInstance>) event.getDetail("elements");
			hideDrawable(elements);
			break;
		case REMOVE_DRAWABLE:
			IDrawable drawable = (IDrawable) event.getDetail("drawable");
			if(drawable != null) {
				removeDrawable(drawable);
			}
			break;
		default:
			throw new IllegalArgumentException("Unexpected value: " + type);
		}

	}



	private void hideDrawable(List<ModelInstance> elements) {
		for(ModelInstance element: elements) {
			this.hiddenElement.add(element);
			this.drawables.remove(element);
			this.objects.remove(element);
		}
	}

	private void dropObject(MepEvent event) {
		try {
			String modelName = (String) event.getDetail("model name");
			Vector2i location = (Vector2i) event.getDetail("location");
			var info = ModelProvider.getProvider().findModelByName(modelName);
			float z = ScreenUtils.readDepth(location.x, location.y);
			float y = ApplicationContext.getHeight() - location.y;
			Vector3f position =  camera.unproject(new Vector3f(location.x, y, z), ApplicationContext.getViewport());
			Triangle face = ObjectSelector.getSelector().selectFace(position);
			float rotation = 0;
			Vector3f normal = new Vector3f(0,0,1);
			if(face != null) {
				Vector4f normal4 =  new Vector4f(face.getNormal(), 0).mul(face.transform).normalize();
				normal.set(normal4.x, normal4.y, normal4.z);
				Vector3f director = new Vector3f(normal).rotateAxis((float) Math.PI/2.0f, 0,1,0);
				rotation = (float) Math.atan2(director.z, director.x);
			}
			position.add(normal.mul(info.getScale().z/2.0f));
			Matrix4f mat = new Matrix4f().translate(position).rotate(-rotation, new Vector3f(0,1,0));
			ModelInstance instance = ModelProvider.getProvider().createInstanceOf(modelName);
			instance.transform(mat);
			System.out.println("drop location :" + position);
			addDrawable(instance);
		}catch (Exception e) {
			e.printStackTrace();
		}
		
	}
	
	

	@Override
	public void render(long delta) {
		GL45.glPolygonMode(GL45.GL_FRONT_AND_BACK, mode);
		super.render(delta);
//		if(quotsDrawer != null)
//			quotsDrawer.draw(camera);
		GL45.glPolygonMode(GL45.GL_FRONT_AND_BACK, GL45.GL_FILL);

	}

	private void invokePipeDrawer(PIPE_TYPE pipeType) {
		PipeDrawer.getDrawer().begin(camera, pipeType);
		InputMultiplexer.getInstance().insertEventHandler(PipeDrawer.getDrawer(), 0);
		InputMultiplexer.getInstance().removeEventHandler(ObjectSelector.getSelector());
		//drawables.remove(ObjectSelector.getSelector());
	}

	@Override
	public void addDrawable(IDrawable drawable) {
		this.drawables.add(drawable);
		this.drawables.sort(new Comparator<IDrawable>() {

			@Override
			public int compare(IDrawable o1, IDrawable o2) {
				if(o1 instanceof ModelInstance && !(o2 instanceof ModelInstance))
					return -1;
				if(!(o1 instanceof ModelInstance) && o2 instanceof ModelInstance)
					return 1;
				
				// both re model instances
				if(o1 instanceof ModelInstance && o2 instanceof ModelInstance) {
					float d1 = getCamera().getPosition().distanceSquared(((ModelInstance) o1).getCenter());
					float d2 = getCamera().getPosition().distanceSquared(((ModelInstance) o2).getCenter());
					return (int) Math.signum(d1 - d2);
				}
				return 0;
			}
		});
		if(drawable instanceof ModelInstance) {
			objects.add((ModelInstance) drawable);
			HashMap<String, Object> details = new HashMap<String, Object>();
			details.put("object", drawable);
			EventBus.getInstance().notify(new MepEvent(EventType.OBJECT_ADDED, details));
			if(quotsDrawer != null)
				quotsDrawer.generateQuoations();
		}
	}
	private void removeDrawable(IDrawable drawable) {
		this.drawables.remove(drawable);
	}
	
	public void setToOrthoAgainst(List<Triangle> target) {
		ArrayList<Vector3f> vertices = new ArrayList<Vector3f>();
		for(Triangle face: target) {
			vertices.addAll(Arrays.asList(face.getCorners()));
		}
		Vector3f[] boundaries =  MathUtilities.getBoundaries(vertices.toArray(new Vector3f[0]));
		Vector3f halfSize = new Vector3f(boundaries[1]).sub(boundaries[0]).mul(0.5f);
		
		Vector3f B = new Vector3f(0,1,0);
		
		Vector3f center = new Vector3f(boundaries[0]).add(boundaries[1]).mul(0.5f);
		Vector3f normal = target.get(0).getNormal().normalize();
		Vector3f T = new Vector3f();
		B.cross(normal, T);
		
		Matrix4f transform = new Matrix4f(new Vector4f(T,0), new Vector4f(B, 0), 
				new Vector4f(normal, 0), new Vector4f(center,1));
		ObjectController.getController().setSurfaceTransform(transform.invert());
		Vector4f low =  new Vector4f(boundaries[0], 1);
		Vector4f max =  new Vector4f(boundaries[1], 1);
		Vector3f lowerBound = new Vector3f(low.x, low.y, low.z);
		Vector3f upperBound = new Vector3f(max.x, max.y, max.z);
		HashMap<String, Object> details = new HashMap<String, Object>();
		details.put("transform", transform);
		details.put("lower bound", lowerBound);
		details.put("upper bound", upperBound);
		details.put("objects", this.objects);
		details.put("camera", camera);
		EventBus.getInstance().notify(new MepEvent(EventType.SURFACE_MODE, details));
		Vector3f cameraPosition = new Vector3f(center).add(normal);
		camera.setPosition(cameraPosition);
		camera.lookAt(center);
		camera.setOrthographic(true);
		cameraController.setFreeze(true);
		addDrawable(SurfaceController.getController());

	}

	public int getMode() {
		return mode;
	}

	public void setMode(int mode) {
		this.mode = mode;
	}
	

	
	
}
