package api.mep;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.joml.Matrix4f;
import org.joml.Vector3f;
import org.lwjgl.opengl.GL11;
import org.lwjgl.opengl.GL45;

import api.backend.ApplicationContext;
import api.event.EventHandler;
import api.event.EventType;
import api.event.MepEvent;
import api.graphics.Camera;
import api.graphics.IDrawable;
import api.graphics.IObjectController;
import api.graphics.Mesh;
import api.graphics.ModelInstance;
import api.graphics.ObjectController;
import api.graphics.ObjectController.OPERATION;
import api.graphics.ObjectPositioner;
import api.graphics.ObjectRotator;
import api.graphics.PbrMaterial;
import api.graphics.SelectionMode;
import api.graphics.Shader;
import api.graphics.geometry.ShapeDrawer;
import api.graphics.geometry.Triangle;
import api.input.InputMultiplexer;
import api.input.MouseEventHandler;
import api.utils.ScreenUtils;

public class ObjectSelector implements MouseEventHandler, IDrawable, EventHandler{
	
	private Camera camera;
	private ArrayList<ModelInstance> objects;
	private Object selection;
	private PbrMaterial selectionMaterial;
	private ShapeDrawer shapeDrawer;	
	private List<Triangle> selectedFaces = new ArrayList<Triangle>();
	private Mesh selectedPart;
	private ModelInstance selectedInstance;
	private static ObjectSelector selector;
	private IObjectController objectController;
	private SelectionMode mode = SelectionMode.OBJECTS; 
	

	private ObjectSelector() {}
	
	public void init() {
		selectionMaterial = new PbrMaterial();
		selectionMaterial.setAlbedo(new Vector3f(1,0,0));
		selectionMaterial.setMetalness(0.04f);
		selectionMaterial.setRoughness(0.9f);
		selectionMaterial.setOpacity(0.5f);
		shapeDrawer = new ShapeDrawer();
		shapeDrawer.prepare();
		subscribe(EventType.OBJECT_POSITIONER_TRIGFERED);
		subscribe(EventType.OBJECT_ROTATOR_TRIGGERED);
		subscribe(EventType.SELECTION_MODE_CHANGED);
		setController(ObjectPositioner.getPositioner());
	}
	
	
	@Override
	public boolean mouseDoubleClick(MouseEvent e) {
//		if(selection != null && selectedFace != null) {
//			Vector3f normal = selectedFace.getTransformedNormal();
//			var target = selection.getCenter();
//			float cameraDistance = camera.getPosition().distance(target);
//			camera.setPosition(new Vector3f(target).add(normal.mul(cameraDistance)));
//			camera.lookAt(target);
//			camera.setToOrthoAgainst(selection);
//			//camera.setToOrthoAgainst(selection);
//		}
		return false;
	}
	
	@Override
	public boolean mouseDown(MouseEvent e) {
		if(e.button == 1) {
			float z = ScreenUtils.readDepth(e.x, e.y);
			float y = ApplicationContext.getHeight() - e.y;
			Vector3f worldPoint =  camera.unproject(new Vector3f(e.x, y, z), ApplicationContext.getViewport());
			select(worldPoint);
		}
		return false;
	}

	private void select(Vector3f worldPoint) {
		switch (mode) {
		case VERTICES:
			selectVertex(worldPoint);
			break;
		case EDGES:
			selectEdge(worldPoint);
			break;
		case FACES:
			selectFace(worldPoint);
			break;
		case PARTS:
			selectPart(worldPoint);
			break;
		default:
			selectObject(worldPoint);
			break;
		}
		
	}
	
	private void selectObject(Vector3f worldPoint) {
		resetSelection();
		Triangle selectedFace = selectFace(worldPoint);
		if(selectedFace != null) {
			selection = selectedInstance;
			//ObjectController.getController().attachTo(selectedInstance, OPERATION.TRANSLATE);
		}
		
	}

	private void selectEdge(Vector3f worldPoint) {
		resetSelection();
		resetSelection();
		for(ModelInstance instance: objects) {
			if(!instance.occupies(worldPoint))
				continue;
			selectedFaces = instance.selectFaces(worldPoint);
			if(selectedFaces != null) {
				for(var face: selectedFaces) {
					Matrix4f transform = new Matrix4f(instance.getTransform()).mul(face.transform);
					face.transform = transform;
					selection = selectedFaces;
					break;
				}
				selectedInstance = instance;				
			}
		}

	}

	private void selectVertex(Vector3f worldPoint) {
		
	}
	
	private void selectPart(Vector3f worldPoint) {
		resetSelection();
		for(ModelInstance instance: objects) {
			if(!instance.occupies(worldPoint))
				continue;
			selectedPart = instance.selectPart(worldPoint);
			if(selectedPart != null) {
				selectedInstance = instance;
				selection = selectedPart;
				break;
			}
		}
	}
	
	public Triangle selectFace(Vector3f worldPoint) {
		resetSelection();
		Triangle selectedFace = null;
		for(ModelInstance instance: objects) {
			if(!instance.occupies(worldPoint))
				continue;
			selectedFace = instance.selectFace(worldPoint);
			if(selectedFace != null) {
				Matrix4f transform = new Matrix4f(instance.getTransform()).mul(selectedFace.transform);
				selectedFace.transform = transform;
				selectedFaces.add(selectedFace);
				selectedInstance = instance;
				selection = selectedFaces;
				break;
			}
		}
		return selectedFace;
	}
	
	
	
	
	private void resetSelection(){
		selection = null;
		if(selectedFaces == null) {
			selectedFaces = new ArrayList<Triangle>();
		}else {
			selectedFaces.clear();
		}
		selectedPart = null;
		selectedInstance = null;
	}
	
	@Override
	public boolean mouseUp(MouseEvent e) {
		return false;
	}
	@Override
	public boolean mouseMove(MouseEvent e) {
		return false;

	}
	@Override
	public boolean mouseScrolled(MouseEvent e) {
		return false;
	}
	@Override
	public void draw(Shader shader) {
		if(selection != null) {
			GL45.glDepthFunc(GL45.GL_LEQUAL);
			shader.use();
			GL11.glLineWidth(5);
			shader.bindMaterial(selectionMaterial);
			if(selection instanceof ModelInstance) {
				ModelInstance instance = (ModelInstance) selection;
				shader.bindMaterial(selectionMaterial);
				instance.drawBoundingBox(shader);
			}else if(selection instanceof Vector3f) {
				drawVertex(selection);
			}else if(selection instanceof Mesh) {
				Mesh mesh = (Mesh) selection;
				var originalMtl = mesh.getMtl();
				mesh.setMtl(selectionMaterial);
				((IDrawable) selection).draw(shader);
				((Mesh) selection).setMtl(originalMtl);
			}else if(selection == selectedFaces ) {
				shapeDrawer.begin(camera);
				shapeDrawer.setOpacity(0.5f);
				for(var face: selectedFaces) {
					shapeDrawer.drawTriangle(face.getV0(), face.getV1(), face.getV2(), face.transform, new Vector3f(1,0,0));
				}
				shapeDrawer.end();
			}
			GL11.glLineWidth(1);
			GL45.glDepthFunc(GL45.GL_LEQUAL);
			objectController.draw(shader);
		}
	}


	private void drawVertex(Object selection2) {
		
	}

	public ArrayList<ModelInstance> getObjects() {
		return objects;
	}

	public Camera getCamera() {
		return camera;
	}


	public void setCamera(Camera camera) {
		this.camera = camera;
	}


	public void setObjects(ArrayList<ModelInstance> objects) {
		this.objects = objects;
	}


	public static ObjectSelector getSelector() {
		if(selector == null)
			selector = new ObjectSelector();
		return selector;
	}

	@Override
	public boolean mouseDragged(MouseEvent e) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public void handle(MepEvent event) {
		EventType type = event.getType();
		switch (type) {
		case OBJECT_POSITIONER_TRIGFERED: 
			setController(ObjectPositioner.getPositioner());
			break;
		case OBJECT_ROTATOR_TRIGGERED:
			this.objectController = ObjectRotator.getRotator();
			break;
		case SELECTION_MODE_CHANGED:
			setSelectionMode((SelectionMode) event.getDetail("mode"));
		default:
			break;
		}
	}

	private void setSelectionMode(SelectionMode mode) {
		this.mode = mode;
	}

	private void setController(IObjectController controller) {
		objectController = controller;

		for(var handler:  InputMultiplexer.getInstance().getEventHandlers()) {
			if(handler instanceof IObjectController)
				InputMultiplexer.getInstance().removeEventHandler(handler);
		}

		InputMultiplexer.getInstance().pushEventHandler(controller);
	}

	public Object getSelection() {
		return selection;
	}


	
	


}
