package supercad.p2.update;

import java.net.URI;
import java.net.URISyntaxException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.e4.ui.workbench.IWorkbench;
import org.eclipse.equinox.p2.core.IProvisioningAgent;
import org.eclipse.equinox.p2.operations.ProvisioningJob;
import org.eclipse.equinox.p2.operations.ProvisioningSession;
import org.eclipse.equinox.p2.operations.UpdateOperation;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.frs.supercad.config.ConfigurationManager;

import java.util.Timer;
import java.util.TimerTask;

public class UpdateHandler {

	private String baseUrl;
	private String repo_loc;
	private IWorkbench workbench;
	private MessageDialog loadingDialog;
	private Shell loadingShell;
	private Label loadingLabel;
	private Timer loadingTimer;
	 private int dotCount = 0;

	public void setBuildNumber(String lastbuildNumber) {
		baseUrl = ConfigurationManager.getProperty("URL_P2_SITE_UPDATE");
		if (baseUrl == null || baseUrl.isEmpty()) {
			throw new IllegalStateException(
					"La propriété 'URL_P2_SITE_UPDATE' est manquante dans le fichier de configuration.");
		}
		repo_loc = baseUrl + lastbuildNumber + "/";
		System.out.println("URL de mise à jour configurée : " + repo_loc);
	}

//	public void execute(final IProvisioningAgent agent, IWorkbench workbench) {
//	    if (agent == null) {
//	        System.err.println("ProvisioningAgent is null - cannot perform update");
//	        return;
//	    }
//	    this.workbench = workbench;
//	    System.out.println("Début du processus de mise à jour");
//
//	    Job updateJob = Job.create("Recherche de mises à jour", monitor -> {
//	        try {
//	            Boolean userConfirmed = Display.getDefault().syncCall(() -> 
//	                MessageDialog.openQuestion(null, "Mise à jour disponible", "Une mise à jour est disponible. Voulez-vous l'installer ?")
//	            );
//
//	            if (userConfirmed != null && userConfirmed) {
//	                // ➔ Après confirmation : afficher "Téléchargement en cours..."
//	                Display.getDefault().syncExec(() -> {
//	                    loadingDialog = new MessageDialog(
//	                        Display.getDefault().getActiveShell(),
//	                        "Mise à jour", 
//	                        null,
//	                        "Téléchargement de la mise à jour en cours...\nVeuillez patienter.",
//	                        MessageDialog.INFORMATION,
//	                        new String[] {}, // pas de bouton pour forcer l'attente
//	                        0
//	                    );
//	                    loadingDialog.create();
//	                    loadingDialog.getShell().setEnabled(false); 
//	                    loadingDialog.open();
//	                });
//
//	                IStatus status = performUpdates(agent, monitor);
//
//	                System.out.println("Update process completed with status: " + status);
//
//	                Display.getDefault().asyncExec(() -> {
//	                    if (loadingDialog != null && loadingDialog.getShell() != null && !loadingDialog.getShell().isDisposed()) {
//	                        loadingDialog.close();
//	                    }
//	                });
//
//	                if (!status.isOK() && status.getSeverity() != IStatus.CANCEL) {
//	                    String errorDetails = getStatusDetails(status);
//	                    System.err.println("Update failed: " + errorDetails);
//
//	                    Display.getDefault().asyncExec(() ->
//	                        MessageDialog.openError(null, "Erreur", "Échec de la mise à jour: " + errorDetails)
//	                    );
//	                }
//	            } else {
//	                System.out.println("Mise à jour annulée par l'utilisateur.");
//	            }
//	        } catch (Exception e) {
//	            System.err.println("Exception during update: " + e.getMessage());
//	            e.printStackTrace();
//	        }
//	    });
//	    updateJob.schedule();
//	}
    public void execute(final IProvisioningAgent agent, IWorkbench workbench) {
        if (agent == null) {
            System.err.println("ProvisioningAgent is null - cannot perform update");
            return;
        }
        this.workbench = workbench;
        System.out.println("Début du processus de mise à jour");

        Job updateJob = Job.create("Recherche de mises à jour", monitor -> {
            try {
                Boolean userConfirmed = Display.getDefault().syncCall(() -> 
                    MessageDialog.openQuestion(null, "Mise à jour disponible", "Une mise à jour est disponible. Voulez-vous l'installer ?")
                );

                if (userConfirmed != null && userConfirmed) {
                    // ➔ Après confirmation : afficher "Téléchargement en cours..."
                    Display.getDefault().syncExec(() -> {
                        openLoadingDialog();
                    });

                    IStatus status = performUpdates(agent, monitor);

                    System.out.println("Update process completed with status: " + status);

                    Display.getDefault().asyncExec(() -> {
                        closeLoadingDialog();
                    });

                    if (!status.isOK() && status.getSeverity() != IStatus.CANCEL) {
                        String errorDetails = getStatusDetails(status);
                        System.err.println("Update failed: " + errorDetails);

                        Display.getDefault().asyncExec(() ->
                            MessageDialog.openError(null, "Erreur", "Échec de la mise à jour: " + errorDetails)
                        );
                    }
                } else {
                    System.out.println("Mise à jour annulée par l'utilisateur.");
                }
            } catch (Exception e) {
                System.err.println("Exception during update: " + e.getMessage());
                e.printStackTrace();
            }
        });
        updateJob.schedule();
    }

    private void openLoadingDialog() {
        loadingShell = new Shell(Display.getDefault().getActiveShell(), SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
        loadingShell.setText("Téléchargement en cours");
        loadingShell.setLayout(new GridLayout(1, false));

        loadingLabel = new Label(loadingShell, SWT.CENTER);
        loadingLabel.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, true, true));
        loadingLabel.setText("Téléchargement en cours");

        loadingShell.setSize(300, 100);
        loadingShell.setLocation(
            (Display.getDefault().getPrimaryMonitor().getBounds().width - loadingShell.getSize().x) / 2,
            (Display.getDefault().getPrimaryMonitor().getBounds().height - loadingShell.getSize().y) / 2
        );

        loadingShell.open();

        // Timer pour animer les points
        loadingTimer = new Timer();
        loadingTimer.scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                Display.getDefault().asyncExec(() -> {
                    if (loadingLabel != null && !loadingLabel.isDisposed()) {
                        String dots = ".".repeat(dotCount % 4); // 0, 1, 2, 3 points
                        loadingLabel.setText("Téléchargement en cours" + dots);
                        loadingLabel.pack();
                        loadingShell.layout();
                        dotCount++;
                    }
                });
            }
        }, 0, 500); // toutes les 500ms
    }

    private void closeLoadingDialog() {
        if (loadingTimer != null) {
            loadingTimer.cancel();
            loadingTimer = null;
        }
        if (loadingShell != null && !loadingShell.isDisposed()) {
            loadingShell.close();
        }
    }

	private IStatus performUpdates(final IProvisioningAgent agent, IProgressMonitor monitor) {
	    System.out.println("Creating provisioning session");
	    final ProvisioningSession session = new ProvisioningSession(agent);
	    final UpdateOperation operation = new UpdateOperation(session);

	    try {
	        URI uri = new URI(repo_loc);
	        System.out.println("Setting repository: " + uri);
	        operation.getProvisioningContext().setArtifactRepositories(uri);
	        operation.getProvisioningContext().setMetadataRepositories(uri);

	        System.out.println("Resolving updates...");
	        final IStatus status = operation.resolveModal(monitor);
	        logStatus("Resolution status", status);

	        if (status.getCode() == UpdateOperation.STATUS_NOTHING_TO_UPDATE) {
	            Display.getDefault().asyncExec(
	                () -> MessageDialog.openInformation(null, "Mise à jour", "Aucune mise à jour disponible.")
	            );
	            return Status.CANCEL_STATUS;
	        }

	        if (!status.isOK()) {
	            return status; // Return the error status
	        }

	        // ➔ SUPPRIMÉ : demande de confirmation

	        ProvisioningJob provisioningJob = operation.getProvisioningJob(monitor);
	        if (provisioningJob == null) {
	            System.err.println("ProvisioningJob is null - cannot proceed with update");
	            Display.getDefault().asyncExec(() -> {
	                MessageDialog.openWarning(Display.getDefault().getActiveShell(), "Update Error",
	                    "Could not create installation job.\n" + "Resolution result: "
	                    + operation.getResolutionResult());
	            });
	            return new Status(IStatus.ERROR, "supercad.p2.update", "ProvisioningJob is null");
	        }
	        System.out.println("Configuring provisioning job !");
	        configureProvisioningJob(provisioningJob);
	        provisioningJob.schedule();
	        return Status.OK_STATUS;

	    } catch (URISyntaxException e) {
	        System.err.println("Invalid repository URI: " + e.getMessage());
	        return new Status(IStatus.ERROR, "supercad.p2.update", "Invalid repository URI", e);
	    }
	}


	private void configureProvisioningJob(ProvisioningJob provisioningJob) {

		provisioningJob.addJobChangeListener(new JobChangeAdapter() {
			@Override
			public void done(IJobChangeEvent event) {
				System.out.println("Provisioning job completed with IStatusCodes : " + event.getResult().getCode());
				System.out.println("Provisioning job completed with Message : " + event.getResult().getMessage());
				System.out.println("Provisioning job completed with Plugin : " + event.getResult().getPlugin());

				if (event.getResult().getException() != null)
					System.out.println("Provisioning job completed with Plugin : "
							+ event.getResult().getException().getMessage());

				if (event.getResult().getSeverity() == IStatus.CANCEL) {
					System.err.println(
							"Provisioning job was cancelled. Possible cause: identical version already installed.");
				}
				if (event.getResult().isOK()) {
					System.out.println("Update installed successfully");	
					System.out.println("Mise à jour terminée avec succès, redémarrage de l'application...");
					Display.getDefault().asyncExec(() -> {
					    // Créer un MessageDialog personnalisé
					    MessageDialog dialog = new MessageDialog(
					            Display.getDefault().getActiveShell(), // Utiliser la fenêtre active
					            "Redémarrage nécessaire",               // Titre
					            null,                                   // Pas d'icône personnalisée
					            "La mise à jour a été installée avec succès.\n"
					            + "L'application doit redémarrer pour appliquer les changements.\n\n"
					            + "Redémarrer maintenant ?",            // Message
					            MessageDialog.QUESTION,                 // Type d'icône
					            new String[] { "Oui", "Non" },           // Boutons en français
					            0                                        // Bouton par défaut ("Oui")
					    );

					    int result = dialog.open();

					    if (result == 0) { // 0 correspond à "Oui"
					        workbench.restart();
					    }
					});
				} else {
					String errorDetails = getStatusDetails(event.getResult());
					System.err.println("Update installation failed: " + errorDetails);

					Display.getDefault().asyncExec(() -> MessageDialog.openError(null, "Erreur",
							"Échec de l'installation de la mise à jour:\n" + event.getResult().getMessage()));
				}
			}
		});
	}
	
	private void logStatus(String prefix, IStatus status) {
		System.out.println(prefix + ": " + status);
		if (status.isMultiStatus()) {
			for (IStatus child : status.getChildren()) {
				System.out.println("  - " + child);
			}
		}
	}

	private String getStatusDetails(IStatus status) {
		StringBuilder sb = new StringBuilder();
		sb.append(status.getMessage());

		if (status.getException() != null) {
			sb.append("\nException: ").append(status.getException().getMessage());
		}

		if (status.isMultiStatus()) {
			for (IStatus child : status.getChildren()) {
				sb.append("\n  - ").append(child.getMessage());
				if (child.getException() != null) {
					sb.append(" [").append(child.getException().getMessage()).append("]");
				}
			}
		}

		return sb.toString();
	}

}
