package geometry;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Stack;

import org.eclipse.swt.widgets.Display;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.Input.Keys;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer.ShapeType;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Actor;
import com.badlogic.gdx.scenes.scene2d.InputEvent;
import com.badlogic.gdx.scenes.scene2d.InputListener;
import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Triangle;
import dressing.model.ModelProvider;
import dressing.model.persistence.dpos.CompoundShapeDPO;
import dressing.model.persistence.mappers.Persistable;
import dressing.ui.engine3d.SceneTexture;
import dressing.ui.util.EdgeMesureWidow;
import gdxapp.assets.AssetsTextures;
import gdxapp.assets.DrawingHelper;
import gdxapp.screens.room.RoomController;
import gdxapp.shaders.PbrMaterial;

/**
 * @author nadhem
 *
 */

@Persistable(persistableForm = CompoundShapeDPO.class)
public class CompoundShape extends Actor implements Shape{
	
	public static ShapeRenderer shapeRenderer;
	private final ArrayList<Shape> nodes = new ArrayList<Shape>();
	private transient ArrayList<Vector2> vertices;
	private transient ArrayList<Vector2> normals;
	private transient ArrayList<Edge> edges;
	private boolean closed = false;
	//to be removed
	private transient CompoundShapeInputListener inputListener;
	private transient float[] mesh;
	private transient short[] indices;
	private transient ArrayList<Triangle> triangles;
	private transient Texture texture;
	private PbrMaterial material;
	private Matrix4 worldTransform;
	private transient ArrayList<Vector2> joints = new ArrayList<Vector2>();
	private transient long time = 0L;
	transient protected PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this.getClass());
	public CompoundShape() {
		super();
		inputListener = new CompoundShapeInputListener();
		addListener(inputListener);
		material = new PbrMaterial();
	}
	
	public void prepareToListen() {
		setWidth(Gdx.graphics.getWidth());
		setHeight(Gdx.graphics.getHeight());
		setX(0);
		setY(0);
		setVisible(true);
		if(inputListener == null) {
			inputListener = new CompoundShapeInputListener();
			addListener(inputListener);
		}
		
	}
	
	public CompoundShape(ArrayList<Shape> nodes) {
		super();
		setNodes(nodes);
	}

	public void addNode(Shape node) {
		nodes.add(node);
		calculateProperties();
	}

	@Override
	public void draw(Batch batch, float parentAlpha) {
		time += parentAlpha;
		if(vertices == null)
			calculateVertices();
		if(shapeRenderer == null)
			shapeRenderer = new ShapeRenderer();
		shapeRenderer.setProjectionMatrix(batch.getProjectionMatrix());
		shapeRenderer.setColor(Color.BLACK);
		boolean batchInterruped = false;
		if(batch.isDrawing()) {
			batch.end();
			batchInterruped = true;
		}
		Color glowColor = new Color((float)Math.sin(0.1f * time), (float)Math.sin(0.1f * time),0,1.0f);
		for(Shape shape: nodes) {
			shape.draw(shapeRenderer, glowColor);
		}
		shapeRenderer.begin(ShapeType.Filled);
		for(Vector2 joint: joints) {
			shapeRenderer.setColor(Color.GREEN);
			shapeRenderer.circle(joint.x, joint.y, 4);
			shapeRenderer.setColor(Color.BLACK);
		}
		shapeRenderer.end();

		if(closed && nodes.size() > 0 ) {
			Vector2 lastVertex = nodes.get(nodes.size()-1).getlastVertex();
			Vector2 firstVertex = nodes.get(0).getVertices().get(0);
			shapeRenderer.setColor(glowColor);
			shapeRenderer.begin(ShapeType.Filled);
			shapeRenderer.rectLine(lastVertex, firstVertex , 5);
			shapeRenderer.end();
			shapeRenderer.setColor(Color.BLACK);
		}
		

    	for(Edge edge: getEdges()) {
    		float length = edge.getLength() / RoomController.getInstance().getScaleX();
    		length = Math.round(1000 * length);
    		Vector2 position = edge.getMiddle().add(edge.getNormal().scl(25));
    		float rotation = edge.getV1().cpy().sub(edge.getV0()).angle();
    		DrawingHelper.drawText("" + length, new Vector3(position,0.0f), rotation, Color.BLACK, 1.0f, RoomController.getInstance().getStage().getBatch());
    	}
        
		if(Gdx.input.isKeyPressed(Keys.T)){
				triangles = EarClipper.triangulate(vertices);
				calculateProperties();
			for(Triangle tr: triangles) {
				tr.draw(shapeRenderer);
			}
		}
		
		if(Gdx.input.isKeyPressed(Keys.F)){
			drawAsPoly(batch);
		}
		
		if(Gdx.input.isKeyPressed(Keys.H)){
			
			ArrayList<Vector2> outerContour = MathUtilities.calculateRectangulairLine(vertices, 50);//MathUtilities.calculateSurfaceContour(getVertices(), isClosed(), 20);
			shapeRenderer.begin(ShapeType.Filled);
			for(int i = 0; i < outerContour.size() -1; i++) {
				shapeRenderer.rectLine(outerContour.get(i), outerContour.get(i+1) , 2);
			}
//			if(isClosed()) {
//				shapeRenderer.rectLine(outerContour.get(0), outerContour.get(outerContour.size()-1) , 2);
//			}
			shapeRenderer.end();
		}
		if(inputListener != null) 
			inputListener.draw();

		if(batchInterruped)
			batch.begin();
	}
	
	
	private void drawAsPoly(Batch batch) {
		if(texture == null) {
			texture = AssetsTextures.getInstance().getDefaultTexture();
		}
		RoomController.getPolyBatch().setProjectionMatrix(batch.getProjectionMatrix().cpy());
		RoomController.getPolyBatch().begin();
		RoomController.getPolyBatch().draw(texture, mesh, 0, mesh.length,indices, 0, indices.length);
		RoomController.getPolyBatch().end();
	}
	
	public void calculateVertices() {
		if(this.vertices == null) {
			vertices = new ArrayList<Vector2>();
		}else {
			vertices.clear();
		}
		for(Shape node: nodes) {
			if(node instanceof Arc)
				((Arc) node).calculateVertices();
			if(!vertices.isEmpty()) {
				if(vertices.get(vertices.size() -1).epsilonEquals(node.getVertices().get(0)))
					vertices.remove(vertices.size() -1);
			}
			vertices.addAll(node.getVertices());
		}
	}
	
	public void calculateEdges() {
		if(this.edges == null) {
			edges = new ArrayList<Edge>();
		}else {
			edges.clear();
		}
		for(Shape shape: nodes) {
			if(shape instanceof Polygon) {
				((Polygon) shape).calculateEdges();
				edges.addAll(((Polygon)shape).getEdges());
			}
		}
		if(isClosed()) {
			edges.add(new Edge(getVertices().get(getVertices().size() - 1), getVertices().get(0)));
		}
	}

	public ArrayList<Shape> getNodes() {
		return nodes;
	}
	

	public void setNodes(ArrayList<Shape> nodes) {
		this.nodes.clear();
		for(Shape node: nodes) {
			this.nodes.add(node);
		}
		calculateProperties();
	}
	
	
	public ArrayList<Edge> getEdges() {
		if(edges == null)
			calculateEdges();
		return edges;
	}

	public void calculateProperties() {
		calculateVertices();
		Vector2[] boundaries = MathUtilities.getBoundariesPlane(vertices);
		float width = Math.max(boundaries[1].x - boundaries[0].x,1);
		float height = Math.max(boundaries[1].y - boundaries[0].y,1);
		setWidth(width);
		setHeight(height);
		setPosition(boundaries[0].x, boundaries[0].y);
		Vector2 origin = boundaries[0].cpy().add(boundaries[1]).scl(0.5f);
		setOrigin(origin.x, origin.y);
		
		triangles = EarClipper.triangulate(vertices);
		indices = new short[triangles.size() * 3];
		if(texture == null) {
			texture = AssetsTextures.getInstance().getDefaultTexture();
		}
		float[] uvRange = {0, 1, 0, 1};
		mesh = MathUtilities.indexMesh(triangles, this.indices, uvRange, Color.WHITE_FLOAT_BITS);
	}
	
	public void calculateJoints() {
		joints.clear();
		for(Shape shape: nodes) {
			if(shape instanceof Polygon) {
				joints.addAll(shape.getVertices());
			}
			if(shape instanceof Arc) {
				joints.add(((Arc) shape).getV0());
				joints.add(((Arc) shape).getV1());
			}
		}
	}
	
	public void scale(Vector2 scale) {
		for(Shape node: nodes) {
			node.scale(scale);
		}
	}

	public ArrayList<Vector2> getVertices() {

			calculateVertices();
		return vertices;
	}
	
	public boolean isClosed() {
		return closed;
	}

	public void setClosed(boolean closed) {
		this.closed = closed;
	}

	public Shape getNearestNode(Vector2 position) {
		Shape nearest = null;
		if(nodes != null) {
			for(Shape shape: nodes) {
				float dst = shape.distanceTo(position);
				if( dst < 50) {
					if(nearest == null || nearest.distanceTo(position) > dst) {
						nearest = shape;
					}
				}
			}
		}
		
		return nearest;
	}
	
	public void removeNode(Shape node) {
		if(nodes == null)
			return;
		nodes.remove(node);
	}
	
	
	public Matrix4 getWorldTransform() {
		if(worldTransform==null) {
			this.worldTransform=RoomController.getInstance().getSceneToWorldTransform();
		}
		return worldTransform;
	}

	public void setWorldTransform(Matrix4 worldTransform) {
		this.worldTransform = worldTransform;
	}

	public CompoundShape cpy() {
		CompoundShape clone = new CompoundShape();
		if(worldTransform != null)
			clone.setWorldTransform(worldTransform.cpy());
		for(Shape shape: nodes) {
			clone.addNode(shape.cpy());
		}
		clone.setClosed(this.closed);
		clone.setMaterial(material.cpy());
		return clone;
	}
	
	public <T extends Shape> ArrayList<T> getNodes(Class clazz) {
		ArrayList<T> filteredNodes = new ArrayList<>();
		for(Shape shape: getNodes()) {
			if(clazz.isInstance(shape)) {
				filteredNodes.add((T) shape);
			}
		}
		return filteredNodes;
	}
	
	private ArrayList<Shape> findNode(Vector2 vertex) {
		ArrayList<Shape> parents = new ArrayList<Shape>();
		for (Shape shape : nodes) {
			if (shape.getVertices().contains(vertex))
				parents.add(shape);
		}
		return parents;
	}
	
	//edge can not be shared by multiple polygon of the same compound shape
	private Polygon findParentPoly(Edge edge) {
		ArrayList<Polygon> polies = getNodes(Polygon.class);
		Polygon parent = null;
		for(Polygon poly: polies) {
			if(poly.getEdges().contains(edge)) {
				parent = poly;
				break;
			}
		}
		return parent;
	}
	
	public void fuseCloseVertexWithinTheSameNode() {
		for(Shape polygon: getNodes(Polygon.class)) {
			((Polygon)polygon).fuseCloseVertices();
		}
		
		
	}
	
	protected void deleteEdge(Edge edge, Polygon parent) {
		if(parent.getEdges().size() < 2) {
			getNodes().remove(parent);
			return;
		}
		Polygon[] newPolygons = parent.split(edge);
		int index = getNodes().indexOf(parent);
		getNodes().remove(index);
		if(newPolygons != null) {
			if(newPolygons[0] != null) {
				getNodes().add(index, newPolygons[0]);
				if(newPolygons[1] != null) {
					getNodes().add(index + 1, newPolygons[1]);
				}
			}
		}	
	}
	
	
	public CompoundShapeInputListener getInputListener() {
		return inputListener;
	}
	
	public void fusePolies() {
		ArrayList<Vector2> starts = new ArrayList<Vector2>(), ends = new ArrayList<Vector2>();
		Shape[] fusable = null;
		for(Shape node: getNodes(Polygon.class)) {
			ends.add(node.getlastVertex());
			starts.add(node.getVertices().get(0));
		}
		for(Vector2 end: ends) {
			for(Vector2 start: starts) {
				if(start.dst(end) < 5) {
					Shape first = getShapeEndingWith(end);
					Shape second = getShapeStartingWith(start);
					fusable = new Shape[] {first, second};
					break;
				}
			}
			if(fusable != null)
				break;
		}
		if(fusable != null) {
				int index = nodes.indexOf(fusable[1]);
				nodes.remove(index);
				fusable[1].getVertices().remove(0);
				fusable[0].getVertices().addAll(fusable[1].getVertices());
				((Polygon)fusable[0]).calculateEdges();
				calculateEdges();
				fusePolies();
		}
	}


	private Shape getShapeStartingWith(Vector2 vertex) {
		Shape shape = null;
		for(Shape node: nodes) {
			if(node.getVertices().get(0).epsilonEquals(vertex)) {
				shape = node;
				break;
			}
		}
		return shape;
	}

	private Shape getShapeEndingWith(Vector2 vertex) {
		Shape shape = null;
		for(Shape node: nodes) {
			if(node.getlastVertex().epsilonEquals(vertex)) {
				shape = node;
				break;
			}
		}
		return shape;
	}
	
	


	public PbrMaterial getMaterial() {
		if(material == null)
			material = new PbrMaterial();
		return material;
	}

	public void setMaterial(PbrMaterial material) {
		this.material = material;
	}
	
	public SceneTexture getTexture() {
			return ModelProvider.getTexture(getMaterial().getAlbedoMapPath());
	}
	

	@Override
	public void draw(ShapeRenderer shapeRenderer, Color color) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public Vector2 getlastVertex() {
		Vector2 last = null;
		if(nodes != null && !nodes.isEmpty())
			last = getNodes().get(nodes.size()-1).getlastVertex();
		return last;
	}

	@Override
	public float distanceTo(Vector2 position) {
		float distance = Float.MAX_VALUE;
		for(Shape node: nodes) {
			float nodeDst = node.distanceTo(position);
			if(nodeDst < distance)
				distance = node.distanceTo(position);
		}
		return distance;
	}

	@Override
	public Vector2 projectPoint(Vector2 point) {
		if(nodes != null && !nodes.isEmpty()) {
			return getNearestNode(point).projectPoint(point);
		}
		return null;
	}

	@Override
	public Object select(Vector2 position) {
		Shape node = getNearestNode(position);
		if(node != null)
			return node.select(position);
		return null;
	}

	//this method assumes that all nodes have the same winding
	@Override
	public int getWinding() {
		if(nodes != null && !nodes.isEmpty())
			return nodes.get(0).getWinding();
		return 0;
	}
	
	
	public void setWinding(int winding) {
		ArrayList<Vector2> vertices = new  ArrayList<Vector2>();
		for(Shape node: nodes) {
			vertices.add(node.getVertices().get(0));
			vertices.add(node.getlastVertex());
		}
		int currentWinding = MathUtilities.getWinding(vertices.toArray(new Vector2[0]));
		if(currentWinding * winding < 0) {
			Stack<Shape> stack = new Stack<Shape>();
			for(Shape node: nodes) {
				stack.push(node);
			}
			this.nodes.clear();
			while(!stack.isEmpty()) {
				nodes.add(stack.pop());
			}
		}
	}


	public void addPropertyChangeListener(PropertyChangeListener propertyChangeListener) {
		if(this.propertyChangeSupport == null)
			propertyChangeSupport = new PropertyChangeSupport(this);
		this.propertyChangeSupport.addPropertyChangeListener(propertyChangeListener);
		// TODO Auto-generated method stub
		
	}
	public void removePropertyChangeListener(PropertyChangeListener propertyChangeListener) {
		this.propertyChangeSupport.removePropertyChangeListener(propertyChangeListener);
	}
	
	
	public class CompoundShapeInputListener extends InputListener{
		
		private Vector2 selectedVertex;
		private Edge selectedEdge;
		private Arc selectedArc;		
		private float startX, startY; 
		private long lastClickDate;
		
		private void handleDoubleClick() {
			if(selectedEdge != null) {
				Display.getDefault().asyncExec(new Runnable() {
					@Override
					public void run() {
						EdgeMesureWidow window = EdgeMesureWidow.getInstance();
						window.setEdge(selectedEdge);
						window.setEdges(CompoundShape.this.getEdges());
						window.open();
					}				
				});
			}
		}

		@Override
		public boolean touchDown(InputEvent event, float x, float y, int pointer, int button) {
			long currentTime = System.currentTimeMillis();
			System.err.println(currentTime - lastClickDate);
			if(currentTime - lastClickDate < 200) {
				handleDoubleClick();
				lastClickDate = currentTime;
				return true;
			}
			startX = x;
			startY = y;
			getStage().setKeyboardFocus(CompoundShape.this);
			calculateJoints();
			makeSelection(new Vector2(x,y));
			lastClickDate = currentTime;
			return true;
		}

		

		@Override
		public void touchUp(InputEvent event, float x, float y, int pointer, int button) {
			if(selectedVertex != null && !Gdx.input.isKeyPressed(Input.Keys.SHIFT_LEFT)) {
				rectifyPosition(selectedVertex);
			}
			fuseCloseVertexWithinTheSameNode();
			fusePolies();
		}

		@Override
		public void touchDragged(InputEvent event, float x, float y, int pointer) {
			Vector2 movement = new Vector2(x - startX , y - startY);
			if(selectedVertex != null) {
				selectedVertex.add(movement);
			}
			if(selectedEdge != null) {
				for(Vector2 vertex: selectedEdge.getVertices()) {
					vertex.add(x - startX, y - startY);
				}
			}
			startX = x;
			startY = y;
		}

		@Override
		public boolean mouseMoved(InputEvent event, float x, float y) {
			// TODO Auto-generated method stub
			return super.mouseMoved(event, x, y);
		}

		@Override
		public void enter(InputEvent event, float x, float y, int pointer, Actor fromActor) {
			// TODO Auto-generated method stub
			super.enter(event, x, y, pointer, fromActor);
		}

		@Override
		public void exit(InputEvent event, float x, float y, int pointer, Actor toActor) {
			// TODO Auto-generated method stub
			super.exit(event, x, y, pointer, toActor);
		}

		@Override
		public boolean scrolled(InputEvent event, float x, float y, int amount) {
			// TODO Auto-generated method stub
			return super.scrolled(event, x, y, amount);
		}

		@Override
		public boolean keyDown(InputEvent event, int keycode) {
			// TODO Auto-generated method stub
			return true;
		}

		@Override
		public boolean keyUp(InputEvent event, int keycode) {
			
			if(keycode == Input.Keys.C) {
				closed = !closed;
			}
			
			if (keycode == Input.Keys.FORWARD_DEL) {
				deleteRequested();
			}
			
			if (keycode == Input.Keys.ESCAPE) {
				endEdit();
			}

			return true;
		}

		private void endEdit() {
			calculateEdges();
			calculateJoints();
			calculateProperties();
			RoomController.getInstance().endWallEdit(CompoundShape.this);
		}

		@Override
		public boolean keyTyped(InputEvent event, char character) {
			// TODO Auto-generated method stub
			return super.keyTyped(event, character);
		}
		
		
		
		public void makeSelection(Vector2 touchPoint) {
			selectedVertex = null;
			selectedEdge = null;
			selectedArc = null;
			for(Vector2 joint: joints) {
				if(joint.dst(touchPoint) < 10) {
					if(selectedVertex == null || selectedVertex.dst(touchPoint) > joint.dst(touchPoint)) {
						selectedVertex = joint;
						return;
					}
				}
			}
			Shape node = getNearestNode(touchPoint);
			if(node instanceof Polygon) {
				for(Edge edge: ((Polygon) node).getEdges()) {
					if(edge.distanceTo(touchPoint) < 10) {
						if(selectedEdge == null || selectedEdge.distanceTo(touchPoint) > edge.distanceTo(touchPoint))
							selectedEdge = edge;
							return;
					}
				}
			}
			if(node instanceof Arc) {
				if(node.distanceTo(touchPoint) < 10) {
					selectedArc = (Arc) node;
				}
			}
		}
		
		public void draw() {
			shapeRenderer.begin(ShapeType.Filled);
			if(selectedVertex != null) {
				shapeRenderer.setColor(Color.RED);
				shapeRenderer.circle(selectedVertex.x, selectedVertex.y,4);
				shapeRenderer.setColor(Color.BLACK);
			}
			if(selectedEdge != null) {
				shapeRenderer.setColor(Color.RED);
				shapeRenderer.rectLine(selectedEdge.getV0(), selectedEdge.getV1(),3);
				shapeRenderer.setColor(Color.BLACK);
			}
			if(selectedArc != null) {
				selectedArc.draw(shapeRenderer, Color.RED);
				shapeRenderer.setColor(Color.BLACK);
			}
			shapeRenderer.end();
		}
		
		private void rectifyPosition(Vector2 vertex) {
			if(joints.size()<=2) {
				return ;
			}
			for(Vector2 joint: joints) {
				if(joint.dst(vertex) < 5) {
					if(vertex.epsilonEquals(joint))
						continue;
					System.err.println("distance: " + joint.dst(vertex));
					vertex.set(joint);
					return;
				}
			}
	        int index = joints.indexOf(vertex);
	        if(index < 2)
	            index = joints.size();
	        Vector2 v0 = joints.get(index-2).cpy();
	        Vector2 v1 = joints.get(index -1).cpy();
	        if(v0.epsilonEquals(v1))
	        	return;
	        Vector2 v01 = new Vector2(v0).sub(v1);
	        Vector2 v21 = new Vector2(vertex).sub(v1);
	        float angle = MathUtilities.signedAngle(v01,v21);
	        float rads15 = (float) Math.toRadians(15);
	        float rectifiedAngle = Math.round(angle/rads15) * rads15;
	        Vector2 rectifiedDirection = v01.nor().rotateRad(rectifiedAngle).scl(v21.len());
	        vertex.set(v1).add(rectifiedDirection);
	    }
		
		void deleteRequested(){
			boolean deletePerformed = false;
			if(selectedVertex != null) {
				shortcutVertex(selectedVertex);
				selectedVertex = null;
				deletePerformed = true;
			}
			if(selectedEdge != null) {
				Polygon parent = findParentPoly(selectedEdge);
				if(parent != null) {
					deleteEdge(selectedEdge, parent);
					selectedEdge = null;
					deletePerformed = true;
				}
			}
			if(selectedArc != null) {
				nodes.remove(selectedArc);
				selectedArc = null;
				deletePerformed = true;
			}
			if(deletePerformed) {
				calculateJoints();
				calculateEdges();
				if(edges.isEmpty())
					endEdit();
			}
		}
		
		protected void shortcutVertex(Vector2 vertex) {
			ArrayList<Shape> parents = findNode(vertex);
			for(Shape parent : parents) {
				if(parent instanceof Polygon) {
					parent.getVertices().remove(vertex);
				}
			}
			joints.remove(vertex);
		}
	}




}
