package org.frs.svg;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Stack;

import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;


public class SvgDrawer {

	public enum RENDERING_MODE {
		LINES, FILLED;
	}

	private ProjectionCamera camera;
	private MeshBuilder meshBuilder = new MeshBuilder();
	private ArrayList<Triangle> tmpFaces = new ArrayList<Triangle>();
	private ArrayList<LineSegment> tmpLines = new ArrayList<LineSegment>();
	private ArrayList<Triangle> tmpClipFaces = new ArrayList<Triangle>();

	private RENDERING_MODE mode;
	private final StringBuilder stringBuilder = new StringBuilder();
	private Stack<SVGRenderingTask> renderingStack = new Stack<SVGRenderingTask>();
	private float lineThickness = 1.0f;
	private Vector3 lightPosition = new Vector3(-5, 5, 5);
	private Vector3 strokeColor;

	public void addRenderable(SVGRenderingTask task) {
		renderingStack.addElement(task);
	}

	public void render(ProjectionCamera camera) {
		while (!renderingStack.isEmpty()) {
			SVGRenderingTask task = renderingStack.pop();
			executeRenderingTask(task);
		}
	}

	public void begin(ProjectionCamera camera) {
		this.setCamera(camera);
		stringBuilder.setLength(0);
		stringBuilder.append(String.format("<svg width=\"%s\" height=\"%s\"  style=\"border:none\">",
				camera.getScreenWidth(), camera.getScreenWidth() * camera.getAspectRatio()));
	}

	public void executeRenderingTask(SVGRenderingTask task) {
		tmpFaces.clear();
		tmpClipFaces.clear();
		tmpLines.clear();
		mode = task.getMode();
		this.lineThickness = task.getLineThickness();
		for (Geometry renderable : task.getRenderables()) {
			cpyPrimitives(renderable);
		}
		draw();
	}

	public void draw() {
		switch (mode) {
		case FILLED:
			renderTriangles();
			break;
		default:
			renderLines();
			break;
		}
	}

	private void renderLines() {
		for (LineSegment line : tmpLines) {
			Vector3 p0 = getCamera().project(line.getV0());
			Vector3 p1 = getCamera().project(line.getV1());
			p0.y = getCamera().getScreenWidth() * getCamera().getAspectRatio() - p0.y;
			p1.y = getCamera().getScreenWidth() * getCamera().getAspectRatio() - p1.y;
			stringBuilder.append(SvgPrinter.writeLine(p0, p1, strokeColor, lineThickness) + "\n");
		}
	}

	private void renderTriangles() {
//		sortByDepth(3);
		for (Triangle face : tmpFaces) {
			Vector3 p0 = getCamera().toScreenSpace(face.getV0());
			Vector3 p1 = getCamera().toScreenSpace(face.getV1());
			Vector3 p2 = getCamera().toScreenSpace(face.getV2());
			tmpClipFaces.add(new Triangle(p0, p1, p2, face.getColor()));
		}
//		meshBuilder.setFaces(tmpClipFaces);
//		meshBuilder.setCamera(camera);
//		meshBuilder.removeNonOverlappingTriangles();
//		try {
//			meshBuilder.scanMesh(tmpClipFaces);
//		}catch (Exception e) {
//			e.printStackTrace();
//		}
//		meshBuilder.scanMesh(tmpClipFaces);
		for(Triangle clippedFace: tmpClipFaces) {
			stringBuilder.append(SvgPrinter.writeTriangle(clippedFace.getV0(), clippedFace.getV1(), clippedFace.getV2(), clippedFace.getColor()));
		}
	}

	public String end() {
		stringBuilder.append("</svg>");
		String image = stringBuilder.toString();
		stringBuilder.setLength(0);
		return image;
	}

	private void cpyPrimitives(Geometry geometry) {
		ArrayList<Triangle> tmpList = new ArrayList<Triangle>(); 
		for (Triangle face : geometry.getFaces()) {
			Triangle triangle = face.cpy().transform(geometry.getTransform());
			triangle.setColor(geometry.getColor());
			//shade(triangle, lightPosition, camera.getPosition());
			tmpList.add(triangle);
		}
		//removeHiddenFaces(tmpList);
		tmpFaces.addAll(tmpList);
		// testDepth();
		if(mode == RENDERING_MODE.LINES) {
			calculateLinearMesh();
		}	
	}
	
	public void removeHiddenFaces(ArrayList<Triangle> faces) {
		ArrayList<Triangle> visible = new ArrayList<Triangle>();
		for (Triangle face : faces) {
			if(getDepthScore(face, faces) == 0)
				visible.add(face);
		}
		faces.clear();
		faces.addAll(visible);
	}

	public void sortByDepth(int depth) {
		System.out.println("sorting faces!");
		ArrayList<Triangle> l0 = new ArrayList<Triangle>();
		ArrayList<Triangle> l1 = new ArrayList<Triangle>();
		ArrayList<Triangle> l2 = new ArrayList<Triangle>();
		ArrayList<Triangle> l3 = new ArrayList<Triangle>();
		for (Triangle face : tmpFaces) {
			switch (getDepthScore(face, tmpFaces)) {
			case 0:
				l0.add(face);
				break;
			case 1:
				l1.add(face);
				break;
			case 2:
				l2.add(face);
				break;
			default:
				l3.add(face);
			}
		}
		tmpFaces.clear();
		if (l2.size() + l1.size() > 0) {
			for (Triangle face : l3) {
				face.tesselate(2, tmpFaces);
			}
			for (Triangle face : l2) {
				face.tesselate(2, tmpFaces);
			}
			
			for (Triangle face : l1) {
				face.tesselate(2, tmpFaces);
			}
		}
		tmpFaces.addAll(l0);
		if(depth < 5)
			sortByDepth(++depth);
	}

	public int getDepthScore(Triangle face, ArrayList<Triangle> mesh) {
		int score = 0;
		for (Vector3 sample : face.getSmaples()) {
			boolean sampleIsHidden = false;
			Line line = new Line(sample, getCamera().getPosition().cpy().sub(sample));
			for (Triangle faceX : mesh) {
				if (faceX == face)
					continue;
				Object intersection = faceX.intersect(line);
				if (intersection instanceof Vector3) {
					Vector3 i = (Vector3) intersection;
					if (i.cpy().sub(sample).dot(getCamera().getPosition().cpy().sub(sample)) >= 0) {
						sampleIsHidden = true;
						break;
					}

				}
			}
			if (sampleIsHidden) {
				score++;
				continue;
			}
		}
		return score;

	}

	private void testDepth() {
		ArrayList<Triangle> hidden = new ArrayList<Triangle>();
		for (Triangle face : tmpFaces) {
			Line line = new Line(face.getIsocenter(), getCamera().getPosition());
			for (Triangle faceX : tmpFaces) {
				if (faceX == face)
					continue;
				Object intersection = faceX.intersect(line);
				if (intersection instanceof Vector3) {
					Vector3 i = (Vector3) intersection;
					if (i.cpy().sub(face.getIsocenter()).dot(getCamera().getPosition().cpy().sub(face.getIsocenter())) >= 0)
						hidden.add(face);
					break;
				}
			}
		}
		for (Triangle toHide : hidden) {
			tmpFaces.remove(toHide);
		}
	}

	public HashMap<Plane, ArrayList<Triangle>> sortFacesbyPlane() {
		HashMap<Plane, ArrayList<Triangle>> groups = new HashMap<Plane, ArrayList<Triangle>>();
		for (Triangle face : tmpFaces) {
			Plane facePlane = face.getPlane();
			ArrayList<Triangle> group = null;
			for (Plane plane : groups.keySet()) {
				if (facePlane.equals(plane)) {
					group = groups.get(plane);
					break;
				}
			}
			if (group == null) {
				group = new ArrayList<Triangle>();
				groups.put(facePlane, group);
			}
			group.add(face);
		}
		return groups;
	}

	public void calculateLinearMesh() {
		if (this.tmpLines == null) {
			this.tmpLines = new ArrayList<LineSegment>();
		} else {
			this.tmpLines.clear();
		}
		HashMap<Plane, ArrayList<Triangle>> facesbyPlane = sortFacesbyPlane();
		for (Plane plane : facesbyPlane.keySet()) {
			ArrayList<Triangle> coplanarfaces = facesbyPlane.get(plane);
			ArrayList<LineSegment> edges = new ArrayList<LineSegment>();
			for (Triangle triangle : coplanarfaces) {
				edges.addAll(Arrays.asList(triangle.getEdges()));
			}
			boolean finish;
			do {
				finish = true;
				for (int i = 0; i < edges.size(); i++) {
					ArrayList<Integer> occurences = new ArrayList<Integer>();
					occurences.add(i);
					for (int j = i + 1; j < edges.size(); j++) {
						if (edges.get(j).equals(edges.get(i)))
							occurences.add(j);
					}
					if (occurences.size() > 1) {
						for (int k = 0; k < occurences.size(); k++) {
							edges.remove(occurences.get(k) - k);
						}
						finish = false;
						break;
					}
				}
			} while (!finish);
			this.tmpLines.addAll(edges);
		}
	}

	// text rendering
	public void writeText(String text, Vector3 position) {
		Vector3 pos = getCamera().project(position);
		pos.y = getCamera().getScreenWidth() * getCamera().getAspectRatio() - pos.y;
		stringBuilder.append(SvgPrinter.writeText(pos.x, pos.y, 0, text, new Vector3(0, 0, 1)));
	}

	public void writeText(String text, Vector2 onScreenPosition, float rotation, SVG_BORDER border) {
		switch (border) {
		case ROUND:
			stringBuilder.append(SvgPrinter.writeCircle(onScreenPosition, 20, null, new Vector3()));
			break;

		case RECTANGULAR:
			break;
		default:
			break;
		}
		stringBuilder.append(SvgPrinter.writeText(onScreenPosition.x, getCamera().getScreenHeight() - onScreenPosition.y,
				rotation, text, new Vector3(0, 0, 1)));
	}

	// methods for drawing shapes
	public void drawArrow(Vector3 start, Vector3 end, Vector3 color) {
		Vector3 startP = getCamera().project(start);
		Vector3 endP = getCamera().project(end);
		startP.y = getCamera().getScreenWidth() * getCamera().getAspectRatio() - startP.y;
		endP.y = getCamera().getScreenWidth() * getCamera().getAspectRatio() - endP.y;
		stringBuilder
				.append(SvgPrinter.generateArrows(new Vector2(startP.x, startP.y), new Vector2(endP.x, endP.y), color));
	}

	// no transformation is applied the coords must be supplied on screen space
	public void drawArrow(Vector2 start, Vector2 end, Vector3 color) {
//		start.y = camera.getScreenHeight()  - start.y;
//		end.y = camera.getScreenHeight() - end.y;
		stringBuilder.append(SvgPrinter.generateArrows(start, end, color));
	}

	public static void shade(Triangle triangle, Vector3 lightPosition, Vector3 cameraPosition) {
		Vector3 N = triangle.getNormal();
		Vector3 color0 = shade(triangle.getV0(), N, lightPosition, cameraPosition);
		Vector3 color1 = shade(triangle.getV1(), N, lightPosition, cameraPosition);
		Vector3 color2 = shade(triangle.getV2(), N, lightPosition, cameraPosition);

		triangle.setColor(color0.add(color1).add(color2).scl(triangle.getColor()).scl(1/3.0f));

	}

	public static Vector3 shade(Vector3 fragment, Vector3 normal, Vector3 lightPosition, Vector3 cameraPosition) {

		Vector3 ambiant = new Vector3(1, 1, 1).scl(0.1f);
		Vector3 L = lightPosition.cpy().sub(fragment).nor();
		Vector3 V = cameraPosition.cpy().sub(fragment).nor();
		if (normal.dot(V) < 0)
			normal.scl(-1);
		float diffuse = Math.max(normal.dot(L), 0);
		float secular = (float) Math.pow(Math.max(V.cpy().add(L).scl(0.5f).dot(normal), 0), 32);
		return new Vector3(1, 1, 1).scl(diffuse).add(ambiant).add(new Vector3(secular, secular, secular));
	}

	public ProjectionCamera getCamera() {
		return camera;
	}

	public void setCamera(ProjectionCamera camera) {
		this.camera = camera;
	}

	public Vector3 getStrokeColor() {
		return strokeColor;
	}

	public void setStrokeColor(Vector3 strokeColor) {
		this.strokeColor = strokeColor;
	}
	
	

}
