package api.mep;

import java.util.ArrayList;
import java.util.List;
import org.joml.Matrix4f;
import org.joml.Vector3f;
import org.joml.Vector4f;
import org.lwjgl.opengl.GL45;

import com.sandmonkey.ttf_renderer.api.FreeTypeFont;
import com.sandmonkey.ttf_renderer.api.Text;

import api.backend.ApplicationContext;
import api.graphics.BoundingBox;
import api.graphics.Camera;
import api.graphics.ModelInstance;
import api.graphics.geometry.ShapeDrawer;
import api.provider.FontProvider;

public class QuotationDrawer {

	private List<ModelInstance> objects;
	private BoundingBox sceneBoundaries;
	private List<Measure> measures = new ArrayList<Measure>();
	private ShapeDrawer shapeDrawer;
	private Text text;

	
	
	public QuotationDrawer(List<ModelInstance> objects) {
		this.objects = objects;
		shapeDrawer = new ShapeDrawer();
		shapeDrawer.prepare();
		text = new Text();
	}
	
	void calculateSceneBoundaries() {
		if(!objects.isEmpty()) {
			ModelInstance instance = objects.get(0);
			var bbox = instance.getBoundingBox();
			sceneBoundaries = new BoundingBox(bbox.getMin(), bbox.getMax(), new Matrix4f(bbox.getTransform()));
			Matrix4f transform = new  Matrix4f(instance.getTransform()).mul(bbox.getTransform());
			sceneBoundaries.setTransform(transform);
			for(int i = 1; i < objects.size(); i++) {
				instance = objects.get(i);
				BoundingBox other = instance.getBoundingBox().cpy();
				transform = new Matrix4f(instance.getTransform()).mul(other.getTransform());
				other.setTransform(transform);
				sceneBoundaries.extend(other);
			}
		}
	}
	
	public void generateQuoations() {
		calculateSceneBoundaries();
		measures.clear();
		boolean first = true;
		for(ModelInstance instance: objects) {
			if(first) {
				first = false;
				continue;
			}
			Vector4f center4 =  new Vector4f(instance.getCenter(),1).mul(sceneBoundaries.getTransform());
			Vector3f center = new Vector3f(center4.x, center4.y, center4.z);
			float xReference = center.x > sceneBoundaries.getCenter().x? sceneBoundaries.getMax().x: sceneBoundaries.getMin().x;
			float yReference = center.y > sceneBoundaries.getCenter().y? sceneBoundaries.getMax().y: sceneBoundaries.getMin().y;
			float zReference = center.z > sceneBoundaries.getCenter().z? sceneBoundaries.getMax().z: sceneBoundaries.getMin().z;
			Measure xMeasure = new Measure(center, new Vector3f(xReference, center.y, center.z));
			Measure yMeasure = new Measure(center, new Vector3f(center.x, yReference, center.z));
			Measure zMeasure = new Measure(center, new Vector3f(center.x, center.y, zReference));
			measures.add(xMeasure);
			measures.add(yMeasure);
			measures.add(zMeasure);
		}
	}
	
	public void draw(Camera camera) {
		for(Measure mesure: measures) {
			Vector4f v0 = new Vector4f(mesure.getV0(), 1).mul(sceneBoundaries.getTransform());
			Vector4f v1 = new Vector4f(mesure.getV1(), 1).mul(sceneBoundaries.getTransform());
			Vector3f[] lines = {new Vector3f(v0.x, v0.y, v0.z), new Vector3f(v1.x, v1.y, v1.z)};
			GL45.glDepthFunc(GL45.GL_ALWAYS);
			shapeDrawer.begin(camera);
			shapeDrawer.drawLines(lines, 1.5f, new Vector3f(0,1,0), new Matrix4f());
			shapeDrawer.end();
			GL45.glDepthFunc(GL45.GL_LEQUAL);

			//render text
			float[] viewport =  ApplicationContext.getViewport();
			Matrix4f projection = new Matrix4f().ortho2D(0, viewport[2], 0, viewport[3]);
		    FreeTypeFont.getShader().use();
		    FreeTypeFont.getShader().setMat4("projection", projection);
		    
    		 var screenPoint0 = camera.project(lines[0], ApplicationContext.getViewport()).mul(1, 1, 0);
    		 var screenPoint1 = camera.project(lines[1], ApplicationContext.getViewport()).mul(1, 1 , 0);
    		 if(screenPoint0.distance(screenPoint1) > 100) {
    			 Vector3f v = new Vector3f(screenPoint1).sub(screenPoint0);
        		 float rotation = (float) Math.toDegrees(Math.atan2(v.y, v.x));
        		 if(rotation > 90)
        			 rotation += 180;
        		 var center = screenPoint0.add(screenPoint1).div(2);
        		 String measure = String.format("%.1f cm", v0.distance(v1) * 100);
    	 		 text.render(FontProvider.getProvider().getLargeFont(), measure, center.x , center.y, 1, rotation , new Vector3f(0,0,1),camera);
    		 }
    		
		}
	}
	
}
