package dressing.mathutils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;

import org.apache.lucene.util.MathUtil;
import org.eclipse.swt.graphics.GCData;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.OrthographicCamera;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.scenes.scene2d.Stage;
import com.badlogic.gdx.utils.GdxNativesLoader;

import dressing.graphics.shapes2d.Line;
import gdxapp.screens.room.RoomController;
import geometry.Box;

public class MathUtilities {

	public static Random rnd;

	public static final Matrix3 satgeToWorldTransform = new Matrix3(new float[] { 0, 1, 0, 0, 0, 0, 1, 0, 0 });
	public static final Matrix3 worldToStageTransform = new Matrix3(new float[] { 0, 0, 1, 1, 0, 0, 0, 1, 0 });

	public static final Matrix4 screenToWorldTransform = new Matrix4(
			new float[] { 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1 });

	public static Vector2 toVector2(Matrix23 projection, Vector3 vector) {
		return projection.mult(vector);

	}

	public static Vector2 bezierQuadratic(Vector2 P0, Vector2 P1, Vector2 P2, float t) {
		float a0 = (float) Math.pow(1 - t, 2);
		float a1 = 2 * t * (1 - t);
		float a2 = (float) Math.pow(t, 2);
		Vector2 v0 = P0.cpy().scl(a0);
		Vector2 v1 = P1.cpy().scl(a1);
		Vector2 v2 = P2.cpy().scl(a2);
		return v0.add(v1).add(v2);
	}

	public static void round(Vector3 vector, float precision) {
		double x = vector.x * Math.pow(10, precision);
		double y = vector.y * Math.pow(10, precision);
		double z = vector.z * Math.pow(10, precision);
		x = Math.round(x);
		y = Math.round(y);
		z = Math.round(z);
		x *= Math.pow(10, -precision);
		y *= Math.pow(10, -precision);
		z *= Math.pow(10, -precision);
		vector.set((float) x, (float) y, (float) z);
	}

	// rotate a vector u by a certain angle provided in degree
	public static Vector2 getDeviateVector(Vector2 origin, Vector2 u, float angle, float scale) {
		float angleR = (float) Math.toRadians(angle);
		Vector2 v = new Vector2();
		float x = (float) (u.x * Math.cos(angleR) - u.y * Math.sin(angleR));
		float y = (float) (u.y * Math.cos(angleR) + u.x * Math.sin(angleR));
		v.set(-x, -y).nor().scl(scale);
		v.add(origin);
		return v;
	}

	public static float getRotation(Vector2 v) {
		return (float) Math.toDegrees(Math.atan2(v.x, v.y));
	}

	public static float getUnsignedAngle(Vector3 a, Vector3 b) {
		double angle = Math.acos(a.dot(b) / (a.len() * b.len()));
		return (float) Math.abs(Math.toDegrees(angle));
	}

	// return the signed angle following a counter clock wise sense
	public static float signedAngle(Vector2 a, Vector2 b) {
		return (float) Math.atan2((b.y * a.x - b.x * a.y), (b.x * a.x + b.y * a.y));
	}

	public static float getSignedAngle(Vector3 a, Vector3 b, Vector3 normal) {
		float cosin = a.dot(b) / (a.len() * b.len());
		double angle = Math.acos(cosin);
		Vector3 n = a.cpy().crs(b);
		if (n.hasOppositeDirection(normal))
			angle *= -1;
		return (float) Math.toDegrees(angle);
	}

	public static Vector2 rotate(Vector2 origin, Vector2 point, float degree) {
		Vector2 m = point.cpy().sub(origin);
		double angle = Math.toRadians(degree);
		Vector2 image = new Vector2();
		image.x = (float) (Math.cos(angle) * m.x - Math.sin(angle) * m.y);
		image.y = (float) (Math.sin(angle) * m.x + Math.cos(angle) * m.y);
		image.x = Math.round(image.x * 10000) / 10000;
		image.y = Math.round(image.y * 10000) / 10000;
		image.add(origin);
		return image;
	}
	//compute the sigend area of polygon which is equal to half
	//of the sum of each vertex exterior product with the next one
	public static float signedArea(Vector2... vertices) {
		float sum = 0;
		ArrayList<Vector2> verticesList = new ArrayList<Vector2>();
		verticesList.addAll(Arrays.asList(vertices));
		verticesList.add(vertices[0]);
		for(int i = 0 ; i < vertices.length; i++) {
			sum += verticesList.get(i).crs(verticesList.get(i+1));
		}
		return sum / 2.0f;
	}

	public static Vector3 rotate(Vector3 origin, Vector3 point, float degree) {
		point.x -= origin.x;
		point.y -= origin.y;
		double angle = Math.toRadians(degree);
		Vector3 image = new Vector3();
		image.x = (float) (Math.cos(angle) * point.x - Math.sin(angle) * point.y);
		image.y = (float) (Math.sin(angle) * point.x + Math.cos(angle) * point.y);
		image.x = Math.round(image.x * 10000) / 10000;
		image.y = Math.round(image.y * 10000) / 10000;
		image.x += origin.x;
		image.y += origin.y;
		image.z = point.z;
		return image;
	}

	public static Vector2 middlepoint(Vector2 point1, Vector2 point2) {
		float x = Math.abs(point1.x - point2.x) / 2 + Math.min(point1.x, point2.x);
		float y = Math.abs(point1.y - point2.y) / 2 + Math.min(point1.y, point2.y);
		return new Vector2(x, y);
	}

	public static float calculateUnsignedAngle(Vector2 u, Vector2 v) {
		float cos = u.dot(v) / (u.len() * v.len());
		if (cos > 1.0f) {
			cos = 1.0f;
		}
		if (cos < -1.0f) {
			cos = -1.0f;
		}
		double angle = (float) Math.acos(cos);
		return (float) angle;
	}

	public static float[] solveQuadraticEquation(float a, float b, float c) {
		float[] solutions = new float[2];
		double delta = Math.pow(b, 2) - 4 * a * c;
		if (delta >= 0) {
			delta = Math.sqrt(delta);
			double x1 = -b - delta;
			double x2 = -b + delta;
			x1 /= 2 * a;
			x2 /= 2 * a;
			solutions[0] = (float) x1;
			solutions[1] = (float) x2;
		}
		return solutions;
	}

	// calculate the parametric equation of a line passing by 2 points a and b
	public static float[] calculateLineEquation(Vector2 p1, Vector2 p2) {
		Vector2 director = p2.cpy().sub(p1);
		Vector2 normal = new Vector2(-director.y, director.x);
		float[] params = new float[3];
		params[0] = normal.x;
		params[1] = normal.y;
		params[2] = -p1.dot(normal);
		return params;
	}

	// calculate the distance between a point and a line
	public static float distance(Vector3 point, Vector3 start, Vector3 end) {
		Vector3 pm = projectPoint(point, start, end);
		if (lineSegmentContainspoint(pm, start, end, 0.01f)) {
			return pm.sub(point).len();
		} else {
			Vector3 ps = point.cpy().sub(start);
			Vector3 pe = point.cpy().sub(end);
			return Math.min(ps.len(), pe.len());
		}
	}

	public static float distance(Vector3 point, Line line) {
		Vector3 v0 = new Vector3(line.getV0(), 0);
		Vector3 v1 = new Vector3(line.getV1(), 0);
		return distance(point, v0, v1);
	}

	public static Vector3 projectPoint(Vector3 point, Vector3 start, Vector3 end) {
		Vector3 se = end.cpy().sub(start);
		Vector3 sm = point.cpy().sub(start);
		float a = sm.dot(se);
		float b = -((sm.x * se.y) - (sm.y * se.x));
		float sinusOfAngle = b / new Vector2(a, b).len();
		if (!Float.isNaN(sinusOfAngle)) {
			float angle = (float) Math.toDegrees(Math.asin(sinusOfAngle));
			if (se.dot(sm) < 0)
				angle = 180 - angle;
			Vector3 projection = se.cpy().scl((float) (Math.cos(Math.toRadians(angle)) * sm.len() / se.len()))
					.add(start);
			return projection;
		}
		return point;

	}

	public static Vector2 projectPoint(Vector2 point, Vector2 start, Vector2 end) {
		Vector2 se = new Vector2(end).sub(start);
		Vector2 sm = new Vector2(point).sub(start);
		float norm = sm.dot(se) / se.len();
		return new Vector2(start).add(se.nor().scl(norm));
	}

	public static Vector3 lineParameters(Vector2 origin, Vector2 director) {
		Vector2 normal = new Vector2(-director.y, director.x);
		Vector3 params = new Vector3();
		params.x = normal.x;
		params.y = normal.y;
		params.z = -normal.dot(origin);
		return params;
	}

	public static Vector2 getIntersectionPoint(Edge edge_1, Edge edge_2) {
		Vector3 params_1 = lineParameters(edge_1.getV0(), edge_1.getDirector());
		Vector3 params_2 = lineParameters(edge_2.getV0(), edge_2.getDirector());
		Matrix2 A = new Matrix2(new float[] { params_1.x, params_1.y, params_2.x, params_2.y, });
		Vector2 B = new Vector2(-params_1.z, -params_2.z);
		return A.cpy().inv().mul(B);
	}

	public static Vector2 getIntersectionPoint(Vector3 lineParams1, Vector3 lineParams2) {
		Matrix2 A = new Matrix2(new float[] { lineParams1.x, lineParams1.y, lineParams2.x, lineParams2.y, });
		Vector2 B = new Vector2(-lineParams1.z, -lineParams2.z);
		return A.cpy().inv().mul(B);
	}

	public static Vector2 rotate(Vector2 a, float alpha) {
		Matrix2 rot = new Matrix2(new float[] { (float) Math.cos(alpha), (float) -Math.sin(alpha),
				(float) Math.sin(alpha), (float) Math.cos(alpha) });
		return rot.mul(a);
	}

	public static boolean lineSegmentContainspoint(Vector3 point, Vector3 start, Vector3 end, float precision) {
		Vector3 sp = point.cpy().sub(start);
		Vector3 ep = point.cpy().sub(end);
		Vector3 se = end.cpy().sub(start);
		float sumOfDistance = sp.len() + ep.len();

		return (Math.abs(sumOfDistance - se.len()) < precision);
	}

	public static Vector3[] getBoundaries(Vector3... vertices) {
		float minX = Float.NaN, minY = Float.NaN, maxX = Float.NaN, maxY = Float.NaN, minZ = Float.NaN,
				maxZ = Float.NaN;
		for (Vector3 vertex : vertices) {
			if (Float.isNaN(minX)) {
				minX = vertex.x;
				minY = vertex.y;
				maxX = vertex.x;
				maxY = vertex.y;
				minZ = vertex.z;
				maxZ = vertex.z;
				continue;
			}
			if (vertex.x < minX)
				minX = vertex.x;
			if (vertex.x > maxX)
				maxX = vertex.x;
			if (vertex.y < minY)
				minY = vertex.y;
			if (vertex.y > maxY)
				maxY = vertex.y;
			if (vertex.z < minZ)
				minZ = vertex.z;
			if (vertex.z > maxZ)
				maxZ = vertex.z;
		}
		return new Vector3[] { new Vector3(minX, minY, minZ), new Vector3(maxX, maxY, maxZ) };
	}

	public static Vector3[] getBoundaries(List<Vector3> vertices) {
		return getBoundaries(vertices.toArray(new Vector3[0]));
	}

	public static Vector2[] getBoundariesPlane(List<Vector2> vertices) {
		float minX = Float.NaN, minY = Float.NaN, maxX = Float.NaN, maxY = Float.NaN;
		for (Vector2 vertex : vertices) {
			if (Float.isNaN(minX)) {
				minX = vertex.x;
				minY = vertex.y;
				maxX = vertex.x;
				maxY = vertex.y;
				continue;
			}
			if (vertex.x < minX)
				minX = vertex.x;
			if (vertex.x > maxX)
				maxX = vertex.x;
			if (vertex.y < minY)
				minY = vertex.y;
			if (vertex.y > maxY)
				maxY = vertex.y;
		}
		return new Vector2[] { new Vector2(minX, minY), new Vector2(maxX, maxY) };
	}

	public static Vector3 calculateCenter(ArrayList<Vector3> vertices) {
		Vector3[] boundaries = getBoundaries(vertices);
		return boundaries[0].add(boundaries[1]).scl(0.5f);
	}

	public static Vector4 calculateContainerBoxDimension(ArrayList<Vector3> vertices) {
		Vector3[] boundaries = getBoundaries(vertices);
		Vector3 diameter = boundaries[1].sub(boundaries[0]);
		float x = Math.abs(diameter.x);
		float y = Math.abs(diameter.y);
		float z = Math.abs(diameter.z);
		return new Vector4(x, y, z, 0);
	}

	public static Vector2 screenToStageCoords(Stage stage, float screenX, float screenY) {
		OrthographicCamera camera = (OrthographicCamera) stage.getCamera();
		float originX = (float) (Gdx.graphics.getWidth() * (1 - 1 / camera.zoom) / 2);
		float originY = (float) (Gdx.graphics.getHeight() * (1 - 1 / camera.zoom) / 2);

		float x = screenX - originX;
		float y = Gdx.graphics.getHeight() - screenY - originY;
		x *= camera.zoom;
		y *= camera.zoom;

		x = Math.round(x);
		y = Math.round(y);
		return new Vector2(x, y);
	}

	public static float distance(Vector2 point, Vector2 v0, Vector2 v1) {
		Vector3 point3 = new Vector3(point, 0);
		Vector3 v30 = new Vector3(v0, 0);
		Vector3 v31 = new Vector3(v1, 0);
		return distance(point3, v30, v31);
	}

	public static Vector2 closest(Vector2 point, ArrayList<Vector2> targets) {
		if (targets != null && targets.size() > 0) {
			Vector2 nearest = targets.get(0);
			for (Vector2 measurepoint : targets) {
				if (measurepoint.dst(point) < nearest.dst(point))
					nearest = measurepoint;
			}
			return nearest;
		}
		return null;
	}

	/**
	 * distance between a point and a line dined with 2 points start and end
	 * 
	 * @param x  targetPoint x index
	 * @param y  targetPoint y index
	 * @param x1 segment start point x index
	 * @param y1 segment start point y index
	 * @param x2 segment end point x index
	 * @param y2 segment end point y index
	 * @return
	 */
	public static float lineToPointDistance(float x, float y, float x1, float y1, float x2, float y2) {

		float A = x - x1;
		float B = y - y1;
		float C = x2 - x1;
		float D = y2 - y1;

		float dot = A * C + B * D;
		float len_sq = C * C + D * D;
		float param = -1;
		if (len_sq != 0) // in case of 0 length line
			param = dot / len_sq;

		float xx, yy;

		if (param < 0) {
			xx = x1;
			yy = y1;
		} else if (param > 1) {
			xx = x2;
			yy = y2;
		} else {
			xx = x1 + param * C;
			yy = y1 + param * D;
		}

		float dx = x - xx;
		float dy = y - yy;
		return (float) Math.sqrt(dx * dx + dy * dy);
	}

	/**
	 * tester si le point donnée est inclus dans le le segment difinée par deux
	 * point
	 * 
	 * @param x  targetPoint x index
	 * @param y  targetPoint y index
	 * @param x1 segment start point x index
	 * @param y1 segment start point y index
	 * @param x2 segment end point x index
	 * @param y2 segment end point y index
	 * @return
	 */

	public static boolean iscontains(float x, float y, float x1, float y1, float x2, float y2) {
		float A = x - x1;
		float B = y - y1;
		float C = x2 - x1;
		float D = y2 - y1;

		float dot = A * C + B * D;
		float len_sq = C * C + D * D;
		float param = -1;
		if (len_sq != 0) // in case of 0 length line
			param = dot / len_sq;

		if (param >= 0 && param <= 1) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * 
	 * @param x  targetPoint x index
	 * @param y  targetPoint y index
	 * @param x1 segment start point x index
	 * @param y1 segment start point y index
	 * @param x2 segment end point x index
	 * @param y2 segment end point y index
	 * @return return the closest point on the segment to the target point
	 */
	public static Vector2 getclosestPoint(float x, float y, float x1, float y1, float x2, float y2) {

		float A = x - x1;
		float B = y - y1;
		float C = x2 - x1;
		float D = y2 - y1;

		float dot = A * C + B * D;
		float len_sq = C * C + D * D;
		float param = -1;
		if (len_sq != 0) // in case of 0 length line
			param = dot / len_sq;

		float xx, yy;

		if (param < 0) {
			xx = x1;
			yy = y1;
		} else if (param > 1) {
			xx = x2;
			yy = y2;
		} else {
			xx = x1 + param * C;
			yy = y1 + param * D;
		}

		return new Vector2(xx, yy);
	}

	public static Matrix4 gbvvbv(Vector3 v0, Vector3 v1, Vector3 v2) {
		Vector3 T = v1.cpy().sub(v0).nor();
		Vector3 B = v2.cpy().sub(v0);
		Vector3 N = T.cpy().crs(B.cpy()).nor();
		B = N.cpy().crs(T);
		Matrix4 TBN = new Matrix4().set(T, B, N, new Vector3(0, 0, 0));
		return TBN;
	}
	public static Matrix4 calculateTBN(Vector3 v0, Vector3 v1, Vector3 v2) {
		Vector3 T = v1.cpy().sub(v0).nor();
		Vector3 B = v2.cpy().sub(v0);
		Vector3 N = T.cpy().crs(B.cpy()).nor();
		B = N.cpy().crs(T);
		return new Matrix4().set(T, B, N, new Vector3());
	}
	
	public static Matrix4 calculateTBN(Vector3 v0, Vector3 v1, Vector3 v2, Vector3 origin) {
		Vector3 T = v1.cpy().sub(v0).nor();
		Vector3 B = v2.cpy().sub(v0);
		Vector3 N = T.cpy().crs(B.cpy()).nor();
		B = N.cpy().crs(T);
		return new Matrix4().set(T, B, N, origin.cpy().scl(-1));
	}

	public static ArrayList<Vector2> calculateSurfaceContour(ArrayList<Vector2> vertices, boolean closed,
			float thickness) {
		ArrayList<Vector2> normals = new ArrayList<Vector2>();
		Vector2 nextVertex, previousVertex;
		for (Vector2 vertex : vertices) {
			int i = vertices.indexOf(vertex);
			int next;
			int previous;
			if (closed) {
				next = (i + 1) % (vertices.size());
				previous = i - 1;
				if (previous < 0)
					previous += vertices.size();
				nextVertex = vertices.get(next).cpy();
				previousVertex = vertices.get(previous).cpy();
			} else {
				nextVertex = vertex.cpy();
				previousVertex = vertex.cpy();
				next = (i + 1);
				if (next < vertices.size()) {
					nextVertex = vertices.get(next).cpy();
				}
				previous = i - 1;
				if (previous >= 0) {
					previousVertex = vertices.get(previous).cpy();
				}
			}
			nextVertex.sub(vertex).nor();
			previousVertex.sub(vertex).nor();
			float angle = calculateUnsignedAngle(nextVertex, previousVertex);
			angle = (float) ((Math.toRadians(180) - angle) / 2.0f);
			float scale = thickness;
			if (!Float.isNaN(angle))
				scale /= (float) (Math.cos(angle));
			Vector2 normal = nextVertex.cpy().sub(previousVertex).rotate(90).nor().scl(scale).add(vertex);
			normals.add(normal);
		}
		return normals;
	}

	public static ArrayList<Vector2> calculateSurfaceContour(ArrayList<Vector2> vertices, ArrayList<Edge> edges,
			float[] thickness) {
		ArrayList<Vector2> normals = new ArrayList<Vector2>();
		int size = vertices.size();
		ArrayList<Edge> vertexEdges = new ArrayList<Edge>();
		for (int i = 0; i < size; i++) {
			Vector2 vertex = vertices.get(i);
			vertexEdges.clear();
			for (int j = 0; j < edges.size(); j++) {
				if (edges.get(j).contains(vertex)) {
					vertexEdges.add(edges.get(j));
				}
			}
			int index1 = edges.indexOf(vertexEdges.get(0));
			if (vertexEdges.size() == 2) {
				Vector2 origin = vertexEdges.get(0).getMiddle().cpy()
						.add(vertexEdges.get(0).getNormal().scl(-thickness[index1]));
				Vector3 line1 = lineParameters(origin, vertexEdges.get(0).getDirector());
				int index2 = edges.indexOf(vertexEdges.get(1));
				Vector2 origin2 = vertexEdges.get(1).getMiddle().cpy()
						.add(vertexEdges.get(1).getNormal().scl(-thickness[index2]));
				Vector3 line2 = lineParameters(origin2, vertexEdges.get(1).getDirector());
				Vector2 intersection = getIntersectionPoint(line1, line2);
				normals.add(intersection);
			} else {
				normals.add(vertex.cpy().add(vertexEdges.get(0).getNormal().scl(-thickness[index1])));
			}

		}
		return normals;
	}

	public static void calculateRecline(ArrayList<Vector2> line, float depth) {
		ArrayList<Vector2> outerLine = new ArrayList<Vector2>();
		int pointsCount = line.size();
		for (int i = 0; i < pointsCount; i++) {
			int previous = i - 1;
			int next = i + 1;
			Vector2 current = line.get(i);
			Vector2 vn = (next < pointsCount) ? line.get(next).cpy().sub(current).rotate(-90).nor() : null;
			Vector2 vp = (previous >= 0) ? current.cpy().sub(line.get(previous)).rotate(-90).nor() : null;
			if (vp != null)
				outerLine.add(vp);
			if (vn != null)
				outerLine.add(vn);
		}
		for (int i = line.size() - 1; i > -1; i--)
			outerLine.add(line.get(i));
		line.clear();
		line.addAll(outerLine);
	}

	public static ArrayList<Vector2> calculateRectangulairLine(ArrayList<Vector2> vertices, float halfThickness) {
		ArrayList<Vector2> rectangulairLine = new ArrayList<Vector2>();
		if (vertices.size() == 2) {
			Vector2 normal = vertices.get(1).cpy().sub(vertices.get(0)).rotate(90).nor().scl(halfThickness);
			rectangulairLine.add(new Vector2().set(vertices.get(0)).add(normal));
			rectangulairLine.add(new Vector2().set(vertices.get(1)).add(normal));
			rectangulairLine.add(new Vector2().set(vertices.get(1)).add(normal.cpy().scl(-1)));
			rectangulairLine.add(new Vector2().set(vertices.get(0)).add(normal.cpy().scl(-1)));
		} else {
			ArrayList<Vector2> interiorPoints = new ArrayList<Vector2>();
			Vector2 n0 = vertices.get(1).cpy().sub(vertices.get(0)).nor().rotate(90).scl(halfThickness);
			rectangulairLine.add(new Vector2().set(vertices.get(0)).add(n0));
			interiorPoints.add(new Vector2().set(vertices.get(0)).add(n0.scl(-1)));
			for (int i = 1; i < vertices.size() - 1; i++) {
				int nextIndex = (i + 1) % vertices.size();
				int previousIndex = (i - 1) % vertices.size();
				if (previousIndex < 0)
					previousIndex += vertices.size();
				Vector2 n = vertices.get(nextIndex).cpy().sub(vertices.get(i)).nor();
				Vector2 p = vertices.get(previousIndex).cpy().sub(vertices.get(i)).nor();
				Vector2 b = n.cpy().add(p).nor();
				float angle = (p.angle() - n.angle()) * 0.5f;
				float scale = (float) (halfThickness / Math.sin(Math.toRadians(angle)));
				b.scl(scale);
				float sign = b.dot(n.cpy().rotate(90));
				if (sign < 0)
					b.scl(-1);
				rectangulairLine.add(new Vector2().set(vertices.get(i)).add(b));
				interiorPoints.add(new Vector2().set(vertices.get(i)).add(b.scl(-1)));
			}
			Vector2 nl = vertices.get(vertices.size() - 1).cpy().sub(vertices.get(vertices.size() - 2)).nor().rotate(90)
					.scl(halfThickness);
			rectangulairLine.add(new Vector2().set(vertices.get(vertices.size() - 1)).add(nl));
			interiorPoints.add(new Vector2().set(vertices.get(vertices.size() - 1)).add(nl.scl(-1)));
			for (int j = interiorPoints.size() - 1; j > -1; j--) {
				rectangulairLine.add(interiorPoints.get(j));
			}
		}

		return rectangulairLine;
	}

	// return the float array containing the vertices and fill the indices array
	public static float[] indexMesh(ArrayList<Triangle> triangles, final short[] indices, float[] uvRange,
			float color) {
		ArrayList<Vector2> uniqueVertices = new ArrayList<Vector2>();
		// create the list of unique vertices
		for (Triangle triangle : triangles) {
			Vector2[] vertices = new Vector2[] { triangle.getV0(), triangle.getV1(), triangle.getV2() };
			for (Vector2 vertex : vertices) {
				boolean exist = false;
				for (Vector2 vertexX : uniqueVertices) {
					if (vertexX.epsilonEquals(vertex)) {
						exist = true;
						break;
					}
				}
				if (!exist) {
					uniqueVertices.add(vertex);
				}
			}
		}
		float uRange = uvRange[1] - uvRange[0];
		float vRange = uvRange[3] - uvRange[2];
		Vector2 minBoundary = getBoundariesPlane(uniqueVertices)[0];
		Vector2 maxBoundary = getBoundariesPlane(uniqueVertices)[1];

		float uScale = uRange / (maxBoundary.x - minBoundary.x);
		float vScale = vRange / (maxBoundary.y - minBoundary.y);

		float[] vertices = new float[uniqueVertices.size() * 5];
		int index = 0;
		int c = 0;
		Vector2 uv = new Vector2();
		for (int i = 0; i < uniqueVertices.size(); i++) {
			uv.set(uniqueVertices.get(i).x, uniqueVertices.get(i).y);
			uv.sub(minBoundary);
			uv.scl(uScale, vScale);
			uv.add(uvRange[0], uvRange[2]);
			index = 5 * i;
			c = 0;
			vertices[index + c] = uniqueVertices.get(i).x;
			vertices[index + ++c] = uniqueVertices.get(i).y;
			vertices[index + ++c] = color;
			vertices[index + ++c] = uv.x;
			vertices[index + ++c] = uv.y;
		}
		index = 0;
		for (int i = 0; i < triangles.size(); i++) {
			c = 0;
			index = i * 3;
			indices[index + c] = getIndexOf(triangles.get(i).getV0(), uniqueVertices);
			indices[index + ++c] = getIndexOf(triangles.get(i).getV1(), uniqueVertices);
			indices[index + ++c] = getIndexOf(triangles.get(i).getV2(), uniqueVertices);
		}
		return vertices;
	}

	private static short getIndexOf(Vector2 vertex, ArrayList<Vector2> verticesSet) {
		short index = -1;
		for (short i = 0; i < verticesSet.size(); i++) {
			if (verticesSet.get(i).epsilonEquals(vertex)) {
				index = i;
				break;
			}
		}
		return index;
	}

	// extension of the floor mod function to the float set
	public static float floorMod(float x, float y) {
		int q = (int) Math.floor(x / y);
		return x - q * y;
	}

	public static boolean isPointInsideConvexPolygon(Vector2 point, List<Vector2> polygonVertices) {
		ArrayList<Vector2> vectors = new ArrayList<Vector2>();
		for (Vector2 vertex : polygonVertices) {
			vectors.add(new Vector2(vertex.cpy().sub(point)));
		}
		float anglesSum = 0;
		for (int i = 0; i < vectors.size(); i++) {
			anglesSum += calculateUnsignedAngle(vectors.get(i), vectors.get((i + 1) % vectors.size()));
		}
		if (Math.abs(anglesSum - Math.toRadians(360)) > 0.001)
			return false;
		return true;
	}

	public static int gcd(int... numbers) {
		int size = numbers.length;
		int lastGCD = 1;
		for (int i = 2; i <= Math.min(numbers[size - 1], numbers[size - 2]); i++) {
			if (numbers[size - 1] % i == 0 && numbers[size - 2] % i == 0)
				lastGCD = i;
		}
		if (size > 2) {
			int[] nums = new int[size - 1];
			for (int i = 0; i < nums.length - 1; i++) {
				nums[i] = numbers[i];
			}
			nums[size - 2] = lastGCD;
			return gcd(nums);
		} else {
			return lastGCD;
		}
	}

	/**
	 * 
	 * @param point    the contained point
	 * @param vertices the vertices that represented the boundries
	 * @return
	 */
	public static boolean isBoxContainsPoint(Vector2 point, Vector2 v0, Vector2 v1, Vector2 v2, Vector2 v3) {
		if (point != null) {
			if (!iscontains(point.x, point.y, v0.x, v0.y, v1.x, v1.y)) {
				return false;
			}
			if (!iscontains(point.x, point.y, v1.x, v1.y, v2.x, v2.y)) {
				return false;
			}
			if (!iscontains(point.x, point.y, v2.x, v2.y, v3.x, v3.y)) {
				return false;
			}
			if (!iscontains(point.x, point.y, v3.x, v3.y, v0.x, v0.y)) {
				return false;
			}
			return true;
		}

		return false;
	}

	/**
	 * @param containedBox
	 * @param v00
	 * @param v01
	 * @param v02
	 * @param v03
	 * @param containingBox
	 * @param v10
	 * @param v11
	 * @param v12
	 * @param v13
	 * @return true @containingBox contains @containedBox
	 */
	public static boolean isBoxContainsBox(Vector2 v00, Vector2 v01, Vector2 v02, Vector2 v03, Vector2 v10, Vector2 v11,
			Vector2 v12, Vector2 v13) {

		if (!isBoxContainsPoint(v00, v10, v11, v12, v13)) {
			return false;
		}
		if (!isBoxContainsPoint(v01, v10, v11, v12, v13)) {
			return false;
		}
		if (!isBoxContainsPoint(v02, v10, v11, v12, v13)) {
			return false;
		}
		if (!isBoxContainsPoint(v03, v10, v11, v12, v13)) {
			return false;
		}

		return true;
	}

	public static void recursiveExtrusion(Box box, ArrayList<Box> boxes, int max) {
		Vector3 dims = box.getDimensions();
		float size = Math.min(Math.min(dims.x, dims.y), dims.z);
		size = Math.min(0.1f, size);
		float maxDims = Math.max(dims.x / 2.0f, Math.max(dims.y / 2.0f, dims.z / 2.0f));
		if (size > 0.02f) {
			Box extrusion = new Box(box.getCenter(), new Vector3(size, size, size));
			boxes.add(new Box(extrusion.getCenter(), extrusion.getDimensions()));
			ArrayList<Box> children = box.extract(extrusion);
			int d = max - 1;
			for (Box child : children) {
				recursiveExtrusion(child, boxes, d);
			}
		} else {
			// boxes.add(box);
		}
	}

	public static boolean boxIsValid(Box box) {
		Vector3 dims = box.getDimensions();
		if (Math.abs(dims.x) < 0.0001f)
			return false;
		if (Math.abs(dims.y) < 0.0001f)
			return false;
		if (Math.abs(dims.z) < 0.0001f)
			return false;
		return true;
	}

	public int getWinding(Vector3... points) {
		Vector3[] bounds = getBoundaries(points);
		Vector3 center = bounds[0].cpy().add(bounds[1]).scl(0.5f);
		Vector3 u = points[0].cpy().sub(center);
		Vector3 v = points[1].cpy().sub(center);
		Vector3 w = u.cpy().crs(v);

		return 0;

	}

	public static int getWinding(Vector2... points) {
		if (points.length < 3) {
			return 0;
		}
		float signedArea = signedArea(points);
		System.out.println("signed area of polygon = " + signedArea);
		return Math.round(Math.signum(signedArea));
	}

	public static void setWinding(List<Vector2> vertices, int winding) {
		Vector2[] array = new Vector2[vertices.size()];
		int currentWinding = getWinding(vertices.toArray(array));
		if (currentWinding == winding)
			return;
		vertices.clear();
		for (int i = array.length - 1; i > -1; i--)
			vertices.add(array[i]);
	}

	public static Matrix4 baseChangeMatrix(Vector3 newOrigin, Vector3... axis) {
		return new Matrix4().set(axis[0], axis[1], axis[2], newOrigin).inv();
	}
	

	static {
		GdxNativesLoader.load();
	}
	public static void main(String[] args) {


	}
}
