package reporting;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;

import com.badlogic.gdx.math.Vector2;
import dressing.model.DesignException;
import dressing.model.Piece2D;
import dressing.cam.model.CamShape;
import dressing.cam.model.Cercle2D;
import dressing.cam.model.Cuboid;

import dressing.cam.model.PlanUsinage;
import dressing.cam.model.Util;
import dressing.model.usinage.Usinage;

public class CotationManager {
	
	private Piece2D piece;
	private Vector2 pieceCenter;
	private PlanUsinage face;
	private CotationPreferences preferences;
	public final ArrayList<Cotation> cotations = new ArrayList<Cotation>();
	public HashMap<Cercle2D, ArrayList<Cotation>> diameterCotations = new HashMap<Cercle2D, ArrayList<Cotation>>();
	public ArrayList<CamShape> shapes;
	public final String DIAMETER_HTML_CODE = "&#8960";
	
	public ArrayList<Cotation> generateCotationForProjection(Piece2D piece, PlanUsinage face, CotationPreferences prefs){
		init(piece, face);
		this.preferences = prefs==null?new CotationPreferences():prefs;
		prepareShapes();
		addPieceDimensionCotation(face);
		generateHoleCotations();
		filter();
		cotations.addAll(selectDiamterCotations());
		updateCotationText();
		return cotations;
	}

	private void updateCotationText() {
		for(Cotation cot: cotations) {
			if(cot.getLines().size() > 1) {
				
			}else {
				Line line = cot.getLines().get(0);
				Vector2 middle = line.getMiddle();
				if(middle.x > pieceCenter.x * 2 || middle.y > pieceCenter.y * 2) {
					line = line.getInversed();
				}
				float rot = line.getDirection().angle();
				Vector2 normal = line.getNormal().nor();
				normal.scl(-Math.signum(normal.dot(pieceCenter.cpy().sub(middle))));
				cot.setTextPos(middle.add(normal.scl(5)));
				cot.setTextRot(-rot);
			}
		}
	}

	private ArrayList<Cotation> selectDiamterCotations() {
		removeOverlappingWithDistanceCot();
		HashMap<Cercle2D, Cotation> cotationsMap = new HashMap<Cercle2D, Cotation>();
		ArrayList<Cotation> chosenCotations = new ArrayList<Cotation>();
		do {
			ArrayList<Cercle2D> firstDegreeGroups = new ArrayList<Cercle2D>();
			for(Cercle2D circle: diameterCotations.keySet()) {
				ArrayList<Cotation> group = diameterCotations.get(circle);
				if(group.size() == 1) {
					firstDegreeGroups.add(circle);
					chosenCotations.add(group.get(0));
				}
			}
			//remove single cotation groups
			for(Cercle2D circle: firstDegreeGroups) {
				diameterCotations.remove(circle);
			}
			for(Cercle2D circle: diameterCotations.keySet()) {
				ArrayList<Cotation> group = diameterCotations.get(circle);
				int[] score = new  int[]{0,0,0,0};
				for(int i = 0; i < group.size(); i++) {
					Cotation cot = group.get(i);
					for(Cotation chosen: chosenCotations) {
						if(cot.overlap(chosen)) {
							score[i] = score[i]++;
						}
					}
				}	
				int worstIndex = 0;
				int worst = score[0];
				for(int i = 1; i < 4; i++) {
					if(score[i] > worst)
						worstIndex = i;
				}
				group.remove(worstIndex);
			}
		}while(!diameterCotations.isEmpty());
		return chosenCotations;
	}
	
	private void removeOverlappingWithDistanceCot() {
		ArrayList<Line> lines = new ArrayList<Line>();//hold every line segment in the image
		//add piece boundaries
		lines.add(new Line(new Vector2(), pieceCenter.cpy().scl(2,0), false));
		lines.add(new Line(pieceCenter.cpy().scl(2,0), pieceCenter.cpy().scl(2), false));
		lines.add(new Line(pieceCenter.cpy().scl(2), pieceCenter.cpy().scl(0,2), false));
		lines.add(new Line(pieceCenter.cpy().scl(0,2), new Vector2(), false));		
		for(Cotation cot: cotations) {
			if(cot == null)
				continue;
			lines.addAll(cot.getLines());
		}
		for(Cercle2D cercle: diameterCotations.keySet()) {
			ArrayList<Cotation> choices = diameterCotations.get(cercle);
			boolean repeat = false;
			do {
				repeat = false;
				for(Cotation cot: choices) {
					boolean remove = false;
					for(Line lineX: lines) {
						if(cot.overlap(lineX)) {
							remove = true;
							break;
						}
					}
					if(remove) {
						choices.remove(cot);
						repeat = choices.size() > 1;
						break;
					}
				}
			}while(repeat);
		}
	}

	private void addPieceDimensionCotation(PlanUsinage face) {
		int width = (int) Math.round(piece.getWidth(face));
		int height = (int) Math.round(piece.getHight(face));
		Cotation widthCot = new Cotation();
		widthCot.addline(new Line(new Vector2(), new Vector2(width, 0), false));
		widthCot.setText(width + "");
		widthCot.setTextPos(new Vector2(width/2.0f, 0));
		Cotation heightCot = new Cotation();
		heightCot.addline(new Line(new Vector2(), new Vector2(0, height), false));
		heightCot.setText(height + "");
		heightCot.setTextPos(new Vector2(0, height/2.f));
		cotations.add(widthCot);
		cotations.add(heightCot);
	}

	private void filter() {
		HashSet<Cotation> cotationSet = new HashSet<Cotation>();
		for(Cotation cotation: cotations) {
			cotationSet.add(cotation);
		}
		HashMap<Vector2, ArrayList<Cotation>> cotationsGroup = groupByDirection(cotationSet);
		ArrayList<ArrayList<Cotation>> groups = new ArrayList<ArrayList<Cotation>>();
		for(Vector2 dir: cotationsGroup.keySet()) {
			groups.addAll(splitBySide(cotationsGroup.get(dir)));
		}
		for(ArrayList<Cotation> group: groups) {
			sort(group);
		}
		cotations.clear();
		for(ArrayList<Cotation> group: groups) {
			cotations.addAll(align(group));
		}
	}
	
	private ArrayList<Cotation> align(ArrayList<Cotation> group) {
		Vector2 cotationDir = group.get(0).getLines().get(0).getDirection().nor();
		Vector2 translationDirection =  group.get(0).getLines().get(0).getMiddle().cpy().sub(pieceCenter);
		translationDirection.sub(cotationDir.cpy().scl(translationDirection.dot(cotationDir))).nor();
		float scale = this.preferences.ALIGN_DST * (group.size());
		for(Cotation cot: group) {
			Vector2 translation = translationDirection.cpy().scl(scale);
			if(cot == null)
				continue;
			cot.getLines().get(0).translate(translation.x, translation.y);
			scale -= this.preferences.ALIGN_DST;

		}
		return group;
	}

	//split a collection of cotations having same directions by position
	private ArrayList<ArrayList<Cotation>> splitBySide(ArrayList<Cotation> cotations){
		HashMap<Line, ArrayList<Cotation>> groups = new HashMap<Line, ArrayList<Cotation>>();
		for(Cotation cotation: cotations) {
			Vector2 point = cotation.getLines().get(0).getMiddle();
			ArrayList<Cotation> group = null;
			for(Line line: groups.keySet()) {
				if(line.isOnline(point)) {
					group = groups.get(line);
					group.add(cotation);
					break;
				}
			}
			if(group == null) {
				group = new ArrayList<Cotation>();
				group.add(cotation);
				groups.put(cotation.getLines().get(0), group);
			}
		}
		ArrayList<ArrayList<Cotation>> cotationsArray = new ArrayList<ArrayList<Cotation>>();
		for(Line line: groups.keySet()) {
			cotationsArray.add(groups.get(line));
		}
		
		
		return cotationsArray;
	}

	private void sort(ArrayList<Cotation> cotations) {
		ArrayList<Cotation> sortedCotations = new ArrayList<Cotation>();
		while(sortedCotations.size() < cotations.size()) {
			Cotation largest = null;
			for(Cotation cot: cotations) {
				if(sortedCotations.contains(cot))
					continue;
				if(largest == null) {
					largest = cot;
					continue;
				}
				if(cot.getLines().get(0).getLength() > largest.getLines().get(0).getLength()) {
					largest = cot;
				}
			}
			sortedCotations.add(largest);
		}
		if(sortedCotations.contains(null)) {
			cotations.get(0).equals(cotations.get(cotations.size() - 1));
			System.err.println("null in list");

		}
		cotations.clear();
		
		cotations.addAll(sortedCotations);
	}

	private HashMap<Vector2, ArrayList<Cotation>> groupByDirection(HashSet<Cotation> cotations) {
		HashMap<Vector2, ArrayList<Cotation>> cotationsGroup = new HashMap<Vector2, ArrayList<Cotation>>();
		for(Cotation cotation: cotations) {
			Vector2 dir = cotation.getLines().get(0).getDirection();
			ArrayList<Cotation> group = null;
			for(Vector2 direction: cotationsGroup.keySet()) {
				if(direction.isOnLine(dir, 0.0001f)) {
					group = cotationsGroup.get(direction);
					group.add(cotation);
					break;
				}
			}
			if(group == null) {
				group = new ArrayList<Cotation>();
				group.add(cotation);
				cotationsGroup.put(dir, group);
			}
		}
		return cotationsGroup;
	}

	private void init(Piece2D piece, PlanUsinage face) {
		this.face = face;
		this.piece = piece;
		cotations.clear();
		diameterCotations.clear();
		if(shapes != null)
			shapes.clear();
		float width =(float) Util.getPieceL(piece, face);
		float height =(float) Util.getPieceH(piece, face);
		this.pieceCenter = new Vector2(width , height).scl(0.5f);
	}

	private void generateHoleCotations() {
		ArrayList<Cercle2D> circles = getHoles();
		for(Cercle2D hole: circles) {
			generateDiameterCotations(hole);
			generateCenterCotations(hole);
		}
	}
	
	private void generateCenterCotations(Cercle2D hole) {
		float width =(float) Util.getPieceL(piece, face);
		float height =(float) Util.getPieceH(piece, face);
		Vector2 origin = new Vector2();
		Vector2 center = hole.getCenter();
		origin.x = center.x < 0.5f * width?0:width;
		origin.y = center.y < 0.5f * height?0:height;
		Vector2 p = center.cpy().sub(origin);
		Line v = new Line(origin, origin.cpy().add(p.cpy().scl(0,1)), false);
		v.translate(origin.x - v.getMiddle().x, 0);
		Cotation cotation = new Cotation();
		cotation.addline(v);
		cotation.setText(Math.round(v.getLength()) + ""); 
		cotation.setTextRot(v.getDirection().angle());
		cotation.setTextPos(v.getMiddle());
		cotations.add(cotation.cpy());
		
		Line h = new Line(origin, origin.cpy().add(p.cpy().scl(1,0)), false);
		h.translate(0, origin.y - h.getMiddle().y);
		cotation.clearLines();
		cotation.addline(h);
		cotation.setText(Math.round(h.getLength()) + ""); 
		cotation.setTextRot(h.getDirection().angle());
		cotation.setTextPos(h.getMiddle());
		cotations.add(cotation.cpy());
	}

	private void generateDiameterCotations(Cercle2D hole) {
		ArrayList<Cotation> cotations = new ArrayList<Cotation>();
		String text = DIAMETER_HTML_CODE + " " +Math.round(hole.getDiameter());
		Vector2 center = hole.getCenter();
		Vector2  d = new Vector2(1,1).nor().scl(preferences.DIAMETER_DIAG_LEN);
		Vector2 h = new Vector2(1,0).nor().scl(preferences.DIAMETER_HOR_LEN);
		Line diag = new Line(center, center.cpy().add(d.cpy().rotate(0)), false);
		Line hor = new Line(diag.getEnd(), diag.getEnd().cpy().add(h), false);
		//top right
		Cotation cotation = new Cotation();
		cotation.addline(diag.cpy(), hor.cpy());
		cotation.setText(text);
		cotation.setTextPos(hor.getMiddle());
		cotation.setTextRot(0);
		cotations.add(cotation.cpy());
		//top left
		diag = new Line(center, center.cpy().add(d.rotate(90)), false);
		hor = new Line(diag.getEnd(), diag.getEnd().cpy().add(h.scl(-1)), false);
		cotation.clearLines();
		cotation.addline(diag.cpy(), hor.cpy());
		cotation.setTextPos(hor.getMiddle());
		cotations.add(cotation.cpy());
		//bottom left
		diag = new Line(center, center.cpy().add(d.rotate(90)), false);
		hor = new Line(diag.getEnd(), diag.getEnd().cpy().add(h), false);
		cotation.clearLines();
		cotation.addline(diag.cpy(), hor.cpy());
		cotation.setTextPos(hor.getMiddle());
		cotations.add(cotation.cpy());
		//bottom right
		diag = new Line(center, center.cpy().add(d.rotate(90)), false);
		hor = new Line(diag.getEnd(), diag.getEnd().cpy().add(h.scl(-1)), false);
		cotation.clearLines();
		cotation.addline(diag.cpy(), hor.cpy());
		cotation.setTextPos(hor.getMiddle());
		cotations.add(cotation.cpy());
		this.diameterCotations.put(hole, cotations);
	}
	
	private void prepareShapes() {
		if(shapes == null) {
			shapes = new ArrayList<CamShape>();
		}else {
			shapes.clear();
		}
		for (Usinage usinage : piece.getusinage()) {
			CamShape shape = null;
			try {
				shape = usinage.getShape();
			} catch (DesignException e) {
				e.printStackTrace();
				shape = null;
			}
			if (shape != null && shape.getPlan()== face) {
				shapes.add(shape);
			}
		}
	}
	
	public ArrayList<Cercle2D> getHoles(){
		ArrayList<Cercle2D> holes = new ArrayList<Cercle2D>();
		for(CamShape shape: shapes) {
			if(shape instanceof Cercle2D)
				holes.add((Cercle2D) shape);
		}
		return holes;
	}
	
	public ArrayList<Cercle2D> getRainures(){
		ArrayList<Cercle2D> rainure = new ArrayList<Cercle2D>();
		for(CamShape shape: shapes) {
			if(shape instanceof Cuboid)
				rainure.add((Cercle2D) shape);
		}
		return rainure;
	}

}
