#version 400 core

out vec4 FragColor;

const float PI = 3.14159265359;

uniform vec3  u_albedo;
uniform float u_opacity;
uniform float u_roughness;
uniform float u_metallness;


uniform sampler2D albedoMap;
uniform sampler2D normalMap;
uniform sampler2D metallicMap;
uniform sampler2D roughnessMap;

uniform vec3 camera_position;

uniform int has_albedo_map;
uniform int has_normal_map;
uniform int has_metallic_map;
uniform int has_roughness_map;


struct DirectionalLight
{
    vec3 intensity;
    vec3 direction;
};

in VertexData{
    vec3 v_position;
    vec3 v_normal;
    vec2 v_texcoord;
}  inData;




vec3 getNormalFromMap()
{
    vec3 tangentNormal = texture(normalMap, inData.v_texcoord).xyz * 2.0 - 1.0;

    vec3 Q1 = dFdx(inData.v_position);
    vec3 Q2 = dFdy(inData.v_position);
    vec2 st1 = dFdx(inData.v_texcoord);
    vec2 st2 = dFdy(inData.v_texcoord);

    vec3 N = inData.v_normal;
    vec3 T = normalize(Q1 * st2.t - Q2 * st1.t);
    vec3 B = -normalize(cross(N, T));
    mat3 TBN = mat3(T, B, N);

    return normalize(TBN * tangentNormal);
}
// ----------------------------------------------------------------------------
float DistributionGGX(vec3 N, vec3 H, float roughness)
{
    float a = roughness * roughness;
    float a2 = a * a;
    float NdotH = max(dot(N, H), 0.0);
    float NdotH2 = NdotH * NdotH;

    float nom = a2;
    float denom = (NdotH2 * (a2 - 1.0) + 1.0);
    denom = PI * denom * denom;

    return nom / denom;
}
// ----------------------------------------------------------------------------
float GeometrySchlickGGX(float NdotV, float roughness)
{
    float r = (roughness + 1.0);
    float k = (r * r) / 8.0;

    float nom = NdotV;
    float denom = NdotV * (1.0 - k) + k;

    return nom / denom;
}
// ----------------------------------------------------------------------------
float GeometrySmith(vec3 N, vec3 V, vec3 L, float roughness)
{
    float NdotV = max(dot(N, V), 0.0);
    float NdotL = max(dot(N, L), 0.0);
    float ggx2 = GeometrySchlickGGX(NdotV, roughness);
    float ggx1 = GeometrySchlickGGX(NdotL, roughness);

    return ggx1 * ggx2;
}

vec3 fresnelSchlick(float cosTheta, vec3 F0)
{
    return F0 + (1.0 - F0) * pow(clamp(1.0 - cosTheta, 0.0, 1.0), 5.0);
}


vec3 directionalLightContribution(vec3 lightDir, vec3 N, vec3 albedo, float roughness, float metallic)
{
    vec3 L = normalize(-lightDir);
    vec3 V = normalize(camera_position - inData.v_position);
    vec3 H = normalize(V + L);
    vec3 radiance = vec3(2);

    // Cook-Torrance BRDF
    float NDF = DistributionGGX(N, H, roughness);
    float G = GeometrySmith(N, V, L, roughness);
    // calculate reflectance at normal incidence; if dia-electric (like plastic) use F0 
    // of 0.04 and if it's a metal, use the albedo color as F0 (metallic workflow)  
    vec3 F0 = vec3(0.04);
    F0 = mix(F0, albedo, metallic);
    vec3 F = fresnelSchlick(max(dot(H, V), 0.0), F0);

    vec3 numerator = NDF * G * F;
    float denominator = 4.0 * max(dot(N, V), 0.0) * max(dot(N, L), 0.0) + 0.0001; // + 0.0001 to prevent divide by zero
    vec3 specular = numerator / denominator;
    // kS is equal to Fresnel
    vec3 kS = F;
    vec3 kD = vec3(1.0) - kS;
    kD *= 1.0 - metallic;
    float NdotL = max(dot(N, L), 0.0);
    vec3 contribution = (kD * albedo / PI + specular) * radiance * NdotL;
    return contribution + vec3(0.1) * albedo;
}

void main()
{
    vec3 albedo;
    if(has_albedo_map > 0.01){
        albedo = pow(texture(albedoMap, inData.v_texcoord).rgb, vec3(2.21));
    } else{
        albedo = u_albedo; 
    }
    float metallic = u_metallness;
    if(has_metallic_map > 0.01){
       metallic =  texture(metallicMap, inData.v_texcoord).r;
    }
    metallic = max(metallic, 0.04);
    float roughness = u_roughness;
    if(has_roughness_map > 0.01){
        roughness = texture(roughnessMap, inData.v_texcoord).r;
    } 
    float ao = 0.2; //texture(aoMap, inData.v_texcoord).r;

    vec3 N = inData.v_normal;
    if(has_normal_map > 0.01){
        N = getNormalFromMap();
    }


    vec3 Lo = vec3(0.0);
    Lo += directionalLightContribution(vec3(-1, -1,-1), N, albedo, roughness, metallic);
    Lo += directionalLightContribution(vec3(1, 1,1), N, albedo, roughness, metallic);

    vec3 ambient = vec3(0.3) * albedo * ao;

    vec3 color = ambient + Lo;

    //HDR tonemapping
    color = color / (color + vec3(1.0));
    //gamma correct
    color = pow(color, vec3(1.0 / 2.2));
    FragColor = vec4(color, u_opacity);
}