package dressing.ui.engine3d;

import java.io.File;

import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.files.FileHandle;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.TextureData;
import com.badlogic.gdx.graphics.Texture.TextureFilter;
import com.badlogic.gdx.graphics.Texture.TextureWrap;
import com.badlogic.gdx.graphics.g2d.TextureRegion;

import dressing.config.WorkspaceConfiguration;
import gdxapp.assets.AssetsTextures;
import jakarta.xml.bind.annotation.XmlRootElement;

@XmlRootElement(name = "texture")
public class SceneTexture {

	public enum CATEGORY {
		GENERIC, ELEMENT, MARBLE, WALL, FLOOR, VIEW, METALS,KNOBS;

		public String getDisplayName() {
			switch (this) {
			case GENERIC:
				return "Generic";
			case ELEMENT:
				return "Elément";
			case MARBLE:
				return "Plan de Travail";
			case WALL:
				return "Mur";
			case FLOOR:
				return "Sol";
			default:
				return this.name();
			}
		}
	};

	private String path;
	private CATEGORY category;
	private transient ImageData imageData;
	private transient TextureData textureData;
	private transient Texture texture;
	private transient TextureRegion textureRegion;
	private transient Image labelImage;
	private transient Image preview;

	public SceneTexture() {
	}

	public SceneTexture(String path) {
		super();
		this.path = path;
	}

	public void loadTexture() {
		if (path == null || path == "")
			return;
		try {
			texture = AssetsTextures.getInstance().loadTextureFromFile(path, true);
			textureRegion = new TextureRegion(texture);
		} catch (Exception e) {
			e.printStackTrace();
			loadTextureData();
		}
	}

	public void loadTextureData() {
		FileHandle fh = new FileHandle(path);
		this.textureData = TextureData.Factory.loadFromFile(fh, true);
		Gdx.app.postRunnable(new Runnable() {

			@Override
			public void run() {
				prepareData();
			}
		});
	}

	public void prepareData() {
		try {
			texture = new Texture(textureData);
			texture.setWrap(TextureWrap.MirroredRepeat, TextureWrap.MirroredRepeat);
			texture.setFilter(TextureFilter.Linear, TextureFilter.Linear);
			texture.bind();
			Gdx.gl20.glGenerateMipmap(GL20.GL_TEXTURE_2D);
			textureData = null;
			textureRegion = new TextureRegion(texture);
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	public String getPath() {
		return path;
	}

	public String getMinifiedPath() {
		return WorkspaceConfiguration.minifiedPictureFolders + File.separator + new File(path).getName();
	}

	public void setPath(String path) {
		this.path = path;
	}

	public Texture getTexture() {
		if (texture == null) {
			if (!new File(this.path).exists()) {
				return AssetsTextures.getInstance().getDefaultTexture();
			}
			loadTexture();
		}
		return texture != null ? texture : AssetsTextures.getInstance().getDefaultTexture();
	}

	public boolean isReady() {
		return texture != null;
	}

	public Image getLabelImage() {
		return labelImage;
	}

	public void setLabelImage(Image labelImage) {
		this.labelImage = labelImage;
	}

	public Image getPreview() {
		return preview;
	}

	public void setPreview(Image preview) {
		this.preview = preview;
	}

	public void setTexture(Texture texture) {
		this.texture = texture;
		textureRegion = new TextureRegion(texture);

	}

	public ImageData getImageData() {
		return imageData;
	}

	public void setImageData(ImageData imageData) {
		this.imageData = imageData;
	}

	public TextureData getTextureData() {
		return textureData;
	}

	public void setTextureData(TextureData textureData) {
		this.textureData = textureData;
	}

	public TextureRegion getTextureRegion() {
		if (texture == null)
			loadTexture();
		return textureRegion;
	}

	public void setTextureRegion(TextureRegion textureRegion) {
		this.textureRegion = textureRegion;
	}

	public CATEGORY getCategory() {
		return category;
	}

	public void setCategory(CATEGORY category) {
		this.category = category;
	}

	public SceneTexture clone() {
		return new SceneTexture(path);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((path == null) ? 0 : path.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SceneTexture other = (SceneTexture) obj;
		if (path == null) {
			if (other.path != null)
				return false;
		} else if (!path.equals(other.path))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return "SceneTexture =" + path;
	}

}
