package geometry;

import java.util.ArrayList;

import org.eclipse.swt.widgets.Display;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.InputProcessor;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Triangle;
import dressing.ui.util.EdgeMesureWidow;
import gdxapp.assets.DrawingHelper;
import gdxapp.screens.room.RoomController;
import gdxapp.screens.wall.Wall2D;

public class PolygonDrawer  implements InputProcessor, IDrawer {
	
	private final ArrayList<Vector2> vertices = new ArrayList<>();
	private ArrayList<Edge> edges = new ArrayList<Edge>();
    private Vector2 tempVertex;
    Vector2 currentMouseLocation;
    ShapeRenderer shapeRenderer;
    private BitmapFont font;
    private Vector3[] angles;
    private ArrayList<Triangle> triangles;
    private Vector2 selectedVertex;
    private Edge selectedEdge;
    private boolean closed;
    float xoffset, yoffset;
    long lastClickDate;
    ArrayList<Shape> walls;
    Shape nearestShape;
    Vector2 projection;
    Vector2[] bounds;

    public PolygonDrawer() {}

    private boolean selectVertex(Vector2 point) {
        if(!this.vertices.isEmpty()){
            float[] distances = new float[this.vertices.size()];
            for(int i =0; i < distances.length; i++){
                distances[i] = vertices.get(i).dst(point);
            }
            float min = distances[0];
            int minIndex = 0;
            for(int i = 1; i < distances.length; i++){
                if(distances[i] < min){
                    min = distances[i];
                    minIndex = i;
                }
            }
            if(distances[minIndex] < 10 ){
                selectedVertex = vertices.get(minIndex);
                return true;
            }
        }
        return false;
    }
    
    private boolean selectEdge(Vector2 point) {
    	selectedEdge = null;
        if(!this.edges.isEmpty()){
        	Edge nearest = edges.get(0);
        	for(int i = 1; i < edges.size(); i++){
        		if(edges.get(i).distanceTo(point) < nearest.distanceTo(point))
        			nearest = edges.get(i);
        	}
            if(nearest.distanceTo(point) < 10 ){
                selectedEdge = nearest;
                tempVertex = null;
                return true;
            }
        }
        return false;
    }
    
    public void updateEdgesList() {
    	if(edges == null) {
    		edges = new ArrayList<Edge>();
    	}else {
    		edges.clear();
    	}
    	int count = vertices.size();
    	if(count > 1) {
    		for(int i = 0; i < count-1; i++) {
        		edges.add(new Edge(vertices.get(i), vertices.get(i+1)));
        	}
    		if(closed)
    			edges.add(new Edge(vertices.get(count - 1), vertices.get(0)));
    	}
    }
    
    

    private void addVertex(Vector2 point) {
        if(vertices.isEmpty()){
            vertices.add(point);
        }else if(vertices.size() == 1){
            alignVH(point, vertices.get(0));
            vertices.add(point);
        }else{
        	Edge edge = new Edge(vertices.get(vertices.size() -1), point);
        	for(Edge edgeX: edges) {
        		Vector2 intersection =  MathUtilities.getIntersectionPoint(edgeX, edge);
        		if(!(edgeX.hasVertex(intersection)|| edge.hasVertex(intersection)) &&
        				(edgeX.contains(intersection) && edge.contains(intersection))) {
        			System.out.println("plygon is self intersecting!");
        			float a = edgeX.getV0().dst(intersection);
        			float b = edgeX.getV1().dst(intersection);
        			float c = edgeX.getV0().dst(edgeX.getV1());
        			System.out.println(a + " " + b + " " +c);
        			return;
        		}
        	}
            vertices.add(point);
            rectifyPosition(point);

        }
        updateBoundaries();
        updateEdgesList();
    }
    
    private void updateBoundaries() {
    	Vector2 lastPosition =  vertices.get(vertices.size() - 1);
    	if(this.bounds == null) {
    		this.bounds = new Vector2[4];
    		this.bounds[0] = lastPosition.cpy();
    		this.bounds[1] = lastPosition.cpy();
    		this.bounds[2] = lastPosition.cpy();
    		this.bounds[3] = lastPosition.cpy();

    	}else {
    		this.bounds[0].x = Math.min(lastPosition.x, bounds[0].x);
    		this.bounds[0].y = Math.min(lastPosition.y, bounds[0].y);
    		this.bounds[1].x = Math.max(lastPosition.x, bounds[1].x);
    		this.bounds[1].y = Math.max(lastPosition.y, bounds[1].y);
    		this.bounds[2].set(this.bounds[0].x, this.bounds[1].y);
    		this.bounds[3].set(this.bounds[1].x, this.bounds[0].y);
    	}
	}

	public boolean link(Vector2 point) {
    	Vector2 nearestPoint = null;
    	for(Vector2 attachmentPoint: RoomController.getInstance().getCriticalPoints()) {
    		if(nearestPoint == null) {
    			nearestPoint = attachmentPoint;
    			continue;
    		}
    		if(attachmentPoint.dst(point) < nearestPoint.dst(point)) {
    			nearestPoint = attachmentPoint;
    		}
    	}
    	
    	if(nearestPoint != null && nearestPoint.dst(point) < 20) {
            point.set(nearestPoint);
            return true;
    	}
    	
    	Shape nearestSurface = findNearestSurface(point);
    	if(nearestSurface != null) {
    		Vector2 projectedPoint = nearestSurface.projectPoint(point);
    		if(point.dst(projectedPoint) < 20) {
    			point.set(projectedPoint);
                return true;
    		}
    	}
    	return false;
    }

    private void rectifyPosition(Vector2 v2) {
    	
		if(vertices.size()<=2) {
			return ;
		}
        int index = vertices.indexOf(v2);
        if(index < 2) {
            if(index == 0)
            	return;
            alignVH(v2, vertices.get(0));
            return;
        }
        Vector2 v0 = vertices.get(index-2).cpy();
        Vector2 v1 = vertices.get(index -1).cpy();
        Vector2 v01 = new Vector2(v0).sub(v1);
        Vector2 v21 = new Vector2(v2).sub(v1);
        float angle = MathUtilities.signedAngle(v01,v21);
        float rads15 = (float) Math.toRadians(15);
        float rectifiedAngle = Math.round(angle/rads15) * rads15;
        Vector2 rectifiedDirection = v01.nor().rotateRad(rectifiedAngle).scl(v21.len());
        v2.set(v1).add(rectifiedDirection);
//        if(bounds != null) {
//        	Vector2 closestCorner = bounds[0];
//        	for(int i = 1; i < bounds.length; i++) {
//        		if(v2.dst(bounds[i]) < v2.dst(closestCorner)) {
//        			closestCorner = bounds[i];
//        		}
//        	}
//        	
//        	if(closestCorner.dst(v2) < 20) {
//        		v2.set(closestCorner);
//        	}
//        }
    }

    private void alignVH(Vector2 point, Vector2 origin) {
        Vector2 po = point.cpy().sub(origin);
        po.x = Math.abs(po.x);
        po.y = Math.abs(po.y);
    	float cm10 = RoomController.getInstance().getScaleX() * 0.3f;
        if(po.x < po.y){
            if(po.x < cm10){
                point.x = origin.x;
            }
        }else{
            if(po.y < cm10 ){
                point.y = origin.y;
            }
        }
    }
    void calculateAngles(){
    	if(this.vertices != null && this.vertices.size() > 2) {
    		this.angles = new Vector3[this.vertices.size()];
            int previousIndex, nextIndex;
            Vector2 v01,v21;
            for(int i = 0 ; i < vertices.size(); i++){
                previousIndex = (i-1)%vertices.size();
                if(previousIndex < 0)
                    previousIndex += vertices.size();
                nextIndex = (i+1)%vertices.size();
                v01 = new Vector2(vertices.get(previousIndex)).sub(vertices.get(i));
                v21 = new Vector2(vertices.get(nextIndex)).sub(vertices.get(i));
                float angle = MathUtilities.signedAngle(v01,v21);
                angles[i] = new Vector3(vertices.get(i),angle);
            }
    	}
    }
    
    public void draw(Matrix4 projection){
    	if(shapeRenderer == null)
    		shapeRenderer = new ShapeRenderer();
    	shapeRenderer.setProjectionMatrix(projection);
        shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
        int verticesCount = closed?vertices.size() + 1: vertices.size();
        int nextIndex;
        for(int i = 0; i < vertices.size() ; i++){
            shapeRenderer.setColor(Color.GREEN);
            if(vertices.get(i) == selectedVertex)
                shapeRenderer.setColor(Color.RED);
            shapeRenderer.circle(vertices.get(i).x, vertices.get(i).y,5.0f);
        }
        
        for(int i = 0; i < verticesCount -1 ; i++){
            nextIndex = (i+1)%(vertices.size());
            shapeRenderer.setColor(Color.GREEN);
            shapeRenderer.rectLine(vertices.get(i),vertices.get(nextIndex),5);
        }
        if(selectedEdge != null) {
        	shapeRenderer.setColor(Color.RED);
        	shapeRenderer.rectLine(selectedEdge.getV0(), selectedEdge.getV1(), 4);
        	shapeRenderer.setColor(Color.BLACK);
        }
        shapeRenderer.end();

        if(edges != null) {
        	for(Edge edge: edges) {
        		float length = edge.getLength() / RoomController.getInstance().getScaleX();
        		length = Math.round(1000 * length);
        		Vector2 position = edge.getMiddle().add(edge.getNormal().scl(25));
        		float rotation = edge.getV1().cpy().sub(edge.getV0()).angle();
        		DrawingHelper.drawText(String.format("%.0f", length), new Vector3(position,0.0f), rotation, Color.BLACK, 1.0f, RoomController.getInstance().getStage().getBatch());
        	}
        }
        if(vertices.size() >2 ) {
        	for(int i = 0; i < vertices.size(); i++) {
        		if(!closed && (i == 0 || i ==  vertices.size() -1))
        			continue;
        		int nextVertex = (i+1)%vertices.size();
        		int previousVertex = (i-1)%vertices.size();
        		if(closed && i == 0) {
        			previousVertex = vertices.size()-1;
        		}
        		if(closed && i ==  vertices.size() -1) {
        			nextVertex=0;
        		}
        		Vector2 vp = vertices.get(previousVertex).cpy().sub(vertices.get(i));
        		Vector2 vn = vertices.get(nextVertex).cpy().sub(vertices.get(i));
        		float rotation = (float) Math.toDegrees(MathUtilities.signedAngle(vp, vn));
        		Vector2 translation = vp.cpy().nor().add(vn.cpy().nor()).nor().scl(-50);
        		DrawingHelper.drawText(String.format("%.0f", rotation), new Vector3(translation.cpy().add(vertices.get(i)),0.0f), translation.cpy().scl(-1).rotate(90).angle(), Color.BLUE, 1.0f, RoomController.getInstance().getStage().getBatch());
        	}
        }

    }
    @Override
    public boolean keyDown(int keycode) {
        return false;
    }

    @Override
    public boolean keyUp(int keycode) {
        return false;
    }

    @Override
    public boolean keyTyped(char character) {
        return false;
    }

    @Override
    public boolean touchDown(int screenX, int screenY, int pointer, int button) {
    	long currentTime = System.currentTimeMillis();
    	
        xoffset = screenX;
        yoffset = Gdx.graphics.getHeight() - screenY;
        Vector2 point = new Vector2(xoffset, yoffset);
        if(!selectVertex(point)) {
        	if(selectEdge(point) && (currentTime - lastClickDate < 500)) {
        		Display.getDefault().asyncExec(new Runnable() {
					public void run() {

						EdgeMesureWidow window = EdgeMesureWidow.getInstance();
						window.setEdge(selectedEdge);
						window.setEdges(edges);
						window.show();
					}
				});
        	}
        }
        lastClickDate = currentTime;
        return true;
    }

    @Override
    public boolean touchUp(int screenX, int screenY, int pointer, int button) {
        Vector2 point = new Vector2(screenX, Gdx.graphics.getHeight() - screenY);
        if(button == Input.Buttons.LEFT) {
            if(selectedVertex == null && selectedEdge == null){
            	if(Gdx.input.isKeyPressed(Input.Keys.X)) {
        			link(point);
        			vertices.add(point);
        		}else {
        			addVertex(point);
        		}
            }else{
            	if(selectedVertex != null &&( !Gdx.input.isKeyPressed(Input.Keys.SHIFT_LEFT) && !Gdx.input.isKeyPressed(Input.Keys.SHIFT_RIGHT)))
                {
            		if(Gdx.input.isKeyPressed(Input.Keys.X)) {
            			link(selectedVertex);
            		}else {
                		rectifyPosition(selectedVertex);
            		}
                }
            }
            
            
        }
        if(button == Input.Buttons.RIGHT){
            selectVertex(point);
            removeVertex(selectedVertex);
        }
        updateEdgesList();
        calculateAngles();
        setTempVertex(null);
        selectedVertex = null;
        return false;
    }

    private void removeVertex(Vector2 vertex) {
    	selectedEdge = null;
    	vertices.remove(vertex);
	}

	@Override
    public boolean touchDragged(int screenX, int screenY, int pointer) {

        setCurrentMouseLocation(screenX,Gdx.graphics.getHeight() - screenY);
        float xmov = screenX - xoffset;
        float ymov = Gdx.graphics.getHeight() - screenY - yoffset;
        if(vertices.size()>0 && Gdx.input.isButtonPressed(Input.Buttons.LEFT) && selectedVertex == null && selectedEdge == null){
            setTempVertex(currentMouseLocation);
        }
        if(selectedVertex != null){
            selectedVertex.add(xmov,ymov);
        }else if(selectedEdge != null) {
        	selectedEdge.translate(xmov,ymov);
        }
        xoffset = screenX;
        yoffset = Gdx.graphics.getHeight() - screenY;
        return false;
    }

    @Override
    public boolean mouseMoved(int screenX, int screenY) {
        setCurrentMouseLocation(screenX,Gdx.graphics.getHeight() - screenY);
        findNearestSurface(currentMouseLocation);
        Gdx.app.debug("mouse move",currentMouseLocation + "");
        return false;
    }





	public void setTempVertex(Vector2 tempVertex) {
        this.tempVertex = tempVertex;
    }

    @Override
    public void begin(Vector2 vertex) {
    	clear();
        if(vertex != null)
            vertices.add(vertex);
    }

    @Override
    public Shape end() {
    	if(this.vertices!=null && this.vertices.size()>1) {
    		Polygon poly = new Polygon();
    		poly.setVertices(this.vertices);
    		clear();
            return poly;
    	}
        clear();
        return null;
    }

	@Override
	public boolean scrolled(int amount) {
		return false;
	}


	@Override
	public ArrayList<Vector2> getVertices() {
		return vertices;
	}

	@Override
	public void finish() {
		clear();
		
	}
	

	public void edit(Polygon poly) {
		clear();
		for(Vector2 vertex: poly.getVertices()) {
			this.vertices.add(vertex);
		}
		updateEdgesList();
		calculateAngles();
	}
	private void clear() {
		vertices.clear();
		bounds = null;
		angles = null;
		edges.clear();
		if(walls != null)
			walls.clear();
		tempVertex = null;
		triangles = null;
		selectedEdge = null;
		selectedVertex = null;
	}

	public Vector2 getCurrentMouseLocation() {
		return currentMouseLocation;
	}

	public void setCurrentMouseLocation(float x , float y) {
		if(currentMouseLocation == null)
			currentMouseLocation = new Vector2();
		currentMouseLocation.set(x,y);
	}
	
	

	public Shape findNearestSurface(Vector2 location) {
		nearestShape = null;
		if(walls == null) {
			walls = new ArrayList<>();
		}
		if(walls.isEmpty()) {
			for(Wall2D wall: RoomController.getInstance().getWalls()) {
				walls.addAll(wall.getSides());
			}
		}
		
		if(!walls.isEmpty()) {
			for(int i = 0; i < walls.size(); i++) {
				if(nearestShape == null) {
					nearestShape = walls.get(0);
					continue;
				}
				if(nearestShape.distanceTo(location) > walls.get(i).distanceTo(location)) {
					nearestShape = walls.get(i);
				}
			}
		}
		if(nearestShape != null) {
			projection = nearestShape.projectPoint(location);
		}
		return nearestShape;
	}

	public boolean isClosed() {
		return closed;
	}

	public void setClosed(boolean closed) {
		this.closed = closed;
	}
	
	
	
}
