package dressing.model;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.attribute.FileTime;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import java.util.zip.ZipOutputStream;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.swt.widgets.Display;

import dressing.config.UserPreference;
import dressing.config.WorkspaceConfiguration;
import dressing.config.persistence.ResourceManagers;
import dressing.handlers.cam.GcodeFileManager;
import dressing.io.IOUtilities;
import dressing.model.persistence.mappers.MapperProvider;
import dressing.ui.ProgressBarMonitor;
import dressing.ui.parts.Edit2DPiece;
import dressing.ui.parts.GdxPart;
import gdxapp.scenes.Scene;
import gdxapp.scenes.SceneManager;
import gdxapp.screens.room.RoomController;
import param.MechanicDesign;
import param.impl.MechanicDesignImpl;
import tech.frsdev.shapes2D.Piece2DDraw;

public class ProjectManager {
	
	private SuperCadProject currentProject;
	private List<SuperCadProject> projects=new ArrayList<SuperCadProject>();
	public static ProjectManager instance =null;
	private String currentProjectDirectory;
	ProjectRoot root = new ProjectRoot();
	public Scene desfaultScene = new Scene();
	private final static Properties properties = new Properties();

	Timer time;
	AutoSaveTask task;
	private void initialize() {
		this.projects=openProjects();
		if(this.root==null) {
			this.root=new ProjectRoot();
		}
	}
	public void addProject(SuperCadProject project) {
		this.projects.add(project);
	}
	public List<SuperCadProject> getProjects(){
		return this.projects;
	}
	
	public ProjectRoot getRoot() {
		return root;
	}

	public boolean saveProject(SuperCadProject project, boolean savegcode, boolean saveVues) {
		if(!project.isLoaded())
			return false;
		String path=WorkspaceConfiguration.getProjectsPath() + "\\" + project.getName();
		backupProject(path,project);
		return saveProject(project, path,savegcode,saveVues);
	}
	public void backupProject(String path, SuperCadProject project) {
		 File target = new File(path);
		 if(!target.exists()) {
			 return;
		 }
		 String backupfolder=WorkspaceConfiguration.getProjectsPath() +File.separator+"backup" ;
		 File backup=new  File(backupfolder);
		 if(!backup.exists()) {
			 backup.mkdirs();
		 }
	     System.err.println(backup.setWritable(true)+" "+backup.setExecutable(true)+"  "+backup.setReadable(true));
	     
		 String projectBackup=backupfolder+File.separator+ project.getName();	
		 File f1=new File(projectBackup);
		 if(!f1.exists()) {
			 f1.mkdirs();
		 }else {
			 //folder exist, check for older back ups and remove older backups. The max number of back ups is defined by the user preferences
			File[] backups = f1.listFiles();
			int maxBackUpsNbre = Math.max(UserPreference.getPreference().getMaxProjectBackUps() -1, 1);
			if(backups.length > maxBackUpsNbre) {
				ArrayList<File> sortedByTime = new ArrayList<File>();
				sortedByTime.addAll(Arrays.asList(backups));
				sortedByTime.sort(new Comparator<File>() {
					@Override
					public int compare(File f0, File f1) {
						long f0CreationTime = 0;
						long f1CreationTime = 0;
						try {
							f0CreationTime = ((FileTime) Files.getAttribute(f0.toPath(), "creationTime")).toMillis();
							f1CreationTime = ((FileTime) Files.getAttribute(f1.toPath(), "creationTime")).toMillis();
						} catch (IOException e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						}
						return (int) (f1CreationTime - f0CreationTime);
					}
				});
				
				for(int i = maxBackUpsNbre; i < sortedByTime.size(); i++) {
					sortedByTime.get(i).delete();
				}
			}
		 }
	     System.err.println(f1.setWritable(true)+" "+f1.setExecutable(true)+"  "+f1.setReadable(true));
		 String backupFile=backupfolder+File.separator+ project.getName()+File.separator+ project.getName()+"_"+(new Date().getTime())+".cad";	
		 File backupFile1=new File(backupFile);

	     try {
		       FileOutputStream fos = new FileOutputStream(backupFile);
		        ZipOutputStream zipOut = new ZipOutputStream(fos);
		        File fileToZip = target;
		 
		        IOUtilities.zipFile(fileToZip, fileToZip.getName(), zipOut);
		        zipOut.close();
		        fos.close();

		 	} catch (FileNotFoundException e1) {
				e1.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}
	}
	public boolean saveProject(SuperCadProject project,String projectDir, boolean savegcode, boolean saveVues ) {
		UserPreference.getPreference().save();
		currentProjectDirectory = projectDir;
		File dir = new File(projectDir);
		dir.mkdirs();
		File file = new File(projectDir + "\\proj.xml");
		if(!file.exists()) {
			try {
				file.createNewFile();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		try {
			MapperProvider.getXmlMapper().write(project, file);
		}catch (Exception e) {
			e.printStackTrace();
		}
		for(Kitchen kitchen: project.getKitchens()) {
			String kitchenpath = projectDir + "//" + kitchen.getName();
			saveKitchen(kitchen, kitchenpath,savegcode,saveVues);
		}
		
		System.out.println("saving project!");
		return false;
	}
	
	public void saveKitchen(Kitchen kitchen, String path, boolean savegcode, boolean saveVues) {
		ResourceManagers resourcesManagers=ResourceManagers.getIntance();
		File kitchedir = new File(path);
		kitchedir.mkdirs();kitchedir.setExecutable(true);
		kitchedir.setWritable(true);kitchedir.setReadable(true);
		File file = new File(kitchedir + "//" + "elements.xml");
		if(!file.exists())
			try {
				file.createNewFile();
			} catch (IOException e2) {
				// TODO Auto-generated catch block
				e2.printStackTrace();
			}
		URI uri=URI.createFileURI(file.getAbsolutePath());
		Resource resource = resourcesManagers.loadResource(uri);
		resource.getContents().clear();
		kitchen.getElementDefinitions().clear();
		for(DesignObject3D designObject :kitchen.getElements()) {
			if(designObject.getMechanicDesignDefinition() != null) {
				MechanicDesignImpl eobject = (MechanicDesignImpl) designObject.getMechanicDesignDefinition();
				resource.getContents().add(eobject);	
				kitchen.getElementDefinitions().add(eobject);
			}
		}
		try {
			resource.save(Collections.EMPTY_MAP);
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		String kitchenPath = path;
		SceneManager.instance.saveScene(kitchen.getScene(), kitchenPath + "//scene.xml");
		if(savegcode) {
			saveGcode(kitchen, kitchenPath + File.separator + "/gcodes");
		}
		if(saveVues) {
			saveVues(kitchen, kitchenPath + File.separator + "/vues");
		}
	}
	
	public void saveVues(Kitchen kitchen, String path) {
		File dir = new File(path);
		IOUtilities.deleteFolder(dir);

		if (!dir.exists())
			dir.mkdirs();
		
		File output;
		for(DesignObject3D element: kitchen.getElements()) {
			output = new File(path + File.separator+  element.getName() +" n " +  kitchen.getElements().indexOf(element));
			if(!output.exists())
				output.mkdirs();
			for(DesignObject3D piece: element.getListPieces()) {
				if(piece instanceof Piece2D) {
					Piece2D pieceX = (Piece2D) piece;
					Piece2DDraw pieceDraw = new Piece2DDraw(pieceX, 640, 640);
					Edit2DPiece.paintToFile(output.getAbsolutePath() + File.separator + pieceX.getName(), pieceDraw);
				}
			}
		}
		
	}
	private void saveGcode(Kitchen kitchen,String kitchenPath) {
		// TODO Auto-generated method stub
		GcodeFileManager codeFileManager = new GcodeFileManager(".gcode");
		codeFileManager.saveGcode(kitchen, kitchenPath);
	}
	
	public ArrayList<SuperCadProject> openProjects() {
		File file = new File(WorkspaceConfiguration.getProjectsPath());
		String[] directories = file.list(new FilenameFilter() {
		  @Override
		  public boolean accept(File current, String name) {
		    return new File(current, name).isDirectory();
		  }
		});
		ArrayList<SuperCadProject> projects = new ArrayList<SuperCadProject>();
		//ProgressBarMonitor.addtask();
		for(String directory : directories) {
			if(directory.contentEquals("tmp")|| directory.contentEquals("backup")) {
				continue;
			}
			String rootDir = WorkspaceConfiguration.getProjectsPath() + File.separator + directory;
			
			SuperCadProject project;
			try {
				project = readProjectfromDisk(rootDir);
				if(project!=null) {
					projects.add(project);
				}
			} catch (FileNotFoundException e) {
				e.printStackTrace();
			}
		}
		//ProgressBarMonitor.removetask();
		return projects;
	}

	public SuperCadProject readProjectfromDisk(String rootDir) throws FileNotFoundException { 
		SuperCadProject project = null;
		File file = new File(rootDir + File.separator + "proj.xml");		
		try {
			project = (SuperCadProject) MapperProvider.getXmlMapper().fromXml(file, SuperCadProject.class);
			project.initAfterRead();
			project.setProjectDirectory(rootDir);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return project;
	}
	public void loadProject(SuperCadProject project) {
		ResourceManagers resourcesManagers=ResourceManagers.getIntance();
		String rootDir =project.getProjectDirectory();
		if(rootDir==null) {
			return;
		}
		for(Kitchen kitchen : project.getKitchens()) {
			kitchen.setProject(project);
			kitchen.getElementDefinitions().clear();
			Resource resource = resourcesManagers.loadResource(URI.createFileURI(rootDir + File.separator + kitchen.getName() + File.separator +"elements.xml"));
			 for(Object obj: resource.getContents()) {
				 if(obj instanceof MechanicDesign)
				 {
					 kitchen.getElementDefinitions().add((MechanicDesign)obj);
				 }
			 }
			 
			 do {
				 System.out.println("loading...");
				 
			 }while(SceneManager.instance == null);
			 Scene scene = SceneManager.instance.loadScene(rootDir + File.separator + kitchen.getName() + File.separator +"scene.xml");
			 kitchen.setScene(scene);
		}
		
	}
	public void openProject(SuperCadProject project) {
		if(task!=null)
		{
			task.cancel();
		}
		if(time!=null)
		{
			time.cancel();
			time.purge();
		}
		long start, end;
		start = new Date().getTime();
		//ProgressBarMonitor.addtask();
		loadProject(project);
		if(project.getKitchens() != null) {
			for(Kitchen kitchen :project.getKitchens()) {
				kitchen.buildElements();
				end = new Date().getTime() - start;
				end *= 0.001;
				System.out.println("finsihed loading in " +  String.valueOf(end) + "s");
			}
		}
		 time = new Timer(); // Instantiate Timer Object
		 task=  new AutoSaveTask();
		 time.schedule(task, 300000, 300000);//plz do not reset the delay time to zero again 
		 
	}
	public void openCurrentProject() {
		openProject(currentProject);
	}
	public ProjectManager() {
		super();
		initialize();
	}

	public SuperCadProject getCurrentProject() {
		return currentProject;
	}

	public void setCurrentProject(SuperCadProject currentProject, boolean open) {
		this.root.addProject(currentProject,false);
		this.currentProject = currentProject;
		//build kitchen element from definitions
		if(open)
			openCurrentProject();
		if(this.currentProject!=null && this.currentProject.getKitchens()!=null 
				&&this.currentProject.getKitchens().size()>0  ) {
			this.currentProject.setCurrentKitchen(this.currentProject.getKitchens().get(0));
			SceneManager.instance.clearStage();
			SceneManager.instance.loadActors();
			getCurrentScene().requireRefresh=true;
			SceneManager.instance.clean();
			RoomController.refresh=true;
		}
	}
	
	public void setCurrentKitchen(Kitchen kitchen) {
		if(this.currentProject!=null && this.currentProject.getKitchens()!=null 
				&& this.currentProject.getCurrentKitchen()!=null &&kitchen !=null && !this.currentProject.getCurrentKitchen().equals(kitchen)
				&&this.currentProject.getKitchens().size()>0 &&  this.currentProject.getKitchens().contains(kitchen) ) {
			this.currentProject.setCurrentKitchen(kitchen);
			//build the currentScene Actors from its definitions witch loaded from the file
			SceneManager.instance.loadActors();
			getCurrentScene().requireRefresh=true;
			RoomController.refresh=true;
		}
	}
	
	public static ProjectManager getManager() {
		if(instance ==null) {
			instance = new ProjectManager();		
		}
		return instance;
	}
	
	public void createNewProject(SuperCadProject project) {
		setCurrentProject(project, true);
	}

	public String getCurrentProjectDirectory() {			
		return currentProject.getProjectDirectory();
	}

	public void setCurrentProjectDirectory(String currentProjectDirectory) {
		this.currentProjectDirectory = currentProjectDirectory;
	}
	
	public Kitchen getCurrentKitchen() {
		if(currentProject==null) {
			return null;
		}
		return currentProject.getCurrentKitchen();
	}
	
	public Scene getCurrentScene() {
		if(currentProject!=null && currentProject.getCurrentKitchen() !=null) {
			return currentProject.getCurrentKitchen().getOrCreateNewScene();
		}else {
			return desfaultScene;
		}
	}
	
	public static Properties getProperties() {
		if(properties.isEmpty()) {
			loadProperties();
		}
		return properties;
	}
	private static void loadProperties() {
		File f = new File(WorkspaceConfiguration.SETTING_PATH);
		FileInputStream inputstream = null;
		if (f.exists()) {
			try {
				inputstream = new FileInputStream(f);
				ProjectManager.properties.load(inputstream);
				inputstream.close();
			} catch (IOException e) {
				e.printStackTrace();
			} finally {
				try {
					if (inputstream != null)
						inputstream.close();
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
	}
	public void dispose() {
		if(task!=null)
		{
			task.cancel();
		}
		if(time!=null)
		{
			time.cancel();
			time.purge();
		}
	}
	class AutoSaveTask extends TimerTask  {

		   public AutoSaveTask(){}

		   public void run() {
			   
			   try {
			    System.out.println("auto saving at" + LocalTime.now().toString());
				ProjectManager.getManager().saveProject(ProjectManager.getManager().getCurrentProject(),false,false);
				}catch (Exception e) {
					System.out.print("\n + error saving project");
					e.printStackTrace();
				}

		    }
		}
}
