package api.ui;

import java.io.File;
import java.util.HashMap;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.opengl.GLCanvas;
import org.eclipse.swt.opengl.GLData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.wb.swt.SWTResourceManager;
import org.joml.Vector2i;
import org.joml.Vector3f;
import org.lwjgl.opengl.GL;

import api.backend.ApplicationContext;
import api.event.EventBus;
import api.event.EventHandler;
import api.event.EventType;
import api.event.MepEvent;
import api.graphics.ModelInstance;
import api.provider.ModelProvider;
import api.utils.RessourceLoader;

public class EditorContainer extends Composite implements EventHandler {

	private GLCanvas canvas;
	private ToolBar mainMenuToolBar; 
	private RightMenuToolBarManager rightMenuToolBarManager;

	private StackLayout stackLayout;

	private PlumbingMenu plumbingMenu;
	private ModelLibrary library;
	private SelectionMenu selectionMenu;

	private Composite rightBarContainer;
	private Shell[] distanceDisplays;
	private Text[] textFields;


	public EditorContainer(Composite parent, int style) {
		super(parent, style);
		GridLayout gridLayout = new GridLayout(1, true);
		gridLayout.verticalSpacing = 0;
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		setLayout(gridLayout);

		Composite topComposite = new Composite(this, SWT.NONE);
		GridLayout gl_topComposite = new GridLayout(1, false);
		gl_topComposite.marginWidth = 0;
		gl_topComposite.verticalSpacing = 0;
		gl_topComposite.marginHeight = 0;
		gl_topComposite.horizontalSpacing = 0;
		topComposite.setLayout(gl_topComposite);
		topComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		

		Composite mainComposite = new Composite(this, SWT.NONE);
		GridLayout gl_mainComposite = new GridLayout(2, false);
		gl_mainComposite.horizontalSpacing = 0;
		gl_mainComposite.marginHeight = 0;
		gl_mainComposite.marginWidth = 0;
		gl_mainComposite.verticalSpacing = 0;
		mainComposite.setLayout(gl_mainComposite);
		mainComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		Composite canvasContainer = new Composite(mainComposite, SWT.NONE);
		canvasContainer.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		GridLayout gl_canvasContainer = new GridLayout(1, false);
		gl_canvasContainer.horizontalSpacing = 0;
		gl_canvasContainer.verticalSpacing = 0;
		gl_canvasContainer.marginHeight = 0;
		gl_canvasContainer.marginWidth = 0;
		canvasContainer.setLayout(gl_canvasContainer);
		
		GLData data = new GLData();
		data.doubleBuffer = true;
		canvas = new GLCanvas(canvasContainer, SWT.BORDER, data);
		canvas.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		canvas.setCurrent();
		GL.createCapabilities();
		ApplicationContext.setCanvas(canvas);

		DropTarget dropTarget = new DropTarget(canvas, DND.DROP_MOVE | DND.DROP_COPY | DND.DROP_DEFAULT);
		dropTarget.setTransfer(TextTransfer.getInstance());
		
		Composite sideContainer = new Composite(mainComposite, SWT.NONE);
		sideContainer.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, true, 1, 1));
		GridLayout gl_sideContainer = new GridLayout(1, true);
		gl_sideContainer.marginHeight = 0;
		gl_sideContainer.verticalSpacing = 0;
		gl_sideContainer.marginWidth = 0;
		gl_sideContainer.horizontalSpacing = 0;
		sideContainer.setLayout(gl_sideContainer);
		RightMenuToolBarManager toolBarManager2 = new RightMenuToolBarManager(sideContainer);
		rightBarContainer = new Composite(sideContainer, SWT.NONE);
		rightBarContainer.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, true, 1, 1));
		rightBarContainer.setBackground(SWTResourceManager.getColor(SWT.COLOR_WIDGET_DARK_SHADOW));
		stackLayout = new StackLayout();
		rightBarContainer.setLayout(stackLayout);
		plumbingMenu = new PlumbingMenu(rightBarContainer, SWT.NONE);
		library = new ModelLibrary(rightBarContainer, SWT.NONE);
		selectionMenu = new SelectionMenu(rightBarContainer, SWT.NONE);
		
		createMainMenuToolBar(topComposite);
		
		dropTarget.addDropListener(new DropTargetAdapter() {

			@Override
			public void drop(DropTargetEvent event) {
				HashMap<String, Object> eventDetails = new HashMap<String, Object>();
				eventDetails.put("model name", event.data);
				Point canvasOrigin = canvas.toDisplay(0, 0);
				eventDetails.put("location", new Vector2i(event.x - canvasOrigin.x, event.y - canvasOrigin.y));
				EventBus.getInstance().notify(new MepEvent(EventType.DROP_OBJECT, eventDetails));
			}

		});
		Composite bottomComposite = new Composite(this, SWT.NONE);
		bottomComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		
		createDistanceDisplayers();
		setLayoutForSelection();

		subscribeToEvent();
		

	}

	private void createDistanceDisplayers() {
		distanceDisplays = new Shell[2];
		textFields = new Text[2];
		VerifyListener verifiListener = new VerifyListener() {
			@Override
			public void verifyText(VerifyEvent e) {
				 String input = e.text;
		            if (!input.matches("\\d*")) {
		                e.doit = false;
		            }				
			}
		};
		for(int i = 0; i <2; i++) {
			distanceDisplays[i] = new Shell(getDisplay(), SWT.BORDER);
			distanceDisplays[i].setSize(80, 40);
			distanceDisplays[i].setLayout(new GridLayout(1, false));
			distanceDisplays[i].setAlpha(192);
			textFields[i] = new Text(distanceDisplays[i], SWT.SINGLE);
			textFields[i].setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, true));
			textFields[i].setText("1234");
			textFields[i].addVerifyListener(verifiListener);
			distanceDisplays[i].open();
			distanceDisplays[i].setVisible(false);
		}
		
	}

	private void createMainMenuToolBar(Composite parent) {
		this.mainMenuToolBar = new ToolBar(parent, SWT.NONE);
		mainMenuToolBar = new ToolBar(parent, SWT.FLAT | SWT.RIGHT);
		mainMenuToolBar.setBackground(SWTResourceManager.getColor(SWT.COLOR_GRAY));
		mainMenuToolBar.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, true, false));
		//selcetion item
		ToolItem positionItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
		positionItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				EventBus.getInstance().notify(new MepEvent(EventType.OBJECT_POSITIONER_TRIGFERED, null) );
			}
		});
		String imagePath = ModelProvider.root + File.separator +  "icons/icons8-drag-32.png";
		Image positionImg = new Image(Display.getDefault(), imagePath);
		positionItem.setImage(positionImg);
		//plumbing menu item
		ToolItem rotationItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
		rotationItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				EventBus.getInstance().notify(new MepEvent(EventType.OBJECT_ROTATOR_TRIGGERED, null));
				
			}
		});
		imagePath = ModelProvider.root + File.separator +  "icons/icons8-3d-rotate-32.png";
		Image rotationImg = new Image(Display.getDefault(), imagePath);
		rotationItem.setImage(rotationImg);
		//electric menu item
		ToolItem scaleItem = new ToolItem(mainMenuToolBar, SWT.RADIO);
		scaleItem.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
			}
		});
		imagePath = ModelProvider.root + File.separator +  "icons/icons8-stretch-tool-32.png";
		Image scaleImg = new Image(Display.getDefault(), imagePath);
		scaleItem.setImage(scaleImg);
		positionItem.setSelection(true);
		
	}

	private void subscribeToEvent() {
		subscribe(EventType.OBJECT_SELECTION_TRIGFERED);
		subscribe(EventType.PLUMBING_EDITING_TRIGGERED);
		subscribe(EventType.SELECTION_MENU_TRIGGERED);
		subscribe(EventType.DISPLAY_DISTANCE_LABELS);
	}

	public GLCanvas getCanvas() {
		return canvas;
	}

	public void setCanvas(GLCanvas canvas) {
		this.canvas = canvas;
	}

	@Override
	public void handle(MepEvent event) {
		EventType type = event.getType();
		if (type == EventType.OBJECT_SELECTION_TRIGFERED) {
			setLayoutForObjectSelection();
		} else if (type == EventType.PLUMBING_EDITING_TRIGGERED) {
			setLayoutForPLumbing();
		}else if(type == EventType.SELECTION_MENU_TRIGGERED) {
			setLayoutForSelection();
		}else if(type == EventType.DISPLAY_DISTANCE_LABELS) {
			displayDistanceLabels(event.getDetails());
		}

	}

	private void displayDistanceLabels(HashMap<String, Object> details) {
		Vector3f[] locations = (Vector3f[]) details.get("locations");
		String[] labels = (String[]) details.get("labels");
		Vector3f[] anchors = (Vector3f[]) details.get("anchors");
		for(int i =0; i < 2; i++) {
			Point point =  canvas.toDisplay(Math.round( locations[i].x), Math.round( locations[i].y));
			distanceDisplays[i].setLocation(point);
			textFields[i].setText(labels[i]);
			Vector3f anchor = anchors[i];
			textFields[i].addListener(SWT.KeyUp, (e) -> {
				if(e.character == SWT.CR) {
					Text text = (Text) e.widget;
					Float value = Float.parseFloat(text.getText());
					ModelInstance target = (ModelInstance) details.get("target");
					var translation = new Vector3f(anchor).sub(target.getCenter()).add(
							new Vector3f(target.getCenter()).sub(anchor).normalize().mul(value/1000.0f));
					target.translate(translation);
				}
			});
			distanceDisplays[i].setVisible(true);
			distanceDisplays[i].setActive();
		}
	}

	private void setLayoutForPLumbing() {
		getDisplay().asyncExec(() -> {
			stackLayout.topControl = plumbingMenu;
			layout(true, true);
		});

	}

	private void setLayoutForObjectSelection() {
		getDisplay().asyncExec(() -> {
			stackLayout.topControl = library;
			layout(true, true);
		});
	}
	
	private void setLayoutForSelection() {
		getDisplay().asyncExec(()-> {
			stackLayout.topControl = selectionMenu;
			layout(true, true);
		});
	}
}
