package api.mep;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.widgets.Event;
import org.joml.Matrix4f;
import org.joml.Vector3f;
import org.lwjgl.opengl.GL45;

import com.badlogic.gdx.physics.box2d.Transform;
import com.sandmonkey.ttf_renderer.api.FreeTypeFont;
import com.sandmonkey.ttf_renderer.api.Text;

import api.backend.ApplicationContext;
import api.event.EventBus;
import api.event.EventType;
import api.event.MepEvent;
import api.graphics.Camera;
import api.graphics.IDrawable;
import api.graphics.Mesh;
import api.graphics.Model;
import api.graphics.ModelInstance;
import api.graphics.PbrMaterial;
import api.graphics.Shader;
import api.graphics.Vertex;
import api.graphics.VertexAttributes;
import api.graphics.geometry.BezierCurve;
import api.graphics.geometry.ShapeDrawer;
import api.input.InputMultiplexer;
import api.input.KeyboardEventHandler;
import api.input.MouseEventHandler;
import api.provider.FontProvider;
import api.utils.ScreenUtils;

public class PipeDrawer implements MouseEventHandler, KeyboardEventHandler, IDrawable{
	
	private Camera camera;
	private ShapeDrawer shapeDrawer;
	private static PipeDrawer drawer;
	private PIPE_TYPE type;
	private Text text;
	ArrayList<Vector3f[]> rings = new ArrayList<Vector3f[]>();
	
	
	private final Vector3f BLUE = new Vector3f(0,0,1);
	private final Vector3f RED = new Vector3f(1,0,0);
	private final Vector3f GRAY = new Vector3f(0.2f,0.2f,0.2f);
	private final Vector3f YELLOW = new Vector3f(0.2f,0.2f,0.2f);



	private ArrayList<Vector3f> trajectory = new ArrayList<Vector3f>();
	
	@Override
	public boolean handle(Event event) {
		if(!MouseEventHandler.super.handle(event))
			return KeyboardEventHandler.super.handle(event);
		return true;
	}
	
	public void begin(Camera camera, PIPE_TYPE pipeType) {
		this.camera = camera;
		trajectory.clear();
		rings.clear();
		this.type = pipeType;
	}
	
	@Override
	public boolean mouseDoubleClick(MouseEvent e) {
		return false;
	}

	@Override
	public boolean mouseDown(MouseEvent e) {
		if(e.button == 1) {
			float z = ScreenUtils.readDepth(e.x, e.y);
			float y = ApplicationContext.getHeight() - e.y;
			Vector3f worldPoint =  camera.unproject(new Vector3f(e.x, y, z), ApplicationContext.getViewport());
			addPoint(worldPoint);
		}
		return false;
	}

	private void addPoint(Vector3f worldPoint) {
		int size = trajectory.size();
		if(size > 0) {
			var lastPoint = trajectory.get(size -1);
			Vector3f v = new Vector3f(worldPoint).sub(trajectory.get(size -1));
			if(v.x * v.x > Math.max(v.y * v.y, v.z * v.z)) {
				v.mul(1,0,0);
			}else if(v.y * v.y > Math.max(v.x * v.x, v.z * v.z)){
				v.mul(0,1,0);
			}else {
				v.mul(0,0,1);
			}
			worldPoint.set(lastPoint).add(v);
		}
		trajectory.add(worldPoint);		
	}

	@Override
	public boolean mouseUp(MouseEvent e) {
		return false;
	}

	@Override
	public boolean mouseMove(MouseEvent e) {
		return false;
	}

	@Override
	public boolean mouseScrolled(MouseEvent e) {
		return false;
	}

	@Override
	public boolean keyPressed(KeyEvent e) {
		if(e.keyCode == SWT.CR) {
			cretaePipe(16);
			return true;
		}
		return false;
	}

	
	public void calculateAxisPoint() {
		if(trajectory.size() < 2)
			return;
		if(trajectory.size() > 2) {
			ArrayList<Vector3f> axisPoint = new ArrayList<Vector3f>();
			axisPoint.add(trajectory.get(0));
			for(int i = 1; i < trajectory.size() - 1; i++) {
				Vector3f current = new Vector3f(trajectory.get(i));
				Vector3f previous = new Vector3f(trajectory.get(i-1)).sub(trajectory.get(i));
				Vector3f next = new Vector3f(trajectory.get(i+1)).sub(trajectory.get(i));
				if(next.cross(previous, new Vector3f()).lengthSquared()/(next.lengthSquared() * previous.lengthSquared()) < 0.01f) {
					axisPoint.add(trajectory.get(i));
				}else {
					ArrayList<Vector3f> controlPoints = new ArrayList<Vector3f>();
					Vector3f p = new Vector3f(current).add(new Vector3f(previous).normalize().mul(0.02f));
					Vector3f n = new Vector3f(current).add(new Vector3f(next).normalize().mul(0.02f));
					controlPoints.add(p);
					controlPoints.add(current);
					controlPoints.add(n);
					BezierCurve curve = new BezierCurve(controlPoints);
					axisPoint.addAll(curve.calculatePoint(10));
				}
			}
			axisPoint.add(trajectory.get(trajectory.size() - 1));
			trajectory.clear();
			trajectory.addAll(axisPoint);
		}
	}
	
	private void cretaePipe(int ringSegs) {
		calculateAxisPoint();
		calculateRings(ringSegs);
		ArrayList<Integer> indicesList = new ArrayList<Integer>();
		int verticesCount = ringSegs * (trajectory.size());
		Vertex[] vertices = new Vertex[verticesCount];
		int c = 0;
		for(int k = 0; k < rings.size(); k++) {
			var ring = rings.get(k);
			for(int i = 0; i < ringSegs; i++) {
				vertices[c] = new Vertex( ring[i]);
				c++;
			}
		}
		
		for(int i = 0; i < trajectory.size() - 1; i++) {
			int offset = i * ringSegs;
			for(int j = 0; j < ringSegs; j++) {
				Integer i0 = offset + j;
				Integer i1 = offset + ((j + 1)%ringSegs);
				Integer i2 = offset + ringSegs + ((j+1) % ringSegs);
				Integer i3 = offset + ringSegs + j;
				Integer[] quadIndices = new Integer[] {i0, i1, i2, i0, i2, i3};
				indicesList.addAll(Arrays.asList(quadIndices));
			}
		}
		
		
		int[] indices = new int[indicesList.size()];
		for(int i =0; i < indices.length ; i++)
			indices[i] = indicesList.get(i);
		PbrMaterial mtl = new PbrMaterial();
		mtl.setAlbedo(getPipeColor());
		Mesh mesh = new Mesh(vertices, indices, GL45.GL_TRIANGLES, VertexAttributes.POSITION, mtl, new Matrix4f());
		ModelInstance pipe = new ModelInstance(new Model(mesh));
		pipe.setQuotable(true);
		HashMap<String, Object> details = new HashMap<String, Object>();
		details.put("renderable", pipe);
		EventBus.getInstance().notify(new MepEvent(EventType.ADD_RENDERABLE, details));
		end();
	}

	private void end() {
		rings.clear();
		trajectory.clear();
		InputMultiplexer.getInstance().removeEventHandler(this);
		InputMultiplexer.getInstance().insertEventHandler(ObjectSelector.getSelector(), 0);
	}

	private void calculateRings(int ringSegments) {
		rings.clear();
		float ringRadius = 0.02f;
		Vector3f normal = new Vector3f();
		Vector3f cross = new Vector3f();
		Vector3f axisY = new Vector3f(0,1,0);
		float angularStep = (float) (Math.PI * 2 /ringSegments);
		Vector3f pointer = new Vector3f(ringRadius, 0, 0);

		for(int i = 0; i < trajectory.size() - 1; i++) {
			Vector3f current = trajectory.get(i);
			Vector3f next = trajectory.get(i + 1);
			normal.set(next).sub(current).normalize();
			axisY.cross(normal, cross);
			float dot = axisY.dot(normal);
			float angle = (float) Math.atan2(cross.length(), dot);
			Vector3f[] ring = new Vector3f[ringSegments];
			for(int j = 0; j < ringSegments; j++) {
				ring[j] = new Vector3f(pointer).rotateAxis(angularStep * j, 0, 1, 0)
						.rotateAxis(angle, cross.x, cross.y, cross.z).add(current);
			}
			rings.add(ring);
		}		
		//add last ring
		Vector3f last = trajectory.get(trajectory.size() - 1);
		Vector3f beforeLast = trajectory.get(trajectory.size() - 2);
		normal.set(last).sub(beforeLast).normalize();
		axisY.cross(normal, cross);
		float dot = axisY.dot(normal);
		float angle = (float) Math.atan2(cross.length(), dot);
		Vector3f[] ring = new Vector3f[ringSegments];
		for(int j = 0; j < ringSegments; j++) {
			ring[j] = new Vector3f(pointer).rotateAxis(angularStep * j, 0, 1, 0)
					.rotateAxis(angle, cross.x, cross.y, cross.z).add(last);
		}
		rings.add(ring);
	}

	@Override
	public boolean keyReleased(KeyEvent e) {
		if(e.keyCode == SWT.CR)
			System.out.println("enter released");
		return false;
		
	}

	public static PipeDrawer getDrawer() {
		if(drawer == null)
			drawer = new PipeDrawer();
		return drawer;
	}

	@Override
	public void draw(Shader shader) {
		if(shapeDrawer == null) {
			shapeDrawer = new ShapeDrawer();
			shapeDrawer.prepare();
			text = new Text();
		}
		if(!rings.isEmpty()) {
			for(var ring: rings) {
				shapeDrawer.begin(camera);
				shapeDrawer.drawPoints(ring, 2, new Vector3f(1,0,0));
				shapeDrawer.end();
			}
		}else if(trajectory.size() > 0) {
			Vector3f[] points = trajectory.toArray(new Vector3f[0]);
			shapeDrawer.begin(camera);
			shapeDrawer.drawPoints(points, 4, new Vector3f());
			shapeDrawer.drawLines(points, 2, getPipeColor(), new Matrix4f());
			shapeDrawer.end();
			//render text
		    if(points.length > 1) {
		    	 for(int i = 0; i < points.length - 1; i++) {
		    		 Vector3f currentPoint = points[i];
		    		 Vector3f nextPoint = points[i+1];
		    		 var screenPoint0 = camera.project(currentPoint, ApplicationContext.getViewport()).mul(1, 1, 0);
		    		 var screenPoint1 = camera.project(nextPoint, ApplicationContext.getViewport()).mul(1, 1 , 0);
		    		if(screenPoint0.distance(screenPoint1) > 50) {
			    		 Vector3f v = new Vector3f(screenPoint1).sub(screenPoint0);
			    		 float rotation = (float) Math.toDegrees(Math.atan2(v.y, v.x));
			    		 var center = screenPoint0.add(screenPoint1).div(2);
			    		 String measure = String.format("%.1f cm", nextPoint.distance(currentPoint) * 100);
			    		 
				 		 text.render(FontProvider.getProvider().getLargeFont(), measure, center.x , center.y, 1, rotation , new Vector3f(),camera);
		    		}
		    	}
		    }
		   
		}
		
		
	}

	private Vector3f getPipeColor() {
		switch (type) {
		case COLD: 
			return BLUE;
		case HOT:
			return RED;
		case EVACUATION:
			return GRAY;
		case GAZ:
			return YELLOW;
		default:
			return BLUE;
		}
	}

	@Override
	public boolean mouseDragged(MouseEvent e) {
		// TODO Auto-generated method stub
		return false;
	}

	

}
