package gdxapp.object3d;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.UUID;

import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder.VertexInfo;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.utils.MeshPartBuilder;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import dressing.mathutils.EarClipper;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Triangle;
import gdxapp.fabs3d.BezierCurve;
import geometry.Triangle3D;

public class GolaProfile extends DoorHandle {
	
	private final String ID = "8dbda3fc-7ebd-425d-a0a8-d749e96daf48";

	private transient Vector3[] definingPoints = { new Vector3(3, 0, 0), new Vector3(), new Vector3(0, 5, 0),
			new Vector3(18, 5, 0), new Vector3(23, 5, 0), new Vector3(23, 12, 0), new Vector3(23, 56.5f, 0),
			new Vector3(26, 56.5f, 0), new Vector3(26, 53.5f, 0) };
	private float thickness = 0.5f;
	private transient ArrayList<Vector3> axisPoints = new ArrayList<Vector3>();
	private transient Vector3[] contour;
	private transient ArrayList<Triangle3D> triangles;
		
	public GolaProfile() {
		super();
		name = "Gola";
		previewPath = "C:\\ProgramData\\supercad\\assets\\handlers\\gola.png";
		modelId = UUID.fromString(ID);
		dimension.set(0.6f,0.0575f,0.027f);
	}
	
	@Override
	public Model getModel() {
		if(model == null)
			createModel();
		return model;
	}



	public Model createModel() {
		clearMeshData();
		axisPoints.add(definingPoints[0]);
		axisPoints.add(definingPoints[1]);
		axisPoints.add(definingPoints[2]);
		BezierCurve curve = new BezierCurve(24, definingPoints[3], definingPoints[4], definingPoints[5]);
		curve.calculateAxisPoints();
		axisPoints.addAll(curve.getVertices());
		for (int i = 6; i < definingPoints.length; i++) {
			axisPoints.add(definingPoints[i]);
		}
		createContour();
		triangulate();
		clean();
		build();
		return model;
	}

	private void clearMeshData() {
		axisPoints.clear();
		contour = null;
		if(triangles != null)
			triangles.clear();
	}

	private void clean() {
		float epsilon = 0.01f;
		ArrayList<Vector3> vertices = new ArrayList<Vector3>();
		for(var t: triangles) {
			for(var vertex: t.getVertices()) {
				boolean exist = false;
				for(var vertexX: vertices) {
					if(vertexX.epsilonEquals(vertex, epsilon)) {
						exist = true;
						break;
					}
				}
				if(!exist) {
					vertices.add(vertex);
				}
			}
		}
		Vector3[] bounds =  MathUtilities.getBoundaries(vertices);
		Vector3 center = bounds[0].cpy().add(bounds[1]).scl(0.5f);
		for(var t: triangles) {
			for(var vertex: t.getVertices()) {
				vertex.sub(center).rotate(Vector3.Y, 90).scl(0.001f);
			}
		}
		
	}

	private void build() {
		// generate normals and uvs
		if(material != null && !material.isReady())
			material.prepare();
		ModelBuilder builder = new ModelBuilder();
		builder.begin();
		int attr = VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal
				| VertexAttributes.Usage.TextureCoordinates;
		MeshPartBuilder meshPartBuilder = builder.part("root", GL20.GL_TRIANGLES, attr, material);
		Vector3 reference = new Vector3();
		for (var triangle : triangles) {
			Vector3 p0 = triangle.getVertices()[0];
			Vector3 p1 = triangle.getVertices()[1];
			Vector3 p2 = triangle.getVertices()[2];
			Vector3 N = p1.cpy().sub(p0).crs(p2.cpy().sub(p0)).nor();
			Vector3 T = p1.cpy().sub(p0).nor();
			Vector3 B = N.cpy().crs(T);
			Matrix4 transform = new Matrix4().set(T, B, N, reference);
			Vector3 uv0 = p0.cpy().mul(transform);
			Vector3 uv1 = p1.cpy().mul(transform);
			Vector3 uv2 = p2.cpy().mul(transform);
			VertexInfo i0 = new VertexInfo();
			i0.setPos(p0);
			i0.setNor(N);
			i0.setUV(uv0.x, uv0.y);
			VertexInfo i1 = new VertexInfo();
			i1.setPos(p1);
			i1.setNor(N);
			i1.setUV(uv1.x, uv1.y);
			VertexInfo i2 = new VertexInfo();
			i2.setPos(p2);
			i2.setNor(N);
			i2.setUV(uv2.x, uv2.y);
			meshPartBuilder.triangle(i0, i1, i2);
		}
		model = builder.end();
	}

	private void triangulate() {
		ArrayList<Vector2> planeVertices = new ArrayList<Vector2>();
		for (int i = 0; i < contour.length; i++) {
			planeVertices.add(new Vector2(contour[i].x, contour[i].y));
		}
		float depth = 600;
		ArrayList<Triangle> plane = EarClipper.triangulate(planeVertices);
		ArrayList<Triangle3D> front = new ArrayList<Triangle3D>();
		ArrayList<Triangle3D> back = new ArrayList<Triangle3D>();
		var d = new Vector3(0, 0, depth / 2.0f);
		plane.stream().forEach(v -> {
			var v0 = new Vector3(v.getV0(), 0);
			var v1 = new Vector3(v.getV1(), 0);
			var v2 = new Vector3(v.getV2(), 0);

			front.add(new Triangle3D(v0.cpy().add(d), v1.cpy().add(d), v2.cpy().add(d)));
			back.add(new Triangle3D(v0.cpy().sub(d), v2.cpy().sub(d), v1.cpy().sub(d)));
		});
		triangles = new ArrayList<Triangle3D>();
		triangles.addAll(back);
		triangles.addAll(front);
		// side triangles
		ArrayList<Vector3> frontVertices = new ArrayList<Vector3>();
		ArrayList<Vector3> backVertices = new ArrayList<Vector3>();
		Arrays.asList(contour).forEach(v -> {
			frontVertices.add(new Vector3(v).add(d));
			backVertices.add(new Vector3(v).sub(d));
		});
		for (int i = 0; i < frontVertices.size() - 1; i++) {
			Triangle3D t1 = new Triangle3D(frontVertices.get(i).cpy(), backVertices.get(i).cpy(), frontVertices.get(i + 1).cpy());
			Triangle3D t2 = new Triangle3D(frontVertices.get(i + 1).cpy(), backVertices.get(i).cpy(), backVertices.get(i + 1).cpy());
			triangles.add(t1);
			triangles.add(t2);
		}
	}

	public void createContour() {
		int size = axisPoints.size();
		contour = new Vector3[axisPoints.size() * 2];
		var first = axisPoints.get(1).cpy().sub(axisPoints.get(0)).nor().scl(thickness).rotate(Vector3.Z, -90);
		contour[0] = axisPoints.get(0).cpy().add(first);
		contour[size * 2 - 1] = axisPoints.get(0).cpy().sub(first);
		for (int i = 1; i < axisPoints.size() - 1; i++) {
			int nextIndex = i + 1;
			int previousIndex = i - 1;
			Vector3 current = axisPoints.get(i);
			Vector3 next = axisPoints.get(nextIndex);
			Vector3 previous = axisPoints.get(previousIndex);
			Vector3 c0 = next.cpy().sub(axisPoints.get(i)).nor();
			Vector3 c1 = previous.cpy().sub(axisPoints.get(i)).nor();
			float alpha = (float) Math.asin(c0.cpy().crs(c1).len());
			System.out.println("angle: " + Math.toDegrees(alpha));
			float scale = (float) (thickness / Math.sin(alpha));
			System.out.println("scale: " + scale);
			Vector3 c = c0.cpy().add(c1).scl(scale);
			Vector3 n = c0.cpy().rotate(Vector3.Z, 90);
			c.scl(Math.signum(c.dot(n)));
			contour[i] = current.cpy().sub(c);
			contour[size * 2 - 1 - i] = current.cpy().add(c);
		}
		var last = axisPoints.get(size - 1).cpy().sub(axisPoints.get(size - 2)).nor().scl(thickness).rotate(Vector3.Z,
				-90);
		contour[size - 1] = axisPoints.get(size - 1).cpy().add(last);
		contour[size] = axisPoints.get(size - 1).cpy().sub(last);
	}

	public Vector3[] getDefiningPoints() {
		return definingPoints;
	}

	public void setDefiningPoints(Vector3[] definingPoints) {
		this.definingPoints = definingPoints;
	}

	public ArrayList<Vector3> getAxisPoints() {
		return axisPoints;
	}

	public void setAxisPoints(ArrayList<Vector3> axisPoints) {
		this.axisPoints = axisPoints;
	}

	public Vector3[] getContour() {
		return contour;
	}

	public void setContour(Vector3[] contour) {
		this.contour = contour;
	}

	public ArrayList<Triangle3D> getTriangles() {
		return triangles;
	}

	public void setTriangles(ArrayList<Triangle3D> triangles) {
		this.triangles = triangles;
	}

}
