package org.frs.html;

import java.util.ArrayList;
import java.util.Iterator;

import org.frs.utilities.StringUtilities;


public class CrudeHtmlElement implements Iterable<CrudeHtmlElement>{
	
	private String content;
	private String type;
	private CrudeHtmlElement parent;
	private final ArrayList<CrudeHtmlElement> children = new ArrayList<CrudeHtmlElement>();
	private final ArrayList<HtmlElementAttribute> attributes = new ArrayList<HtmlElementAttribute>();
	private String rawContent; 
	
	public CrudeHtmlElement() {}
	
	public CrudeHtmlElement(String content, String type) {
		super();
		this.content = content;
		this.type = type;
	}
	
	public CrudeHtmlElement copy() {
		CrudeHtmlElement clone = new CrudeHtmlElement(this.content, this.type);
		for(HtmlElementAttribute attr: attributes){
			clone.addAttribute(attr.cpy());
		}
		for(CrudeHtmlElement child: this.children) {
			clone.addChild(child.copy());
		}
		return clone;
	}

	public void addChild(CrudeHtmlElement... elements) {
		for(CrudeHtmlElement element: elements) {
			children.add(element);
			element.setParent(this);
		}
		
	}
	
	public void replaceChild(CrudeHtmlElement child, CrudeHtmlElement...replacements) {
		int index = children.indexOf(child) - 1;
		removeChild(child);
		for(CrudeHtmlElement element: replacements) {
			insertChild(element, ++index);
		}
	}
	
	public void insertChild(CrudeHtmlElement child, int index) {
		index = Math.max(0, index);
		if(index > children.size()) {
			children.add(child);
		}else{
			children.add(index, child);
		}
		child.setParent(this);
	}
	
	public void removeChild(CrudeHtmlElement child) {
		children.remove(child);
		child.setParent(null);
	}

	public void addAttribute(HtmlElementAttribute attr) {
		this.attributes.add(attr);
	}
	
	public String getAttr(String key) {
		for(HtmlElementAttribute attr: this.attributes) {
			if(attr.getName().equals(key))
				return attr.getValue().substring(1,attr.getValue().length() - 1);
		}
		return null;
	}
	
	public ArrayList<CrudeHtmlElement> getChildrenByType(String type){
		ArrayList<CrudeHtmlElement> children = new ArrayList<CrudeHtmlElement>();
		for(CrudeHtmlElement child: this) {
			if(child.getType().contentEquals(type)) {
				children.add(child);
			}
		}
		return children;
	}
	
	public String getRawContent() {
		return rawContent;
	}

	public void setRawContent(String rawContent) {
		this.rawContent = rawContent;
	}
	
	public void deserialize() {
		String openingTag = null;
		String closingTag = null;
		int contentStart = -1;
		int contentEnd = -1;
		int finished = 2; 
		int length = rawContent.length() - 1;
		boolean insideStringLiteral = false;
		for(int i = 0; i < rawContent.length(); i++) {
			char current = rawContent.charAt(i);
			if(current == '"')
				insideStringLiteral = !insideStringLiteral;			
			if(openingTag == null && rawContent.charAt(i) == '>' && !insideStringLiteral) {
				openingTag = rawContent.substring(0, i + 1);
				contentStart = i + 1;
				finished--;
				if(openingTag.endsWith("/>"))
					finished--;
			}
			if(closingTag == null && rawContent.charAt(length - i) == '<' && !insideStringLiteral) {
				closingTag = rawContent.substring(length - i);
				contentEnd = length - i;
				finished--;
			}
			if(finished == 0)
				break;
		}
		
		String strippedTag = StringUtilities.replaceUnprintableChars(removeTagDelimeter(openingTag), " ");
		
		String[] strs = strippedTag.split(" ");
		for(String str: strs) {
			if(!str.equals("")) {
				type = str;
				break;
			}
		}
		this.attributes.clear();
		this.attributes.addAll(extractAttributes(strippedTag));
		setContent(rawContent.substring(contentStart, Math.max(contentStart,contentEnd)));
	}

	public static String removeTagDelimeter(String tag) {
		int length = tag.length();
		return tag.substring(1, tag.endsWith("/>")?length-2:length - 1);
	}
	
	private ArrayList<HtmlElementAttribute> extractAttributes(String openingTag) {
		String metaString = new String(openingTag);
		ArrayList<HtmlElementAttribute> attributes = new ArrayList<HtmlElementAttribute>();
		int index = metaString.indexOf("=");
		while(index > 0) {
			String[] left = metaString.substring(0, index).split(" ");
			String rightpart = metaString.substring(index + 1);
			String right = null;
			int start = rightpart.indexOf("\"");
			int end = -1;
			for(int i = start +1; i < rightpart.length(); i++) {
				if(rightpart.charAt(i) == '\"') {
					end = i;
					break;
				}
			}
			if(end > start) {
				right = rightpart.substring(start, end + 1);
				attributes.add(new HtmlElementAttribute(left[left.length - 1], right));
				metaString= rightpart.substring(end + 1);
				index = metaString.indexOf("=");
			}
		}
		return attributes;
	}

	public ArrayList<HtmlElementAttribute> getAttributes() {
		return attributes;
	}

	public String getType() {
		return type;
	}

	public void setType(String type) {
		this.type = type;
	}

	public String getContent() {
		return content;
	}
	public void setContent(String content) {
		this.content = content;
	}
	public ArrayList<CrudeHtmlElement> getChildren() {
		return children;
	}
	
	
	
	public CrudeHtmlElement getParent() {
		return parent;
	}

	public void setParent(CrudeHtmlElement parent) {
		this.parent = parent;
	}

	public String writeAttribs() {
		String attributes = "";
		for(HtmlElementAttribute attr: this.attributes) {
			attributes += String.format(" %s=%s",attr.getName(), attr.getValue());
		}
		return attributes;
	}
	
	String getOpeningTag() {		
		return type.equals("generic")?"":String.format("<%s %s >", this.type, writeAttribs());
	}
	
	String getClosingTag() {
		return type.equals("generic")?"":String.format("</%s>", this.type);
	}

	@Override
	public String toString() {
		String children_content= "";
		for(CrudeHtmlElement child: children) {
			children_content += child.toString() + System.lineSeparator();
		}
		String str =  getOpeningTag() + System.lineSeparator() + "\t" + content + System.lineSeparator()  +  children_content + getClosingTag();
		
		return str;
	}

	@Override
	public Iterator<CrudeHtmlElement> iterator() {
		// TODO Auto-generated method stub
		return new CrudeHtmlElementIterator(this);
	}
	

}
