package geometry;

import java.util.ArrayList;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.glutils.ShapeRenderer;
import com.badlogic.gdx.math.Vector2;
import dressing.mathutils.MathUtilities;
import jakarta.xml.bind.annotation.XmlRootElement;

@XmlRootElement(name="Arc")
public class Arc implements Shape{
	
	private Vector2 center, v0, v1;
    private transient float radius;
    private final ArrayList<Vector2> vertices = new ArrayList<>();

    public Arc() {}

    public Arc(Vector2 center, float radius) {
        this.center = center;
        this.radius = radius;
    }
    
    
    
    public Arc(Vector2 center, Vector2 v0, Vector2 v1) {
		super();
		this.center = center;
		this.v0 = v0;
		this.v1 = v1;
		calculateVertices();
	}

	public Arc(ArrayList<Vector2> vertices) {
        setVertices(vertices);
    }

    public boolean contains(Vector2 point){
        return (center.dst(point) <= radius );
    }

    @Override
    public void draw(ShapeRenderer shapeRenderer, Color color) {
    	if(vertices != null && !vertices.isEmpty()) {
    		boolean halted = false;
    		if(!shapeRenderer.isDrawing()) {
    			shapeRenderer.begin(ShapeRenderer.ShapeType.Filled);
    			halted = true;
    		}
            shapeRenderer.setColor(color);
            for(int i =0; i < vertices.size() - 1; i++){
                shapeRenderer.rectLine(vertices.get(i%vertices.size()).x, vertices.get(i%vertices.size()).y,vertices.get((i+1)%vertices.size()).x, vertices.get((i+1)%vertices.size()).y,3);
            }
            shapeRenderer.setColor(Color.RED);
            shapeRenderer.circle(vertices.get(0).x,vertices.get(0).y,3);
            shapeRenderer.setColor(Color.GREEN);
            shapeRenderer.circle(vertices.get(vertices.size()-1).x,vertices.get(vertices.size()-1).y,3);
            if(halted)
            	shapeRenderer.end();
    	}

    }

    @Override
    public ArrayList<Vector2> getVertices() {
        return this.vertices;
    }

    @Override
    public Vector2 getlastVertex() {
    	if(vertices!= null)
    		return vertices.get(vertices.size() -1);
    	return null;
    }

    public void setCenter(Vector2 center) {
        this.center = center;
    }

    public void setRadius(float radius) {
        this.radius = radius;
    }

    public void setVertices(ArrayList<Vector2> vertices) {
        this.vertices.clear();
        for(int i = 0; i < vertices.size(); i ++){
            this.vertices.add(vertices.get(i));
        }
    }

    public Vector2 getCenter() {
        return center;
    }

    public float getRadius() {
        return radius;
    }
    
    

    public Vector2 getV0() {
		return v0;
	}

	public void setV0(Vector2 v0) {
		this.v0 = v0;
	}

	public Vector2 getV1() {
		return v1;
	}

	public void setV1(Vector2 v1) {
		this.v1 = v1;
	}

	public void drawCircle(ShapeRenderer shapeRenderer) {
        shapeRenderer.begin(ShapeRenderer.ShapeType.Line);
        shapeRenderer.circle(center.x, center.y, radius, 50);
        shapeRenderer.end();
    }

	@Override
	public void scale(Vector2 scale) {
		this.v0.scl(scale);
		this.v1.scl(scale);
		this.center.scl(scale);
	}
	
	void calculateVertices(){
		 	this.radius = v0.dst(center);
	        float angle = MathUtilities.signedAngle(v0.cpy().sub(center),v1.cpy().sub(center));
	        int segs = (int) Math.abs((angle * radius/3));
	        Vector2 pointer = new Vector2(v0).sub(center);
	        ArrayList<Vector2> points = new ArrayList<Vector2>();
	        points.add(v0);
	        float step = angle/segs;
	        for(int i = 0; i < segs; i++){
	            pointer.rotateRad(step);
	            points.add(new Vector2(center).cpy().add(pointer));
	        }
	        points.add(v1);
	        setVertices(points);
	}
	
	
	public boolean contains(Vector2 point, float precision) {
		if(Math.abs(point.dst(center) - radius ) > precision)
			return false;
		Vector2 v0c = v0.cpy().sub(center);
		Vector2 v1c = v1.cpy().sub(center);
		Vector2 vpc = point.cpy().sub(center);
		float a10 = MathUtilities.calculateUnsignedAngle(v0c, v1c);
		float p0 = MathUtilities.calculateUnsignedAngle(vpc, v0c);
		float p1 = MathUtilities.calculateUnsignedAngle(vpc, v1c);
		return 	Math.abs(p0 + p1 - a10  ) < precision;

	}
	
	

	@Override
	public Shape cpy() {
		Arc clone = new Arc();
		clone.setCenter(this.center.cpy());
		clone.setV0(this.v0.cpy());
		clone.setV1(this.v1.cpy());
		clone.calculateVertices();
		return clone;
	}

	@Override
	public float distanceTo(Vector2 position) {
		radius = v0.dst(center);
		Vector2 pc = position.cpy().sub(this.center);
		float angle = MathUtilities.signedAngle(v0.cpy().sub(center), pc);
		float angleX = MathUtilities.signedAngle(v0.cpy().sub(center), v1.cpy().sub(center));
		if(angle * angleX > 0 && Math.abs(angle)  < Math.abs(angleX)) {
			return center.dst(position) - radius;
		}
		return Math.min(position.dst(v0), position.dst(v1));
	}

	@Override
	public Vector2 projectPoint(Vector2 point) {
		Vector2 projected = point.cpy().sub(center).nor().scl(radius).add(center);
		if(contains(projected, 0.0001f))
			return projected;
		return projected.dst(v0) < projected.dst(v1)?v0.cpy():v1.cpy();
	}

	@Override
	public Object select(Vector2 position) {
		float distanceToV0 = position.dst(v0), distanceToV1 = position.dst(v1), distanceToCenter = position.dst(center);
		float minDis = Math.min(Math.min(distanceToV0, distanceToV1), distanceToCenter);
		if(minDis > 40)
			return null;
		if(minDis == distanceToV0)
			return v0;
		if(minDis == distanceToV1)
			return v1;
		return center;
	}

	@Override
	public int getWinding() {
		Vector2 ce = v1.cpy().sub(v0);
		Vector2 cs = v0.cpy().sub(v0);
		return (int) Math.signum(ce.angle(cs));
	}

	@Override
	public void setWinding(int winding) {
		int currentWinding = MathUtilities.getWinding(this.vertices.toArray(new Vector2[0]));
		if(currentWinding * winding < 0) {
			Vector2 tmp = v0.cpy();
			v0.set(v1);
			v1.set(tmp);
			calculateVertices();
		}
	}





}
