package gdxapp.shaders;

import java.util.ArrayList;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.Cubemap;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.g3d.Renderable;
import com.badlogic.gdx.graphics.g3d.Shader;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.FloatAttribute;
import com.badlogic.gdx.graphics.g3d.attributes.TextureAttribute;
import com.badlogic.gdx.graphics.g3d.utils.RenderContext;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.utils.GdxRuntimeException;

import dressing.config.WorkspaceConfiguration;
import dressing.model.ModelProvider;
import gdxapp.assets.AssetsTextures;

public class MapBasedShader implements Shader {
	
	private ShaderProgram program;
    private Camera camera;
    private RenderContext context;
    private ArrayList<Light> lights;
    private int directionalLightCounter = 0;
    private int spotLightCounter = 0;
    private float deltaTime = 0;
	private int poinctualLightCounter;
	private boolean mapEnvironment;
	private Cubemap environmentMap;

    @Override
    public void init() {
        String vertx = Gdx.files.absolute(WorkspaceConfiguration.SHADERS_PATH + "/normal/vertx.glsl").readString();
        String frag = Gdx.files.absolute(WorkspaceConfiguration.SHADERS_PATH + "/normal/frag.glsl").readString();
        program = new ShaderProgram(vertx,frag);
        if(!program.isCompiled())
            throw new GdxRuntimeException(program.getLog());
        program.pedantic = false;
    }

    @Override
    public int compareTo(Shader other) {
        return 0;
    }

    @Override
    public boolean canRender(Renderable instance) {
    	
    	 return false;//instance.material.has(TextureAttribute.Normal) && !instance.userData.equals("defaultshader");
    }

    @Override
    public void begin(Camera camera, RenderContext context) {
        this.camera = camera;
        this.context = context;
        context.setDepthTest(GL20.GL_LEQUAL);
        context.setCullFace(GL20.GL_BACK);
        program.begin();
        program.setUniformMatrix("projection", camera.projection);
        program.setUniformMatrix("view", camera.view);
        program.setUniformf("viewPos",camera.position);
        setUpLightning();
    }

    @Override
    public void render(Renderable renderable) {
        program.setUniformMatrix("model", renderable.worldTransform);
        if(renderable.material.has(TextureAttribute.Diffuse)){
            TextureAttribute textureAttribute = (TextureAttribute) renderable.material.get(TextureAttribute.Diffuse);
            if(textureAttribute.textureDescription.texture.getTextureData() != null){
                program.setUniformi("diffuseMap", context.textureBinder.bind(textureAttribute.textureDescription));
            }
        }

        if(renderable.material.has(TextureAttribute.Normal)){
            TextureAttribute textureAttribute = (TextureAttribute) renderable.material.get(TextureAttribute.Normal);
            if(textureAttribute.textureDescription.texture.getTextureData() != null){
                program.setUniformi("normalMap", context.textureBinder.bind(textureAttribute.textureDescription));
            }
        }
        if(renderable.material.has(TextureAttribute.Specular)){
            TextureAttribute textureAttribute = (TextureAttribute) renderable.material.get(TextureAttribute.Specular);
            if(textureAttribute.textureDescription.texture.getTextureData() != null){
                program.setUniformi("specularMap", context.textureBinder.bind(textureAttribute.textureDescription));
            }
        }
        if(renderable.material.has(ColorAttribute.Specular)){
            Color spec = ((ColorAttribute)renderable.material.get(ColorAttribute.Specular)).color;
            program.setUniformf("material.specular",new Vector3(spec.r,spec.g,spec.b));
        }
        program.setUniformf("material.shininess",16);
        if(renderable.material.has(FloatAttribute.Shininess)){
            Float shininess  = ((FloatAttribute)renderable.material.get(FloatAttribute.Shininess)).value;
            program.setUniformf("material.shininess",shininess);
        }
        program.setUniformf("material.opacity",1.0f);
        if(renderable.material.has(FloatAttribute.AlphaTest)){
            Float opacity  = ((FloatAttribute)renderable.material.get(FloatAttribute.AlphaTest)).value;
            program.setUniformf("material.opacity",opacity);
        }
        program.setUniformf("material.reflectivity",0.0f);
        if(renderable.material.has(ColorAttribute.Reflection)){
            Color ref = ((ColorAttribute)renderable.material.get(ColorAttribute.Reflection)).color;
            program.setUniformf("material.reflectivity",ref.r);
        }
        
        if(environmentMap != null) {
            program.setUniformi("cubeMap", context.textureBinder.bind(environmentMap));
        }
        float reflection=0.0f;
//        if(ModelProvider.getModelroot()!=null &&ModelProvider.getModelroot().getConfig()!=null) {
//        	if(ModelProvider.getModelroot().getConfig().isSceneReflection()) {
        	if(true) {
        		reflection=1.0f;
        	}
//        }
        program.setUniformf("reflection", reflection);
        renderable.meshPart.render(program);
    }

    @Override
    public void end() {
        program.end();
    }

    @Override
    public void dispose() {
        program.dispose();
    }

    public void addDirectionalLight(DirectionalLight light) {
        program.setUniformf("DLights[" + directionalLightCounter+ "].Intensity", light.getIntensity());
        program.setUniformf("DLights[" + directionalLightCounter+ "].Direction", light.getDirection());
    }
    
    public void addSpotLight(SpotLight light) {
        program.setUniformf("SLights[" + spotLightCounter+ "].intensity", light.getIntensity());
        program.setUniformf("SLights[" + spotLightCounter+ "].direction", light.getDirection());
        program.setUniformf("SLights[" + spotLightCounter+ "].exponent", light.getExponent());
        program.setUniformf("SLights[" + spotLightCounter+ "].cutoff", light.getCutoff());
        program.setUniformf("SLights[" + spotLightCounter+ "].position", light.getPosition());
    }

    public void setUpLightning() {
    	 //reset counters
        directionalLightCounter = 0;
        spotLightCounter = 0;
        poinctualLightCounter = 0;
        for(Light light: lights) {
        	if(light instanceof DirectionalLight) {
                addDirectionalLight((DirectionalLight) light);
                directionalLightCounter++;
        	}
        	if(light instanceof SpotLight) {
        		((SpotLight) light).getPosition().y = (float) (1.5f);
        		((SpotLight) light).setDirection(new Vector3((float) (  Math.sin(0.5f * deltaTime)), 0, (float) ( Math.cos(0.5f * deltaTime))).nor().scl(-1));
        		light.setIntensity(new Vector3((float)(1 + Math.sin(deltaTime)), (float)(1 + Math.cos(deltaTime)), (float) (1 + Math.sin(2 * deltaTime))).nor());
        		addSpotLight((SpotLight) light);
        		spotLightCounter++;
        	}
        	if(light instanceof PonctualLight) {
        		addPoinctualLight((PonctualLight) light);
        		poinctualLightCounter++;
        		
        	}
        }
    }
    
    private void addPoinctualLight(PonctualLight light) {
   	 program.setUniformf("PLights[" + poinctualLightCounter+ "].intensity", light.getIntensity());
     program.setUniformf("PLights[" + poinctualLightCounter+ "].position", light.getPosition());
	}


    public ArrayList<Light> getLights() {
        return lights;
    }

    public void setLights(ArrayList<Light> lights) {
        this.lights = lights;
    }

	public float getDeltaTime() {
		return deltaTime;
	}

	public void setDeltaTime(float deltaTime) {
		this.deltaTime += deltaTime;
	}

	public void setMapEnvironment(boolean b) {
		this.mapEnvironment = b;
	}

	public Cubemap getEnvironmentMap() {
		return environmentMap;
	}

	public void setEnvironmentMap(Cubemap environmentMap) {
		this.environmentMap = environmentMap;
	}
	
	
	
	
    
    

}
