package gdxapp.screens.ObjectEditor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.Input;
import com.badlogic.gdx.InputProcessor;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.GL20;
import com.badlogic.gdx.graphics.Mesh;
import com.badlogic.gdx.graphics.VertexAttribute;
import com.badlogic.gdx.graphics.VertexAttributes;
import com.badlogic.gdx.graphics.VertexAttributes.Usage;
import com.badlogic.gdx.graphics.g3d.Material;
import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.ModelBatch;
import com.badlogic.gdx.graphics.g3d.ModelInstance;
import com.badlogic.gdx.graphics.g3d.Renderable;
import com.badlogic.gdx.graphics.g3d.Shader;
import com.badlogic.gdx.graphics.g3d.attributes.ColorAttribute;
import com.badlogic.gdx.graphics.g3d.model.MeshPart;
import com.badlogic.gdx.graphics.g3d.utils.MeshBuilder;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.graphics.glutils.ShaderProgram;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;

import dressing.events.Event;
import dressing.events.EventDriver;
import dressing.events.EventHandler;
import gdxapp.object3d.Object3D;
import gdxapp.scenes.SceneEvent;
import gdxapp.shaders.DiffuseShader;

public class BoxBuilder implements DrawableInputProcessor, EventHandler {

	Camera camera;

	private final Vector3 v0 = new Vector3(Float.NaN, Float.NaN, Float.NaN);
	private final Vector3 v1 = new Vector3(Float.NaN, Float.NaN, Float.NaN);
	private float height = Float.NEGATIVE_INFINITY;

	private final Vector3 tmp = new Vector3();
	
	private final Vector3 tmpNormal = new Vector3(Float.NaN, Float.NaN, Float.NaN);
	
	private final Vector3 p0 = new Vector3(Float.NaN, Float.NaN, Float.NaN),
						  p1 = new Vector3(Float.NaN, Float.NaN, Float.NaN),
						  p2 = new Vector3(Float.NaN, Float.NaN, Float.NaN),
			              p3 = new Vector3(Float.NaN, Float.NaN, Float.NaN);
			  
	
	private final Vector3 d0 = new Vector3(Float.NaN, Float.NaN, Float.NaN),
						  d1 = new Vector3(Float.NaN, Float.NaN, Float.NaN),
						  d2 = new Vector3(Float.NaN, Float.NaN, Float.NaN),
						  d3 = new Vector3(Float.NaN, Float.NaN, Float.NaN);

	private Mesh mesh;
	
	boolean acquireV2 = false;
	float y1 = Float.NEGATIVE_INFINITY;
	float y2 = Float.NEGATIVE_INFINITY;
	float v1depth;

	public BoxBuilder(Camera camera) {
		this.camera = camera;
		subscribe(SceneEvent.TARGET_CAMERA_CHANGED.name());
	}

	public void draw(ShaderProgram shader) {
		if (!Float.isNaN(v0.x) && !Float.isNaN(v1.x)) {
			calculateVertices();
			updateDrawingPoints();
			ArrayList<Float> vertices = new ArrayList<Float>();
			Float bottom_face[] = { d0.x, d0.y, d0.z,
					 				d1.x, d1.y, d1.z,
									
					 				d1.x, d1.y, d1.z,
					 				d2.x, d2.y, d2.z,
									
					 				d2.x, d2.y, d2.z,
					 				d3.x, d3.y, d3.z,
									
					 				d3.x, d3.y, d3.z,
					 				d0.x, d0.y, d0.z};
			Collections.addAll(vertices, bottom_face);
			if (!Float.isNaN(height)) {
				Vector3 translation = tmpNormal.cpy().scl(height);
				Float top_face[] = { 
									d0.x + translation.x, d0.y + translation.y, d0.z + translation.z,
									d1.x + translation.x, d1.y + translation.y, d1.z + translation.z,
						
									d1.x + translation.x, d1.y + translation.y, d1.z + translation.z,
									d2.x + translation.x, d2.y + translation.y, d2.z + translation.z,
						
									d2.x + translation.x, d2.y + translation.y, d2.z + translation.z,
									d3.x + translation.x, d3.y + translation.y, d3.z + translation.z,
						
									d3.x + translation.x, d3.y + translation.y, d3.z + translation.z,
									d0.x + translation.x, d0.y + translation.y, d0.z + translation.z};
				
				Float side_edges[] = {  d0.x, d0.y, d0.z,
										d0.x + translation.x, d0.y + translation.y, d0.z + translation.z,
									  
										d1.x, d1.y, d1.z,
										d1.x + translation.x, d1.y + translation.y, d1.z + translation.z,
										
										d2.x, d2.y, d2.z,
										d2.x + translation.x, d2.y + translation.y, d2.z + translation.z,
										
										d3.x, d3.y, d3.z,
										d3.x + translation.x, d3.y + translation.y, d3.z + translation.z
				};
				Collections.addAll(vertices, top_face);
				Collections.addAll(vertices, side_edges);

			}

		
			float vertexData[] = new float[vertices.size()];
			for (int i = 0; i < vertices.size(); i++) {
				vertexData[i] = vertices.get(i);
			}
			mesh = new Mesh(true, vertices.size(), 0, // static mesh with 4 vertices and no indices
					new VertexAttribute(Usage.Position, 3, ShaderProgram.POSITION_ATTRIBUTE));
			mesh.setVertices(vertexData);
		}
		if (mesh != null) {
			shader.begin();
			Gdx.gl.glEnable(GL20.GL_DEPTH_TEST);
			Gdx.gl.glDepthFunc(GL20.GL_LEQUAL);
			shader.setUniformf("color", 1.0f, 1.0f, 0.0f, 1.0f);
			shader.setUniformMatrix("model", new Matrix4());
			mesh.render(shader, GL20.GL_LINES);
			shader.end();
		}

	}
	
	private void calculateVertices() {
		Vector3 v = v1.cpy().sub(v0);
		p0.set(v0);
		p1.set(v.x, 0, 0).add(v0);
		p2.set(v1);
		p3.set(0, v.y,v.z).add(v0);
		
	}

	private void updateDrawingPoints() {
		float deviation = Math.signum(camera.position.cpy().sub(v0).dot(tmpNormal)) * 0.01f; //move the points 1 cm in the direction of camera to avoid z fighting
		d0.set(p0).add(tmpNormal.cpy().scl(deviation));
		d1.set(p1).add(tmpNormal.cpy().scl(deviation));
		d2.set(p2).add(tmpNormal.cpy().scl(deviation));
		d3.set(p3).add(tmpNormal.cpy().scl(deviation));


	}

	private void reset() {
		v0.set(Float.NaN, Float.NaN, Float.NaN);
		v1.set(Float.NaN, Float.NaN, Float.NaN);
		height = Float.NEGATIVE_INFINITY;
		y1 = Float.NEGATIVE_INFINITY;
		mesh = null;
	}

	private void buildBox(Vector3 center, Vector3 dimensions) {
		ModelBuilder  modelBuilder = new ModelBuilder();
		Model model =  modelBuilder.createBox(dimensions.x, dimensions.y, dimensions.z, new Material(ColorAttribute.createDiffuse(Color.WHITE)),
				VertexAttributes.Usage.Position | VertexAttributes.Usage.Normal);
		Object3D instance = new Object3D(model);
		instance.transform.setToTranslation(center);
		instance.calculateProperties();
		Event event = new Event(SceneEvent.OBJECT_CREATED.name(), instance);
		EventDriver.getDriver().deliverEvent(event);
		reset();
	}
	
	
	private void calculateNormal() {
		Vector3 v = v1.cpy().sub(v0).nor();
		float dotX = Math.abs(v.dot(1, 0, 0)), dotY = Math.abs(v.dot(0,1,0)), dotZ = Math.abs(v.dot(0, 0, 1)); 
		float min = Math.min(dotX, Math.min(dotY, dotZ));
		if(min == dotX) {
			tmpNormal.set(1, 0, 0);
		}else if(min == dotY) {
			tmpNormal.set(0, 1, 0);
		} else {
			tmpNormal.set(0, 0, 1);
		}
	}

	@Override
	public boolean keyDown(int keycode) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public boolean keyUp(int keycode) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public boolean keyTyped(char character) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public boolean touchDown(int screenX, int screenY, int pointer, int button) {
		if(button == Input.Buttons.LEFT) {
			if(Float.isNaN(v0.x)) {
				v0.set(tmp);
			}else if(Float.isFinite(height)) {
				Vector3 center = v1.cpy().add(v0).add(tmpNormal.cpy().scl(height)).scl(0.5f);
				Vector3 v = v1.cpy().sub(v0).add(tmpNormal.cpy().scl(height));
				Vector3 dims = new Vector3(Math.abs(v.x), Math.abs(v.y), Math.abs(v.z));
				buildBox(center, dims);

				reset();
			}
			
			return true;
		}
		return false;
		
	}



	@Override
	public boolean touchUp(int screenX, int screenY, int pointer, int button) {
		if(!(Float.isNaN(v0.x) || Float.isNaN(v1.x))) {
			acquireV2 = true;
			return true;
		}else {
			v0.set(Float.NaN, Float.NaN, Float.NaN);
			acquireV2 = false;
			return false;
		}
	}

	@Override
	public boolean touchDragged(int screenX, int screenY, int pointer) {
		if (!Float.isNaN(v0.x)) {
			float z = ScreenUtilities.getDepthFromBuffer(screenX, Gdx.graphics.getHeight() - screenY);
			v1.set(camera.unproject(new Vector3(screenX, screenY, z)));
			v1depth = z;
			calculateNormal();
			return true;
		}
		return false;
	}



	@Override
	public boolean mouseMoved(int screenX, int screenY) {
		float z = ScreenUtilities.getDepthFromBuffer(screenX, Gdx.graphics.getHeight() - screenY);
		tmp.set(camera.unproject(new Vector3(screenX, screenY, z)));
		Event event = new Event(SceneEvent.MSG.name(), tmp.toString());
		EventDriver.getDriver().deliverEvent(event);
		if(acquireV2) {
			Vector3 v1Screen = camera.project(v1.cpy());
			Vector3 v2Screen = new Vector3(screenX, screenY, v1Screen.z);
			height = camera.unproject(v2Screen.cpy()).sub(v1).len() * Math.signum(v2Screen.y - v1Screen.y);	

		}
		return true;
	}

	@Override
	public boolean scrolled(int amount) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public void draw(ModelBatch batch) {
		DiffuseShader shader = (DiffuseShader) batch.getShaderProvider().getShader(new Renderable());
		draw(shader.getProgram());
	}

	public void setCamera(Camera camera) {
		this.camera = camera;
	}

	@Override
	public void handle(Event event) {
		SceneEvent sceneEvent = SceneEvent.valueOf(event.getTopic());
		switch(sceneEvent) {
			case TARGET_CAMERA_CHANGED:
			cameraChanged(event);
		}
	}
	
	private void cameraChanged(Event event) {
		if(event.getData() instanceof Camera) {
			setCamera((Camera) event.getData());
		}
	}

}
