package gdxapp.object3d;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Optional;

import com.badlogic.gdx.graphics.g3d.Model;
import com.badlogic.gdx.graphics.g3d.utils.ModelBuilder;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.math.collision.BoundingBox;
import dressing.mathutils.Direction;
import dressing.mathutils.Edge;
import dressing.mathutils.MathUtilities;
import dressing.mathutils.Surface;
import dressing.model.ProjectManager;
import dressing.model.persistence.dpos.WallDPO;
import dressing.model.persistence.mappers.Persistable;
import gdxapp.screens.room.RoomController;
import gdxapp.screens.wall.Wall2D;
import gdxapp.screens.wall.WallFragment;
import gdxapp.shaders.PbrMaterial;
import geometry.Arc;
import geometry.CompoundObject;
import geometry.CompoundShape;
import geometry.Shape;


@Persistable(persistableForm = WallDPO.class)
public class Wall extends GeometryObject{
	
	private Direction selectedSurface;	
	private float thickness;
	private float height;
	private PbrMaterial material;
	transient protected PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this.getClass());
		
	private transient Object3D[] fragments3D;
	
	public Wall() {
		super();
		setType(ObjectType.POLY);
		moveable = false;
		setStaticobject(true);
		name = "Wall";
		this.material = ProjectManager.getManager().getCurrentScene().getPreferences().getWallMtl();
		this.geometry = new CompoundObject();
	}
	
	
	public Wall(float height, float thickness) {
		this();
		this.height = height;
		this.thickness = thickness;
		this.geometry = new CompoundObject();
	}
	
	
	
	@Override
	public WallDPO toDPO() {
		return new WallDPO(this);
	}


	public Object3D[] get3DObjects() {
		if(fragments3D == null)
			 buildFragments();
		return fragments3D;
	}
	
	public void buildFragments() {
		
		if(getPerimeter() != null) {
			ArrayList<Model> models = new ArrayList<Model>();
			ArrayList<Vector2> outerContour = MathUtilities.calculateSurfaceContour(getPerimeter().getVertices(), getPerimeter().isClosed(),thickness * RoomController.getInstance().getScaleX());
			ArrayList<Vector2> innerContour = new ArrayList<Vector2>();
			int size = getPerimeter().getVertices().size();
			for(int i = 0; i < size; i++) {
				innerContour.add(getPerimeter().getVertices().get(size - 1 - i));
			}
			if(!getPerimeter().isClosed()) {
				outerContour.addAll(innerContour);
				innerContour.clear();
			}
			if(!getMaterial().isReady()) {
				material.load();
				material.prepare();
			}
			ModelBuilder modelBuilder = PolygonBuilder.getModelBuilder();
			for(Edge edge: getPerimeter().getEdges()) {
				Optional<WallFragment> fragment = ProjectManager.getManager().getCurrentScene().findFragment(edge);
				float thickness = this.thickness;
				float height = this.height;

				if(fragment.isPresent()) {
					thickness = fragment.get().getThickness();
					height = fragment.get().getHeight();
				}
				Vector3 v0 = new Vector3(edge.getV0(),0.0f).mul(RoomController.getInstance().getSceneToWorldTransform());
				Vector3 v1 = new Vector3(edge.getV0(),height).mul(RoomController.getInstance().getSceneToWorldTransform());
				Vector3 v2 = new Vector3(edge.getV1(),height).mul(RoomController.getInstance().getSceneToWorldTransform());
				Vector3 v3 = new Vector3(edge.getV1(),0).mul(RoomController.getInstance().getSceneToWorldTransform());
				float[] door = ProjectManager.getManager().getCurrentScene().getDoors(edge, getPerimeter().getWorldTransform());
				Surface surface = new Surface(v0, v1, v2,v3);
				ArrayList<Vector2> border = new ArrayList<Vector2>();
				border.add(surface.getPlaneCoords(v0));
				border.add(surface.getPlaneCoords(v1));
				border.add(surface.getPlaneCoords(v2));
				border.add(surface.getPlaneCoords(v3));
				if(door != null) {
					Vector3 center = new Vector3(door[0],door[1], 0).mul(getPerimeter().getWorldTransform().cpy());
					float scaleH =door[2]/2f;
					float scaleV =door[3];
					Vector3 fullV = v1.cpy().sub(v0).nor().scl(scaleV);
					Vector3 halfH = v2.cpy().sub(v1).nor().scl(scaleH);
					Vector3 v4, v5 ,v6 ,v7;
					v4 = center.cpy().add(halfH);
					v5 = v4.cpy().add(fullV);
					v6 = v5.cpy().sub(halfH.scl(2.0f));
					v7 = v6.cpy().sub(fullV);
					border.add(surface.getPlaneCoords(v4));
					border.add(surface.getPlaneCoords(v5));
					border.add(surface.getPlaneCoords(v6));
					border.add(surface.getPlaneCoords(v7));
				}
				MathUtilities.setWinding(border, 1);
				ArrayList<Vector2> hole = new ArrayList<Vector2>();

				float[] constraint = ProjectManager.getManager().getCurrentScene().getConstraints(edge,getPerimeter().getWorldTransform()); 
				if(constraint != null) { 
					float scaleH =constraint[3]/2f;
					float scaleV =constraint[4]/2f;
					Vector3 center = new Vector3(constraint[0],constraint[1], constraint[2]).mul(RoomController.getInstance().getSceneToWorldTransform());
					Vector3 halfV = v1.cpy().sub(v0).nor().scl(scaleV);
					Vector3 halfH = v2.cpy().sub(v1).nor().scl(scaleH);
					hole.add(surface.getPlaneCoords(center.cpy().add(halfV).add(halfH)));
					hole.add(surface.getPlaneCoords(center.cpy().add(halfV).sub(halfH)));
					hole.add(surface.getPlaneCoords(center.cpy().sub(halfV).sub(halfH)));
					hole.add(surface.getPlaneCoords(center.cpy().sub(halfV).add(halfH)));
				}
				modelBuilder.begin();
				PolygonBuilder.createPolygoneNode(modelBuilder, "edge " + getPerimeter().getEdges().indexOf(edge), border, hole, material,
						surface.getSurfaceToWorldTransform(), new float[] {thickness, 0});
				models.add(modelBuilder.end());
			}
			for(Shape shape: getPerimeter().getNodes()) {
				if(shape instanceof Arc) {
					Arc arc = (Arc) shape;
					ArrayList<Vector2> arcOuterContour = MathUtilities.calculateSurfaceContour(arc.getVertices(), false ,thickness * RoomController.getInstance().getScaleX());
					ArrayList<Vector2> arcInnerContour = new ArrayList<Vector2>();
					int arcSize = arc.getVertices().size();
					for(int i = 0; i < arcSize; i++) {
						arcOuterContour.add(arc.getVertices().get(arcSize - 1 - i));
					}
					modelBuilder.begin();
					PolygonBuilder.createPolygoneNode(modelBuilder, "arc " + getPerimeter().getNodes().indexOf(arc), arcOuterContour, arcInnerContour, material,
							RoomController.getInstance().getSceneToWorldTransform(), new float[] {2.5f, 0.0f});
					models.add(modelBuilder.end());
				}
			}
			ArrayList<Object3D> objects = new ArrayList<Object3D>();
			for(Model model: models) {
				WallSide side = new WallSide(this);
				BoundingBox boundingBox = new BoundingBox();
				model.calculateBoundingBox(boundingBox);
				Vector3 dimension = new Vector3();
				boundingBox.getDimensions(dimension);
				ObjectModel objectModel = new ObjectModel(model, null);
				side.setModel(objectModel);
				side.setRealWorldDimension(dimension);
				Vector3 center = new Vector3();
				boundingBox.getCenter(center);
				side.setRealWorldPosition(center);
				Object3D fragment = new Object3D(side);
				fragment.calculateProperties();
				fragment.calculateEffectiveBoundingBox();
				objects.add(fragment);
			}
			fragments3D = new Object3D[objects.size()];
			objects.toArray(fragments3D);
			setRequireRefrech(false);
		}
	}

	public Direction getSelectedSurface() {
		return selectedSurface;
	}

	public void setSelectedSurface(Direction selectedSurface) {
		this.selectedSurface = selectedSurface;
	}

	public CompoundShape getPerimeter() {
		return geometry.getBorder();
	}

	public void setPerimeter(CompoundShape perimeter) {
		getGeometry().setBorder(perimeter);
	}

	public float getThickness() {
		return thickness;
	}

	public void setThickness(float thickness) {
		this.thickness = thickness;
	}
	
	public float getHeight() {
		return height;
	}

	public void setHeight(float height) {
		if(height != this.height) {
			this.height = height;
			requireRefrech = true;
		}
	}

	public PbrMaterial getMaterial() {
		if(material == null)
			material = ProjectManager.getManager().getCurrentScene().getPreferences().getWallMtl().cpy();
		return material;
	}

	public void setMaterial(PbrMaterial material) {
		if( this.material == null || !material.equals(this.material)) {
			this.material = material;
			setRequireRefrech(true);
		}
	}
	public void addPropertyChangeListener(PropertyChangeListener propertyChangeListener) {
		if(this.propertyChangeSupport == null)
			propertyChangeSupport = new PropertyChangeSupport(this);
		this.propertyChangeSupport.addPropertyChangeListener(propertyChangeListener);
	}
	public void removePropertyChangeListener(PropertyChangeListener propertyChangeListener) {
		this.propertyChangeSupport.removePropertyChangeListener(propertyChangeListener);
	}



	@Override
	public Wall2D create2DObject() {
		return new Wall2D(this);
	}
	
	
	
}
