package gdxapp.animation;

import java.util.HashMap;
import com.badlogic.gdx.graphics.Camera;
import com.badlogic.gdx.math.Quaternion;
import com.badlogic.gdx.math.Vector3;

import dressing.mathutils.MathUtilities;
import gdxapp.object3d.Animation;

public class CameraAnimator implements IAnimator<Camera> {

	private Camera camera;	
	private static CameraAnimator instance;
	public CameraAnimator() {
	}

	public static void setInstance(CameraAnimator instance) {
		CameraAnimator.instance = instance;
	}

	
	@Override
	public Animation moveTo(Vector3 destination, float duration) {
		Vector3 translation = destination.cpy().sub(camera.position);
		Vector3 velocity = translation.cpy().scl(1.0f/duration);
		HashMap<String, Object> env = new HashMap<String, Object>();
		env.put("animation_duration", duration);
		env.put("destination", destination);
		env.put("velocity", velocity);
		Animation animation = new Animation(env, null) {
			@Override
			public
			Animation act(float delta) {
				setPlaying(true);
				Vector3 cameraDestination = (Vector3)this.getContext().get("destination");
				Vector3 cameraVelocity = (Vector3)getContext().get("velocity");
				if(cameraDestination.cpy().sub(camera.position).dot(cameraVelocity) > 0) {   //not there yet
					camera.position.add(velocity.cpy().scl(delta));
					return this;
				}else {
					 camera.position.set(cameraDestination);
					 setPlaying(false);
				}				
				return null;
			}
		};
		return animation;

	}
	
	@Override
	public Animation turnTo(Quaternion orientation, float duration) {
		Vector3 up = camera.up.cpy().nor();
		Vector3 dir = camera.direction.cpy().nor();
		Vector3 left = camera.up.cpy().crs(camera.direction);
		
		Quaternion currentOrientation = new Quaternion().setFromAxes(left.x, left.y, left.z,
																	up.x, up.y, up.z,
																	dir.x, dir.y, dir.z).nor();
		Quaternion delta = orientation .cpy().mulLeft(currentOrientation .cpy().conjugate());
		Vector3 rotationAxis = new Vector3();
		float angle = delta.getAxisAngle(rotationAxis);
		if(Math.abs(angle) < 0.5f)
			return null;
		if(angle > 180)
			angle -= 360;
		HashMap<String, Object> env = new HashMap<String, Object>();
		env.put("animation_duration", duration);
		env.put("angle", angle);
		env.put("destination", orientation);
		env.put("angular_velocity", angle/duration);
		env.put("rotation_axis", rotationAxis.nor());
		env.put("traveled", 0.0f);
		Animation animation = new Animation(env, null) {
			@Override
			public Animation act(float delta) {
				setPlaying(true);
				Quaternion finalOrientation = (Quaternion) getContext().get("destination");
				float angularVelocity = (float) getContext() .get("angular_velocity");
				Vector3 rotationAxis = (Vector3) getContext() .get("rotation_axis");
				float totalDst = (float) getContext().get("angle");
				float step = angularVelocity * delta;
				float traveled = (float) getContext().get("traveled");				
				if( (totalDst - traveled) * (totalDst - (traveled + step)) > 0) {
					camera.rotate (rotationAxis, -step);
					camera.update();
					getContext().put("traveled", traveled + step);
					return this;
				}else {
					 setPlaying(false);

					// align(camera, finalOrientation);
				}				
				return null;
			}
		};
		return animation;
	}
	

	public static CameraAnimator getInstance() {
		if(instance == null)
			instance = new CameraAnimator();
		return instance;
	}


	@Override
	public Camera getTransformable() {
		return camera;
	}

	@Override
	public void setTarget(Camera transformable) {
		this.camera = transformable;
		
	}
	
	@Override
	public void align(Camera camera, Quaternion q) {
		Vector3 up = camera.up.cpy().nor();
		Vector3 dir = camera.direction.cpy().nor();
		Vector3 left = up.cpy().crs(dir);
		Quaternion currentOrientation = MathUtilities.getQuaterionFromBasis(left, up, dir);
		Quaternion delta = q.mulLeft(currentOrientation.cpy().conjugate());
		camera.rotate(delta);
		camera.update();		
	}

}
