package dressing.ui.util;

import java.text.DecimalFormat;

/**
 * Utility class to convert a price (double) into its French textual
 * representation.
 */
public class PriceToTextConverter {

	// Arrays for textual representation of numbers
	private static final String[] UNITS = { "", "UN", "DEUX", "TROIS", "QUATRE", "CINQ", "SIX", "SEPT", "HUIT",
			"NEUF" };

	private static final String[] TEENS = { "", "ONZE", "DOUZE", "TREIZE", "QUATORZE", "QUINZE", "SEIZE", "DIX-SEPT",
			"DIX-HUIT", "DIX-NEUF" };

	private static final String[] TENS = { "", "DIX", "VINGT", "TRENTE", "QUARANTE", "CINQUANTE", "SOIXANTE",
			"SOIXANTE-DIX", "QUATRE-VINGT", "QUATRE-VINGT-DIX" };

	private static final String[] THOUSANDS = { "", "MILLE", "MILLION", "MILLIARD", "BILLION", "BILLIARD", "TRILLION",
			"TRILLIARD" };
	public static String generatetotalPriceText(String initial,double price,boolean isDecimalChiffre) {
		return new StringBuilder().append(initial).append(convertPriceToText(price,isDecimalChiffre)).toString();
	}
	/**
	 * Converts a given price to its French textual representation. The price is
	 * formatted to keep up to 3 decimal places.
	 *
	 * @param price The price to convert
	 * @return The French textual representation of the price
	 */
	public static String convertPriceToText(double price,boolean isDecimalChiffre) {
		DecimalFormat df = new DecimalFormat("#.###"); // Format the price to three decimal places

		// Splitting the price into integer and decimal parts
		long integerPart = (long) price;
		int decimalPart = (int) Math.round((price - integerPart) * 1000); // Keep up to 3 decimal places

		// Convert the integer part to text
		String integerPartText = convertToText(integerPart);

		// Convert the decimal part to text
		String decimalPartText = convertToText(decimalPart);

		// Forming the complete textual representation
		StringBuilder builder = new StringBuilder();
//		builder.append("Arrêtée la facture proforma n° 400 à la somme de ");
		builder.append(integerPartText);
		builder.append(" dinar(s)");

		if (decimalPart > 0) {
			builder.append(" et ");
			if(isDecimalChiffre) {
				builder.append(decimalPart);

			}else {
				builder.append(decimalPartText);
			}
			builder.append(" millime(s)");
		}

		return builder.toString();
	}

	/**
	 * Converts a given number to its French textual representation.
	 *
	 * @param number The number to convert
	 * @return The French textual representation of the number
	 */
	private static String convertToText(long number) {
		if (number == 0) {
			return "zéro";
		}

		StringBuilder builder = new StringBuilder();
		int index = 0;

		while (number > 0) {
			if (number % 1000 != 0) {
				builder.insert(0, convertChunkToText((int) (number % 1000)) + " " + THOUSANDS[index] + " ");
			}
			number /= 1000;
			index++;
		}

		return builder.toString().trim();
	}

	/**
	 * Converts a given chunk of number to its French textual representation.
	 * Handles the tens and units place.
	 *
	 * @param number The chunk of number to convert
	 * @return The French textual representation of the chunk
	 */
	private static String convertChunkToText(int number) {
		StringBuilder builder = new StringBuilder();

		// Handle hundreds
		if (number >= 100) {
			builder.append(UNITS[number / 100] + " cent ");
			number %= 100;
		}
		// Handle specific cases for tens and teens
		if ((number % 10) == 0 && number != 0) {
			builder.append(TENS[number / 10] + " ");
			number = 0;
		} else if (number > 90) {
			builder.append("quatre-vingt " + TEENS[number % 10] + " ");
			number = 0;
		} else if (number > 70 && number < 80) {
			builder.append("soixante " + TEENS[number % 10] + " ");
			number = 0;
		} else if (number > 10 && number < 20) {
			builder.append(TEENS[number - 10] + " ");
			number = 0;
		} else {
			builder.append(TENS[number / 10] + " ");
			number %= 10;
		}

		// Handle units
		if (number > 0) {
			builder.append(UNITS[number] + " ");
		}

		return builder.toString().trim();
	}

	public static void main(String[] args) {
		double price = 29495.318; // Example price

		String priceText = generatetotalPriceText("Arrêtée la facture proforma n° 400 à la somme de ",price,false);
		System.out.println("Le montant en lettres : \n" + priceText);
	}
}
