package dressing.ui.util;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.UUID;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import dressing.config.WorkspaceConfiguration;

public class ImageChooserDialog extends Dialog {
	public static final String[] extensions= { "*.png;*.jpg;*.jpeg;*jpeg;"};
	private Text imagePath;
	private String imageUri;
	private Label lblImage;
	private String localPath; 
	private Image img;
	/**
	 * Create the dialog.
	 * @param parentShell
	 */
	public ImageChooserDialog(Shell parentShell,String imageUri) {
		super(parentShell);
		this.imageUri=imageUri;
	}

	/**
	 * Create contents of the dialog.
	 * @param parent
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite container = (Composite) super.createDialogArea(parent);
		GridLayout gridLayout = (GridLayout) container.getLayout();
		gridLayout.numColumns = 2;
		
		imagePath = new Text(container, SWT.BORDER);
		imagePath.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		
		Button btnLocal = new Button(container, SWT.NONE);
		btnLocal.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1));
		
		btnLocal.setText("Local");
		
		lblImage = new Label(container, SWT.NONE);
		GridData gd_lblImage = new GridData(SWT.CENTER, SWT.FILL, false, false, 1, 1);
		gd_lblImage.widthHint = 200;
		gd_lblImage.heightHint = 150;
		lblImage.setLayoutData(gd_lblImage);
		
		Button btnPC = new Button(container, SWT.NONE);
		
		btnPC.setLayoutData(new GridData(SWT.FILL, SWT.TOP, false, false, 1, 1));
		btnPC.setText("Ordinateur");
		btnPC.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				String defaultPath = getDefaultImportPath();
				openFileDialog(defaultPath, parent.getShell());
			}
		});
		btnLocal.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				openFileDialog(getLocalPath(), parent.getShell());
			}
		});
		
		if(imageUri!=null && !imageUri.isEmpty()) {
			imagePath.setText(imageUri);
			disposeImage();
			img=ImageLoaderCache.getInstance().resizeImage(ImageLoaderCache.createImageFromPathOrUrl(getImageUri()), 200, 150);
			lblImage.setImage(img);	
		}
		imagePath.addKeyListener(new KeyAdapter() {
			@Override
			public void keyReleased(KeyEvent e) {
				if (imagePath.getText().contentEquals(imageUri)) {
					return;
				}
				imageUri = imagePath.getText();
				if (ImageLoaderCache.isValidImagePath(imageUri) || ImageLoaderCache.isValidImageUrl(imageUri)) {
					disposeImage();
					img = ImageLoaderCache.getInstance()
							.resizeImage(ImageLoaderCache.createImageFromPathOrUrl(getImageUri()), 200, 150);
					lblImage.setImage(img);
				}
			}

		});
		
		
		parent.getShell().setText("Choisir un image");
		parent.getShell().addShellListener(new ShellAdapter() {
			@Override
			public void shellClosed(ShellEvent e) {
				disposeImage();
			}

			@Override
			public void shellDeactivated(ShellEvent e) {
				disposeImage();
			}

			@Override
			public void shellDeiconified(ShellEvent e) {
				disposeImage();
			}
		});
		return container;
	}

	public String getLocalPath() {
		if(localPath==null || localPath.isEmpty()) {
			localPath=WorkspaceConfiguration.getAccessoireImagesPath();
		}
		return localPath;
	}

	public void setLocalPath(String localPath) {
		this.localPath = localPath;
	}

	/**
	 * Create contents of the button bar.
	 * @param parent
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		Button btnOk = createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
		btnOk.setText("Appliquer");
		Button btnCancel = createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
		btnCancel.setText("Annuler");
	}
	
	@Override
	protected void okPressed() {
		if(!ImageLoaderCache.isValidImagePath(imageUri)&&!ImageLoaderCache.isValidImageUrl(imageUri)) {
			MessageDialog.openError(imagePath.getShell(), "Erreur", "Le fichier choisi est une image invalide." );
			return;
		}
		if(img!=null && !img.isDisposed()) {
			img.dispose();
		}
		if(lblImage.getImage()!=null)
		{
			lblImage.getImage().dispose();
		}
		super.okPressed();
	}
	/**
	 * Return the initial size of the dialog.
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(450, 300);
	}

	public String getImageUri() {
		return imageUri;
	}

	public void setImageUri(String imageUri) {
		this.imageUri = imageUri;
	}
	@Override
	public boolean close() {
		if(img!=null && !img.isDisposed()) {
			img.dispose();
		}
		if(lblImage.getImage()!=null)
		{
			lblImage.getImage().dispose();
		}
		return super.close();
	}
	
	
	private void openFileDialog(String initialPath, Shell shell) {
		FileDialog dialog = new FileDialog(shell, SWT.OPEN);
		dialog.setFilterExtensions(extensions);
		dialog.setFilterPath(initialPath);
		String filepath = dialog.open();

		if (filepath != null && !filepath.isEmpty()) {
			if (!ImageLoaderCache.isValidImagePath(filepath)) {
				MessageDialog.openError(shell, "Invalid Image", "Veuillez sélectionner une image valide.");
			} else {
				saveAndSetImage(filepath);
			}
		}
	}

	private void saveAndSetImage(String originalPath) {
		try {
			File originalFile = new File(originalPath);
			File destinationDir = new File(getLocalPath());
			if (!destinationDir.exists()) {
				destinationDir.mkdirs();
			}
			if(originalFile.getParentFile().equals(destinationDir)) {
				imageUri=originalFile.getAbsolutePath();
			}else {
				String newFileName = generateUniqueFileName(originalFile.getName());		
				File destFile = new File(destinationDir, newFileName);
				Files.copy(originalFile.toPath(), destFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
				imageUri = destFile.getAbsolutePath();
			}
			
			imagePath.setText(imageUri);

			disposeImage();
			img = ImageLoaderCache.getInstance().loadImage(imageUri, 200, 150);
			lblImage.setImage(img);

		} catch (Exception ex) {
			MessageDialog.openError(getShell(), "Erreur", "Échec de la copie de l'image : " + ex.getMessage());
		}
	}
	
	
	public static String generateUniqueFileName(String originalFileName) {
		String extension = "";
		if (originalFileName != null && originalFileName.lastIndexOf('.') > -1) {
			int lastDotIndex = originalFileName.lastIndexOf('.');
			extension = originalFileName.substring(lastDotIndex + 1);
		}
		
		long millis = System.currentTimeMillis();
		String uuid = UUID.randomUUID().toString();
		return uuid + '_' + millis + '.' + extension;
	}

	private String getDefaultImportPath() {
		// Preferences preferences = InstanceScope.INSTANCE.getNode("com.dressing").node("SuperDressingProject");
		// return preferences.get("importFromDirectory", System.getProperty("user.home") + File.separator + "Desktop");
		return System.getProperty("user.home");

	}
	
	
	private void disposeImage() {
		if (img != null && !img.isDisposed()) {
			img.dispose();
		}
	}
	
}
